/*
 *	book.c		April 10, 1989
 *
 *	Print documents on 5.5" by 8.5" pages both sides of the
 *	paper, using an HP Laserjet Series II.
 *
 *	Author: Tom Neff
 *
 *	Written for UNIX System V.
 *
 *		To compile:
 *
 *			cc -O book.c -o book
 *
 *	Inspired by Vern Buerg's LJBOOK.
 *
 *	 4/10/89	change default tabs to 8;
 *				handle one-page docos without pausing
 */

#include <string.h>
#include <stdio.h>

#define FF '\014'
#define MAXPAGES 1000
#define MAXROWS 66
#define MAXCOLS 128
#define DEFAULTTABS 8

#define tolpt(x) fputs(x, lpt)

static FILE *docf;
static FILE *lpt = stdout;

static long pagebrks[MAXPAGES];
static char linbuf[MAXCOLS];

static int nsheets, nhalves, ndocpgs;
static int row;
static int lpage, rpage;

static int tabsize=DEFAULTTABS;

static char *PINIT="\033&l1ol8d1e67f5.5C\033(s0p16.66h8.5v0d0b0t1Ci\r";
static char *PRIGHT="\0339\033&a1r96l178M\r";
static char *PLEFT="\0339\033&a1ra0la89M\r";

	/*  Send a filtered line to the output stream  */

ljline(source)
char *source;
{
	char *s = source;
	int col = 0;
	register char c;
	char dest[MAXCOLS], *d = dest;

#define give(c)	*d++=c, col++

	do
	{
		c=*s++;

		if (c=='\n')
		{
			give('\r');
			give('\n');
		}
		else if ((c=='\b') && (col > 0))
			d--, col--;
		else if (c=='\t')
		{
			give(' ');
			while ((col % tabsize) != 0)
				give(' ');
		}
		else if (c >= ' ')
			give(c);
	} while (c != '\0');

	*d++=0;
	tolpt(dest);
}

	/*  Send a page of text to the left or right page  */

dopage(page, inits)
int page;
char *inits;
{
	char *p;

	tolpt(inits);
	if (page >= ndocpgs)
		return;

	fseek(docf, pagebrks[page], 0);
	for (row=0; row < MAXROWS; row++)
	{
		if (fgets(linbuf, sizeof(linbuf), docf) == NULL)
			break;

		while ((p=strchr(linbuf, FF)) != NULL)
			*p = (char) ' ';			/*  filter formfeeds  */

		if (strchr(linbuf, '\r') == NULL)		/*  Add CR's  */
		{
			if ((p = strchr(linbuf, '\n')) != NULL)
			{
				*p = '\r';
				*++p = '\n';
				*++p = '\0';
			}
		}

		ljline(linbuf);
	}
}

	/*  Print an entire file in two passes.  */

bookfile(path)
char *path;
{
	fprintf(stderr, "\nReady to print %s\n", path);

	/*  make a pass and learn where the page breaks are  */

	if ((docf = fopen(path, "r")) == NULL)
	{
		fprintf(stderr, "Error opening %s!\n", path);
		return;
	}

	fseek(docf, 0, 0);
	ndocpgs = 0;

	while ((!feof(docf)) && (ndocpgs < MAXPAGES))
	{
		row = 0;
		pagebrks[ndocpgs] = ftell(docf);
		ndocpgs++;

		while (row++ < MAXROWS)
		{
			if (fgets(linbuf, sizeof(linbuf), docf) == NULL)
				break;

			if (strchr(linbuf, FF) != NULL)
				break;
		}

		if (feof(docf))
			break;
	}

	nsheets = (ndocpgs+3)/4;
	nhalves = nsheets*4;

	fprintf(stderr, "%d sheets to print...\n", nsheets);

	/*  Now let's print the first pass  */

	tolpt(PINIT);

	for (rpage=0, lpage=nhalves-1;
		rpage < lpage;
		rpage+=2, lpage-=2)
	{
		dopage(lpage, PLEFT);
		dopage(rpage, PRIGHT);

		fputc(FF, lpt);
	}

	if (nhalves > 1)
	{
		fflush(lpt);

		fprintf(stderr,"\nReady to print the other side.\n");
		fprintf(stderr,"\nRemove paper and place it back in the in-tray.\n");
		fprintf(stderr,"Press ENTER when ready: ");
		gets(linbuf);

		/*  The second pass  */

		tolpt(PINIT);

		for (rpage=nhalves/2, lpage=rpage-1; lpage >= 0; rpage+=2, lpage-=2)
		{
			dopage(lpage, PLEFT);
			dopage(rpage, PRIGHT);

			fputc(FF, lpt);
		}
	}

	/*  Done with this file  */

	fclose(docf);
	fflush(lpt);

	fprintf(stderr,"Done printing %s\n", path);
}

	/*  The command mainline. */

main(argc, argv)
int argc;
char **argv;
{
	int c;
	int errflg = 0;

	extern char *optarg;
	extern int optind;

	/*  Collect option switches  */

	while ((c=getopt(argc, argv, "o:t:?")) != -1)
		switch (c) 
		{
		case 'o':
			if ((lpt=fopen(optarg, "w")) == NULL)
			{
				fprintf(stderr, "Error opening %s\n", optarg);
				exit(1);
			}
			break;
		case 't':
			if ((tabsize=atoi(optarg)) == 0)
				errflg++;
			break;
		default:
			errflg++;
		}

	/*  Validate args and print usage message if bad  */

	if (argc < 2)
		errflg++;

	if (errflg)
	{
		fprintf(stderr, "Usage: book [-o outfile] [-t tabsize] file ...\n");
		exit(1);
	}

	/*  Now read file args and print them  */

	for ( ; optind < argc; optind++)
	{
		bookfile(argv[optind]);

		if (optind+1 < argc)
		{
			fprintf(stderr,"Press ENTER when ready for next file: ");
			gets(linbuf);
		}
	}
}
