/* Project:		lj2ps
** File:		ljcmds.c
**
** Author:		Christopher Lishka
** Organization:	Wisconsin State Laboratory of Hygiene
**			Data Processing Dept.
**
** Copyright (C) 1990 by Christopher Lishka.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static char * ModuleID = "Module ljcmds: v1.0, production";

  /* Include files
  */
#include <stdio.h>
#include <math.h>
#include "ljcmds.h"
#include "lj.h"
#include "ljfonts.h"
#include "scan.h"
#include "lj2ps.h"

  /* External definitions
  */

  /* Global variables
  */

  /* Global function list
  */
extern void  ljcmd_undefined();	/* Command: undefined */
extern void  ljcmd_E();		/* Command: ^[E  reset */
extern void  ljcmd_Y();		/* Command: ^[Y  display functions on */
extern void  ljcmd_Z();		/* Command: ^[Z  display functions off */
extern void  ljcmd_9();		/* Command: ^[9  clear margins */
extern void  ljcmd_eq();	/* Command: ^[=  half-line feed */
extern void  ljcmd_lp();	/* Command: ^[(  primary symbol set */
extern void  ljcmd_rp();	/* Command: ^[)  secondary symbol set */
extern void  ljcmd_lp_s();	/* Command: ^[(s primary font attributes */
extern void  ljcmd_rp_s();	/* Command: ^[)s secondary font attributes */
extern void  ljcmd_amp_a();	/* Command: ^[&a margins & movement */
extern void  ljcmd_amp_d();	/* Command: ^[&d underline */
extern void  ljcmd_amp_f();	/* Command: ^[&f macros, position stack */
extern void  ljcmd_amp_k();	/* Command: ^[&k HMI, pitch, line termination*/
extern void  ljcmd_amp_l();	/* Command: ^[&l page attributes */
extern void  ljcmd_amp_p();	/* Command: ^[&p transparent print data */
extern void  ljcmd_amp_s();	/* Command: ^[&s end-of-line wrap */
extern void  ljcmd_star_b();	/* Command: ^[*b bitmap transfer */
extern void  ljcmd_star_c();	/* Command: ^[*c graphics, soft fonts */
extern void  ljcmd_star_p();	/* Command: ^[*p device-dependent movement */
extern void  ljcmd_star_r();	/* Command: ^[*r start/end graphics */
extern void  ljcmd_star_t();	/* Command: ^[*t graphics resolution */

  /* Local constants
  */

  /* Local structures and types
  */

  /* Local variables
  */

  /* Local macro definitions
  */
#define operation(X) (variable[strlen(variable) - 1] == X)

  /* Local function list
  */

  /* Function bodies
  */

void
ljcmd_undefined(ifile, ofile)	/* Command: undefined */
     FILE *ifile, *ofile;
{
      warning("unknown LaserJet PCL command");
} /* ljcmd_undefined() */



  /* ljcmd_E(ofile) performs a LaserJet "reset" command.  This initializes
  ** all internal variables to default values.  In addition, if the current
  ** page is not empty, a new page is started.
  */
void
ljcmd_E(ofile)			/* Command: ^[E reset */
     FILE *ofile;
{

    /* If current page is not empty, then start a new one */
  lj_page_end(ofile);
  
    /* Reset the printer */
  lj_reset(ofile);

    /* Begin the next page */
  lj_page_begin(ofile);

} /* ljcmd_E() */



void
ljcmd_Y(ofile)			/* Command: ^[Y display functions on */
     FILE *ofile;
{

  warning("command <esc>Y is not implemented -- ignoring", "");

} /* ljcmd_Y() */



void
ljcmd_Z(ofile)			/* Command: ^[Z display functions off */
     FILE *ofile;
{

  warning("command <esc>Z is not implemented -- ignoring", "");

} /* ljcmd_Z() */



void
ljcmd_9(ofile)			/* Command: ^[9 clear margins */
     FILE *ofile;
{

  margin_left  = 0.0;		/* Reset the left margin */
  text_width  = page_width - margin_left; /* Reset the virtual right margin */

  lj_text_end(ofile);
  fprintf(ofile, "/ml 0.0 def %.4f RM ", /* Reset the clipping path */
	  margin_left + text_width);
  lj_text_begin();

} /* ljcmd_9() */



void
ljcmd_eq(ofile)			/* Command: ^[= half-line feed */
     FILE *ofile;
{

  lj_text_end(ofile);
  lj_undl_flush(ofile);

  lj_cursor_ver_rel(ofile, (char_height / 2));

  lj_undl_mark(ofile);
  lj_text_begin();

} /* ljcmd_eq() */



void
ljcmd_lp(ifile, ofile)		/* Command ^[( primary symbol set */
     FILE *ifile, *ofile;
{

    /* Parse all of the parameters, until the scanner returnes PARAM_END */
  do{
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      warning("more than one symbol set in a command -- ignoring", command);
      break;
    case PARAM_END:
      switch( atoi(number) ){
      case 0:			/* 0B, 0N, 0U */
	if( operation('B') )       font_p.symbol_set = LJ_SS_0B;
	else if( operation('N') )  font_p.symbol_set = LJ_SS_0N;
	else if( operation('U') )  font_p.symbol_set = LJ_SS_0U;
#ifdef VERBOSE_WARNINGS
	else
	  warning("unknown symbol set -- ignoring", command);
#endif
	break;
      case 1:			/* 1U */
	if( operation('U') )  font_p.symbol_set = LJ_SS_1U;
#ifdef VERBOSE_WARNINGS
	else
	  warning("unknown symbol set -- ignoring", command);
#endif
	break;
      case 8:			/* 8U */
	if( operation('U') )  font_p.symbol_set = LJ_SS_8U;
#ifdef VERBOSE_WARNINGS
	else
	  warning("unknown symbol set -- ignoring", command);
#endif
	break;
      case 10:			/* 10U */
	if( operation('U') )  font_p.symbol_set = LJ_SS_10U;
#ifdef VERBOSE_WARNINGS
	else
	  warning("unknown symbol set -- ignoring", command);
#endif
	break;
      case 11:			/* 11U */
	if( operation('U') )  font_p.symbol_set = LJ_SS_11U;
#ifdef VERBOSE_WARNINGS
	else
	  warning("unknown symbol set -- ignoring", command);
#endif
	break;
#ifdef VERBOSE_WARNINGS
      default:
	warning("unknown symbol set -- ignoring", command);
#endif
      } /* switch( number ) */
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
    lj_match_font(&font_p);
    if( current_font == LJ_FT_PRIMARY ){
      lj_text_end(ofile);
      lj_undl_flush(ofile);
      lj_set_font(ofile, font_p);
      lj_undl_mark(ofile);
      lj_text_begin();
    } /* if( ... ) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_lp() */



void
ljcmd_rp(ifile, ofile)		/* Command: ^[) secondary symbol set */
     FILE *ifile, *ofile;
{

  warning("command <esc>) is not implemented -- ignoring", "");

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      break;
    case PARAM_END:
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_rp() */



void
ljcmd_lp_s(ifile, ofile)	/* Command: ^[(s primary font attributes */
     FILE *ifile, *ofile;
{

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
    case PARAM_END:
      if( operation('P') ){	/* Spacing */
	switch( atoi(number) ){
	case 0: font_p.spacing = LJ_SP_FIXED; break;
	case 1: font_p.spacing = LJ_SP_PROPORTIONAL; break;
	default:
	  error("unknown primary font spacing code", number);
	}
      } /* if( P ) */
      else if( operation('H') ){ /* Pitch */
	font_p.pitch = atof(number);
      } /* else if( H ) */
      else if( operation('V') ){ /* Point size */
	font_p.point_size = atof(number);
      } /* else if( V ) */
      else if( operation('S') ){ /* Style */
	switch( atoi(number) ){
	case 0: font_p.style = LJ_ST_UPRIGHT; break;
	case 1: font_p.style = LJ_ST_ITALIC;  break;
	default:
	  error("unknonw primary font style code", number);
	}
      } /* else if( S ) */
      else if( operation('B') ){ /* Stroke weight */
	switch( pos_code ){
	case POS_ABSOLUTE:
	case POS_REL_POSITIVE:
	  font_p.weight = atoi(number); break;
	case POS_REL_NEGATIVE:
	  font_p.weight = -(atoi(number)); break;
	default:
	  internal_error("unknown pos-code in primary font stroke weight", "");
	}
      } /* else if( B ) */
      else if( operation('T') ){ /* Typeface */
	switch( atoi(number) ){
	case  4: font_p.typeface = LJ_TF_HELV;  break;
	case  5: font_p.typeface = LJ_TF_TIMES; break;
	case  8: font_p.typeface = LJ_TF_PRES;  break;
	case  0: font_p.typeface = LJ_TF_LP;    break;
	case  3: font_p.typeface = LJ_TF_COUR;  break;
	default:
	  error("unrecognized typeface code (possibly not supported",
		number);
	}
      } /* else if( T ) */
      else{			/* Unknown command */
	warning("unknown command with prefix <esc>(s -- ignoring", "");
      } /* else */
        /* If at the *end* of the font change command, change the font! */
      if( curr_token.code == PARAM_END ){
	lj_match_font(&font_p);
	if( current_font == LJ_FT_PRIMARY ){
	  lj_text_end(ofile);
	  lj_undl_flush(ofile);
	  lj_set_font(ofile, font_p);
	  lj_undl_mark(ofile);
	  lj_text_begin();
	} /* if( ... ) */
      } /* if( ... ) */
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_lp_s() */



void
ljcmd_rp_s(ifile, ofile)	/* Command: ^[)s secondary font attributes */
     FILE *ifile, *ofile;
{

  warning("command family <esc>)s not implemented -- ignoring", "");

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      break;
    case PARAM_END:
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_rp_s() */



void
ljcmd_amp_a(ifile, ofile)	/* Command: ^[&a margins & movement */
     FILE *ifile, *ofile;
{

  do{
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
    case PARAM_END:
      if( operation('R') ){	/* Move vertically by rows */
	lj_text_end(ofile);
	lj_undl_flush(ofile);
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute postioning */
	  lj_cursor_ver_abs(ofile,
			    page_height -
			    (margin_top + char_height + ro2in(atof(number))) );
	  break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  lj_cursor_ver_rel(ofile, ro2in(atof(number))); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  lj_cursor_ver_rel(ofile, -(ro2in(atof(number)))); break;
	default:
	  internal_error("illegal positioning code", "");
	}
	lj_undl_mark(ofile);
	lj_text_begin();
      }
      else if( operation('V') ){ /* Move vertically by decipoints */
	lj_text_end(ofile);
	lj_undl_flush(ofile);
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	  lj_cursor_ver_abs(ofile,
			    page_height -
			    (margin_top + dp2in(atof(number)))); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  lj_cursor_ver_rel(ofile, dp2in(atof(number))); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  lj_cursor_ver_rel(ofile, -(dp2in(atof(number)))); break;
	default:
	  internal_error("illegal positioning code", "");
	}
	lj_undl_mark(ofile);
	lj_text_begin();
      }
      else if( operation('C') ){ /* Move horizontally by columns */
	lj_text_end(ofile);
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute postioning */
	  lj_cursor_hor_abs(ofile, co2in(atof(number))); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  lj_cursor_hor_rel(ofile, co2in(atof(number))); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  lj_cursor_hor_rel(ofile, -(co2in(atof(number)))); break;
	default:
	  internal_error("illegal positioning code", "");
	}
	lj_text_begin();
      }
      else if( operation('H') ){ /* Move horizontally by decipoints */
	lj_text_end(ofile);
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	  lj_cursor_hor_abs(ofile, dp2in(atof(number))); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  lj_cursor_hor_rel(ofile, dp2in(atof(number))); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  lj_cursor_hor_rel(ofile, -(dp2in(atof(number)))); break;
	default:
	  internal_error("illegal positioning code", "");
	}
	lj_text_begin();
      }
      else if( operation('L') ){ /* Set the left margin */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	  margin_left = co2in(atof(number)); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  margin_left += co2in(atof(number)); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  margin_left -= co2in(atof(number)); break;
	default:
	  internal_error("illegal positioning code", "");
	}
	lj_text_end(ofile);
	fprintf(ofile, "/ml %4f def ", margin_left);
	if( empty_line ) lj_cursor_hor_abs(ofile, margin_left);
	lj_text_begin();
      }
      else if( operation('M') ){ /* Set the right margin */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	  text_width = co2in((atof(number) + 1)) - margin_left; break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  text_width += co2in(atof(number)); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  text_width -= co2in(atof(number)); break;
	default:
	  internal_error("illegal positioning code", "");
	} /* switch( pos_code ) */
	lj_text_end(ofile);
	fprintf(ofile, "%.4f RM ", /* Reset the clipping path */
		margin_left + text_width);
	lj_text_begin();
      } /* else if( M ) */
      else{			/* Unknown command */
	warning("unknown command with prefix <esc>&a -- ignoring", "");
      } /* else */
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_amp_a() */



void
ljcmd_amp_d(ifile, ofile)	/* Command: ^[&d underline */
     FILE *ifile, *ofile;
{

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
    case PARAM_END:
      if( operation('D') ){
	  /* Note: you *must* end the text before setting underline mode.
	  ** Otherwise, the text printed *before* underlining is turned on
	  ** is not flushed from the text buffer, and will also be
	  ** underlined.
	  */
	lj_text_end(ofile);
	lj_undl_begin(ofile);
	lj_text_begin();
      }
      else if( operation('@') ){
	lj_text_end(ofile);
        lj_undl_end(ofile);
        lj_text_begin();
      }
      else{			/* Unknown command */
	warning("unknown command with prefix <esc>&d -- ignoring", "");
      }
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_amp_d() */



void
ljcmd_amp_f(ifile, ofile)	/* Command: ^[&f macros, position stack */
     FILE *ifile, *ofile;
{

  warning("command family <esc>&f not implemented -- ignoring", "");

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      break;
    case PARAM_END:
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_amp_f() */



void
ljcmd_amp_k(ifile, ofile)	/* Command: ^[&k HMI, pitch, line termination*/
     FILE *ifile, *ofile;
{

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
    case PARAM_END:
      if( operation('H') ){	/* Set HMI */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute postioning */
	  char_width  = hi2in(atof(number)); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  char_width += hi2in(atof(number)); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  char_width -= hi2in(atof(number)); break;
	default:
	  internal_error("illegal positioning code", "");
	}
      } /* if( K ) */
      else{			/* Unknown command */
	warning("unknown command with prefix <esc>&k -- ignoring", "");
      }
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_amp_k() */



void
ljcmd_amp_l(ifile, ofile)	/* Command: ^[&l page attributes */
     FILE *ifile, *ofile;
{
  double tmp;
  int old_orientation;

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
    case PARAM_END:
      if( operation('A') ){ /* Set the page size */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	  switch( atoi(number) ){
	  case  1:		/* Executive */
	    page_size.code = LJ_PS_EXECUTIVE;
	    warning("EXECUTIVE paper size is not available", "");
	    break;
	  case  2:		/* Letter */
	    page_size.code = LJ_PS_LETTER;
	    lj_text_end(ofile);
	    fputs("letter ", ofile);
	    lj_text_begin();
	    warning("paper changed to LETTER size", "");
	    break;
	  case  3:		/* Legal */
	    page_size.code = LJ_PS_LEGAL;
	    lj_text_end(ofile);
	    fputs("legal ", ofile);
	    lj_text_begin();
	    warning("paper changed to LEGAL size", "");
	    break;
	  case 26:		/* A4 */
	    page_size.code = LJ_PS_A4;
	    lj_text_end(ofile);
	    fputs("a4 ", ofile);
	    lj_text_begin();
	    warning("paper changed to A4 size", "");
	    break;
	  case 80:		/* Monarch */
	    page_size.code = LJ_PS_MONARCH;
	    warning("MONARCH envelope size is not available", "");
	    break;
	  case 81:		/* Commercial 10 */
	    page_size.code = LJ_PS_COMMERCIAL10;
	    warning("COMMERCIAL-10 envelope size is not available", "");
	    break;
	  case 90:		/* International DL */
	    page_size.code = LJ_PS_INTERNATIONALDL;
	    warning("INTERNATIONAL-DL envelope size is not available", "");
	    break;
	  case 91:		/* International C5 */
	    page_size.code = LJ_PS_INTERNATIONALC5;
	    warning("INTERNATIONAL-C5 envelope size is not available", "");
	    break;
	  default:
	    error("illegal paper size code", number);
	  } /* switch( number ) */
	    /* Find the correct paper size parameters */
	  if( lj_paper_size(&page_size) ){
	    internal_error("illegal page size", "");
	  }
	    /* Set the new page width and height */
	  if( orientation == LJ_OR_PORTRAIT ){
	    page_width = page_size.width;
	    page_height = page_size.height;
	  }
	  else{
	    page_width = page_size.height;
	    page_height = page_size.width;
	  }
	    /* Reset several page control parameters */
	  margin_top  = 0.5;		/* Top and bottom margins are 1/2" */
	  text_height = page_height - margin_top - 0.5;
	  char_height = text_height / panel_form;
	  char_width  = font_p.width;	/* inches per character */
	  text_length = text_height / char_height;
	    /* Flush the current page */
	  lj_page_end(ofile);
	  lj_page_begin();
	  break;
	case POS_REL_POSITIVE:	/* Relative positive */
	case POS_REL_NEGATIVE:	/* Relative negative */
	  error("cannot use relative page size codes!", number);
	default:
	  internal_error("illegal positioning code", "");
	}
      } /* else if( A ) */
      else if( operation('C') ){	/* Set VMI */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute postioning */
	  char_height  = vi2in(atof(number)); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  char_height += vi2in(atof(number)); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  char_height -= vi2in(atof(number)); break;
	default:
	  internal_error("illegal positioning code", "");
	}
      } /* else if( C ) */
      else if( operation('D') ){	/* Set lines per inch */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute postioning */
	  char_height  = 1 / atof(number); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  char_height += 1 / atof(number); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  char_height -= 1 / atof(number); break;
	default:
	  internal_error("illegal positioning code", "");
	}
      } /* else if( D ) */
      else if( operation('E') ){ /* Set the top margin */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	  margin_top =  ro2in(atof(number)); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  margin_top -= ro2in(atof(number)); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  margin_top += ro2in(atof(number)); break;
	default:
	  internal_error("illegal positioning code", "");
	}
	if( empty_page ){
	  current_x    = margin_left;
	  current_y    = page_height - margin_top - char_height;
	  current_line = 1;
	  lj_cursor_abs(ofile, current_x, current_y);
	}
      }
      else if( operation('F') ){ /* Set the text length */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	  text_length =  atoi(number); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  text_length += atoi(number); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  text_length -= atoi(number); break;
	default:
	  internal_error("illegal positioning code", "");
	}
	  /* Reset the text_height */
	text_height = text_length * char_height;
      } /* else if( F ) */
      else if( operation('H') ){ /* Set the paper source */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	  switch( atoi(number) ){
	  case 0:		/* Eject current page */
	    lj_page_end(ofile);
	    lj_page_begin(ofile);
	    break;
	  case 1:		/* Feed from the internal tray */
	    paper_source = LJ_PS_TRAY_1;
	    lj_page_end(ofile);
	    lj_page_begin(ofile);
	    lj_text_end(ofile);
	    fputs("statusdict begin /manualfeed false def end\n", ofile);
	    lj_text_begin();
	    break;
	  case 2:		/* Manual feed */
	    if(   (page_size.code == LJ_PS_EXECUTIVE)
	       || (page_size.code == LJ_PS_LETTER)
	       || (page_size.code == LJ_PS_LEGAL)
	       || (page_size.code == LJ_PS_A4) ){
	      paper_source = LJ_PS_MANUAL;
	    }
	    else{
	      paper_source = LJ_PS_MANUAL_ENVELOPE;
	    }
	    lj_page_end(ofile);
	    lj_page_begin(ofile);
	    lj_text_end(ofile);
	    fputs("statusdict begin /manualfeed true def end\n", ofile);
	    lj_text_begin();
	    break;
	  default:
	    error("illegal paper source code -- ignoring", number);
	  }
	  break;
	case POS_REL_POSITIVE:	/* Relative positive */
	case POS_REL_NEGATIVE:	/* Relative negative */
	  error("illegal paper source code -- ignoring", number);
	  break;
	default:
	  internal_error("illegal positioning code", "");
	}
      } /* else if( H ) */
      else if( operation('O') ){ /* Set the page orientation */
	old_orientation = orientation;
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	case POS_REL_POSITIVE:	/* Relative positive */
	case POS_REL_NEGATIVE:	/* Relative negative */
	  switch( atoi(number) ){
	  case 0:		/* Portrait */
	    orientation = LJ_OR_PORTRAIT;  break;
	  case 1:		/* Landscape */
	    orientation = LJ_OR_LANDSCAPE; break;
	  default:
	    warning("illegal page orientation -- ignoring", "");
	  } /* switch( number ) */
	  break;
	default:
	  internal_error("illegal page orientation", "");
	}
	if( orientation != old_orientation ){
	  lj_page_end(ofile);
	  if( orientation == LJ_OR_PORTRAIT ){ /* Set new width & height */
	    page_width = page_size.width;
	    page_height = page_size.height;
	  }
	  else{
	    page_width = page_size.height;
	    page_height = page_size.width;
	  }
	  lj_page_begin(ofile);
	} /* if( change in orientation ) */
      } /* else if( O ) */
      else if( operation('X') ){ /* Set the number of copies */
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute positioning */
	  copies  = atoi(number); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  copies += atoi(number); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  copies -= atoi(number); break;
	default:
	  internal_error("illegal number of copies", "");
	}
	lj_text_end(ofile);
	lj_copies(ofile, copies);
	lj_text_begin();
      } /* else if( X ) */
      else{			/* Unknown command */
	warning("unknown command with prefix <esc>&l -- ignoring", "");
      }
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_amp_l() */



void
ljcmd_amp_p(ifile, ofile)	/* Command: ^[&p transparent print data */
     FILE *ifile, *ofile;
{

  warning("command family <esc>&p not implemented -- ignoring", "");

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      break;
    case PARAM_END:
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_amp_p() */



void
ljcmd_amp_s(ifile, ofile)	/* Command: ^[&s end-of-line wrap */
     FILE *ifile, *ofile;
{

  warning("command family <esc>&s not implemented -- ignoring", "");

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      break;
    case PARAM_END:
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_amp_s() */



void
ljcmd_star_b(ifile, ofile)	/* Command: ^[*b bitmap transfer */
     FILE *ifile, *ofile;
{

  warning("command family <esc>*b not implemented -- ignoring", "");

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      break;
    case PARAM_END:
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_star_b() */



void
ljcmd_star_c(ifile, ofile)	/* Command: ^[*c graphics, soft fonts */
     FILE *ifile, *ofile;
{

  warning("command family <esc>*c not implemented -- ignoring", "");

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      break;
    case PARAM_END:
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_star_c() */



void
ljcmd_star_p(ifile, ofile)	/* Command: ^[*p device-dependent movement */
     FILE *ifile, *ofile;
{

  do{
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
    case PARAM_END:
      if( operation('X') ){	/* Move horizontally by dots*/
	lj_text_end(ofile);
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute postioning */
	  lj_cursor_hor_abs(ofile, dt2in(atof(number))); break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  lj_cursor_hor_rel(ofile, dt2in(atof(number))); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  lj_cursor_hor_rel(ofile, -(dt2in(atof(number)))); break;
	default:
	  internal_error("illegal positioning code", "");
	}
	lj_text_begin();
      }
      else if( operation('Y') ){ /* Move vertically by dots */
	lj_text_end(ofile);
	lj_undl_flush(ofile);
	switch( pos_code ){
	case POS_ABSOLUTE:	/* Absolute postioning */
	  lj_cursor_ver_abs(ofile,
			    page_height -
			    (margin_top + dt2in(atof(number))));
	  break;
	case POS_REL_POSITIVE:	/* Relative positive */
	  lj_cursor_ver_rel(ofile, dt2in(atof(number))); break;
	case POS_REL_NEGATIVE:	/* Relative negative */
	  lj_cursor_ver_rel(ofile, -(dt2in(atof(number)))); break;
	default:
	  internal_error("illegal positioning code", "");
	}
	lj_undl_mark(ofile);
	lj_text_begin();
      } /* else if( Y ) */
      else{			/* Unknown command */
	warning("unknown command with prefix <esc>*p -- ignoring", "");
      }
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_star_p() */



void
ljcmd_star_r(ifile, ofile)	/* Command: ^[*r start/end graphics */
     FILE *ifile, *ofile;
{

  warning("command family <esc>*r is not implemented -- ignoring", "");

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      break;
    case PARAM_END:
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_star_r() */


void
ljcmd_star_t(ifile, ofile)	/* Command: ^[*t graphics resolution */
     FILE *ifile, *ofile;
{

  warning("command family <esc>*t is not implemented -- ignoring", "");

  do{
      /* Read the next token */
      /* If EOF is reached, then flag an error and return */
    if( (scan() == 0) && (curr_token.code == PARAM_CONTINUE) ){
      error("end-of-file reached while reading parameters", variable);
    }
#ifdef DEBUG
    if( debug ) fprintf(stderr, "{TOKEN:%d}", curr_token.code);
#endif
    switch( curr_token.code ){
    case PARAM_CONTINUE:
      break;
    case PARAM_END:
      break;
    default:
      internal_error("illegal token encountered in parameters", "");
    } /* switch(...) */
  } while( curr_token.code != PARAM_END );

} /* ljcmd_star_t() */
