/* Project:		lj2ps
** File:		ljfonts.c
**
** Author:		Christopher Lishka
** Organization:	Wisconsin State Laboratory of Hygiene
**			Data Processing Dept.
**
** Copyright (C) 1990 by Christopher Lishka.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static char * ModuleID = "Module ljfonts: v1.0, production";

  /* Include files
  */
#include <strings.h>
#include "ljfonts.h"
#include "lj.h"

  /* External definitions
  */

  /* Global variables
  */

  /* Global function list
  */
extern void  lj_match_font();	/* Get a font (closest match) */
extern int   lj_find_font();	/* Find font based on the name */

  /* Local constants
  */
#define MAX_BUFFER  2048
  /* LJ_FONT_ERROR is used to determine if real number attributes (e.g. pitch)
  ** are the same.  0.1 is currently used because otherwise 16.6 does not match
  ** with 16.66.  I have no idea how the LaserJet resolves this problem. - C.L.
  */
#define LJ_FONT_ERROR 0.1 
  /* Font attribute priority values */
#define FAP_ORIENTATION   1024	/* Orientation: landscape/portrait */
#define FAP_SYM_SET        512	/* Symbol set */
#define FAP_SPACING        256	/* Spacing: proportional/fixed */
#define FAP_PITCH          128	/* Pitch (in chars per inch) */
#define FAP_HEIGHT          64	/* Font height (in points) */
#define FAP_STYLE           32	/* Style: normal/italic */
#define FAP_STROKE          16	/* Stroke: -3(light) to 0(normal) to 3(bold) */
#define FAP_TYPEFACE         8	/* Typeface */
#define FAP_SOURCE_SOFT      4	/* Source: downloadable "soft" font */
#define FAP_SOURCE_CART      2	/* Source: cartridge font */
#define FAP_SOURCE_INT       1	/* Source: internal font */

  /* Local structures and types
  */
typedef struct {
    /* Identifier fields */
  int     source;		/* Font source: where the font is located */
  int     number;		/* Font number */
     /* Scaling fields */
  int     scale;		/* True if scaling should be performed */
  double  scale_x;		/* % to scale in x direction */
  double  scale_y;		/* % to scale in y direction */
    /* LaserJet font ID fields */
  int     symbol_set;		/* The symbol set of the font */
  int     spacing;		/* Proportional/fixed */
  double  pitch;		/* Pitch of fixed or space character */
  double  point_size;		/* Height of the character */
  int     style;		/* Upright/italic */
  int     weight;		/* Stroke weight: -3 to +3 */
  int     typeface;		/* Which typeface to use */
  char   *ps_name;		/* The postscript font used for emulation */
} font_node;

  /* Local variables
  */
  /* The table of LaserJet fonts, and the LaserWriter equivalents.
  **
  ** Note: the fonts must be listed in priority order, from most important
  ** to least important.
  **
  ** Note: the pitch value for proportional fonts is the width of the space
  ** character.
  */
/*DBX: static */ font_node lj_fonts[] = {

    /*
    ** Left slot: HP F Font Cartridge
    */
 
    /* L00 (TMS-10): Times-Roman 8U prop. 10pt. upright normal */
  { LJ_FS_LEFT, 0,        0, 1.20, 1.00, /* Note: scaling is OFF */
    LJ_SS_8U, LJ_SP_PROPORTIONAL, 17.7, 10.0, LJ_ST_UPRIGHT, 0, LJ_TF_TIMES,
    "Times-Roman" },

    /* L01 (TMS-10): Times-Roman 0U prop. 10pt. upright normal */
  { LJ_FS_LEFT, 1,        0, 1.20, 1.00, /* Note: scaling is OFF */
    LJ_SS_0U, LJ_SP_PROPORTIONAL, 17.7, 10.0, LJ_ST_UPRIGHT, 0, LJ_TF_TIMES,
    "Times-Roman" },

    /* L02 (TMS-B-10): Times-Roman 0U prop. 10pt. upright bold */
  { LJ_FS_LEFT, 2,        0, 1.20, 1.00, /* Note: scaling is OFF */
    LJ_SS_0U, LJ_SP_PROPORTIONAL, 17.7, 10.0, LJ_ST_UPRIGHT, 3, LJ_TF_TIMES,
    "Times-Bold" },

    /* L03 (TMS-B-10): Times-Roman 8U prop. 10pt. upright bold */
  { LJ_FS_LEFT, 3,        0, 1.20, 1.00, /* Note: scaling is OFF */
    LJ_SS_8U, LJ_SP_PROPORTIONAL, 17.7, 10.0, LJ_ST_UPRIGHT, 3, LJ_TF_TIMES,
    "Times-Bold" },

    /* L04 (TMS-I-10): Times-Roman 0U prop. 10pt. italic normal */
  { LJ_FS_LEFT, 4,        0, 1.20, 1.00, /* Note: scaling is OFF */
    LJ_SS_0U, LJ_SP_PROPORTIONAL, 17.7, 10.0, LJ_ST_ITALIC, 0, LJ_TF_TIMES,
    "Times-Italic" },

    /* L05 (TMS-I-10): Times-Roman 8U prop. 10pt. italic normal */
  { LJ_FS_LEFT, 5,        0, 1.20, 1.00, /* Note: scaling is OFF */
    LJ_SS_8U, LJ_SP_PROPORTIONAL, 17.7, 10.0, LJ_ST_ITALIC, 0, LJ_TF_TIMES,
    "Times-Italic" },

    /* L06 (HELV-B-14.4): Helvetica 0U prop. 14.4pt. upright bold */
  { LJ_FS_LEFT, 6,        0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_0U, LJ_SP_PROPORTIONAL, 15.0, 14.4, LJ_ST_UPRIGHT, 3, LJ_TF_HELV,
    "Helvetica-Bold" },

    /* L07 (HELV-B-14.4): Helvetica 8U prop. 14.4pt. upright bold */
  { LJ_FS_LEFT, 7,        0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_8U, LJ_SP_PROPORTIONAL, 15.0, 14.4, LJ_ST_UPRIGHT, 3, LJ_TF_HELV,
    "Helvetica-Bold" },

    /* L08 (TMS-8): Times-Roman 0U prop. 8pt. upright normal */
  { LJ_FS_LEFT, 8,        0, 1.20, 1.00, /* Note: scaling is OFF */
    LJ_SS_0U, LJ_SP_PROPORTIONAL, 27.4,  8.0, LJ_ST_UPRIGHT, 0, LJ_TF_TIMES,
    "Times-Roman" },

    /* L09 (TMS-8): Times-Roman 8U prop. 8pt. upright normal */
  { LJ_FS_LEFT, 9,        0, 1.20, 1.00, /* Note: scaling is OFF */
    LJ_SS_8U, LJ_SP_PROPORTIONAL, 27.4,  8.0, LJ_ST_UPRIGHT, 0, LJ_TF_TIMES,
    "Times-Roman" },

    /* L10 (LP-8.5): Line-Printer 0U 16.6pitch 8.5pt upright normal */
  { LJ_FS_LEFT, 10,       1, 0.845, 1.00, /* Scaling *****ON***** */
    LJ_SS_0N, LJ_SP_FIXED,        16.67,  8.5, LJ_ST_UPRIGHT, 0, LJ_TF_LP,
    "Courier" },

    /* L11 (LP-8.5): Line-Printer 8U 16.6pitch 8.5pt upright normal */
  { LJ_FS_LEFT, 11,       1, 0.845, 1.00, /* Scaling *****ON***** */
    LJ_SS_8U, LJ_SP_FIXED,        16.67,  8.5, LJ_ST_UPRIGHT, 0, LJ_TF_LP,
    "Courier" },

    /*
    ** Right slot: HP G Font Cartridge
    */

    /* R00 (PRES-10): Prestige 0U 12pitch 10pt. upright normal */
  { LJ_FS_RIGHT, 0,       0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_0U, LJ_SP_FIXED,        12.0, 10.0, LJ_ST_UPRIGHT, 0, LJ_TF_PRES,
    "Courier" },

    /* R01 (PRES-10): Prestige 1U 12pitch 10pt. upright normal */
  { LJ_FS_RIGHT, 1,       0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_1U, LJ_SP_FIXED,        12.0, 10.0, LJ_ST_UPRIGHT, 0, LJ_TF_PRES,
    "Courier" },

    /* R02 (PRES-B-10): Prestige 1U 12pitch 10pt. upright bold */
  { LJ_FS_RIGHT, 2,       0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_1U, LJ_SP_FIXED,        12.0, 10.0, LJ_ST_UPRIGHT, 3, LJ_TF_PRES,
    "Courier-Bold" },

    /* R03 (PRES-B-10): Prestige 0U 12pitch 10pt. upright bold */
  { LJ_FS_RIGHT, 3,       0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_0U, LJ_SP_FIXED,        12.0, 10.0, LJ_ST_UPRIGHT, 3, LJ_TF_PRES,
    "Courier-Bold" },

    /* R04 (PRES-I-10): Prestige 1U 12pitch 10pt. italic normal */
  { LJ_FS_RIGHT, 4,       0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_1U, LJ_SP_FIXED,        12.0, 10.0, LJ_ST_ITALIC, 0, LJ_TF_PRES,
    "Courier-Oblique" },

    /* R05 (PRES-I-10): Prestige 0U 12pitch 10pt. italic normal */
  { LJ_FS_RIGHT, 5,       0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_0U, LJ_SP_FIXED,        12.0, 10.0, LJ_ST_ITALIC, 0, LJ_TF_PRES,
    "Courier-Oblique" },

    /* R06 (PRES-7): Prestige 1U 16.6pitch 7pt. upright normal */
  { LJ_FS_RIGHT, 6,       1, 1.03, 1.00, /* Note: scaling is OFF */
    LJ_SS_1U, LJ_SP_FIXED,        16.67,  7.0, LJ_ST_UPRIGHT, 0, LJ_TF_PRES,
    "Courier" },

    /* R07 (PRES-7): Prestige 0U 16.6pitch 7pt. upright normal */
  { LJ_FS_RIGHT, 7,       1, 1.03, 1.00, /* Note: scaling is OFF */
    LJ_SS_0U, LJ_SP_FIXED,        16.67,  7.0, LJ_ST_UPRIGHT, 0, LJ_TF_PRES,
    "Courier" },

    /* R08: Line Draw is NOT AVAILABLE */
/*  "DRAW-12",         "Line-Draw              12    point     12    pitch",*/
/*  Line draw is not supported! */

    /*
    ** LaserJet II Internal Fonts
    */

    /* I00 (COUR-12): Courier 8U 10pitch 12pt. upright normal */
  { LJ_FS_INTERNAL, 0,    0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_8U, LJ_SP_FIXED,        10.0, 12.0, LJ_ST_UPRIGHT, 0, LJ_TF_COUR,
    "Courier" },

    /* I01 (COUR-12): Courier 10U 10pitch 12pt. upright normal */
  { LJ_FS_INTERNAL, 1,    0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_10U, LJ_SP_FIXED,        10.0, 12.0, LJ_ST_UPRIGHT, 0, LJ_TF_COUR,
    "Courier" },

    /* I02 (COUR-12): Courier 11U 10pitch 12pt. upright normal */
  { LJ_FS_INTERNAL, 2,    0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_11U, LJ_SP_FIXED,        10.0, 12.0, LJ_ST_UPRIGHT, 0, LJ_TF_COUR,
    "Courier" },

    /* I03 (COUR-12): Courier 0N 10pitch 12pt. upright normal */
  { LJ_FS_INTERNAL, 3,    0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_0N, LJ_SP_FIXED,        10.0, 12.0, LJ_ST_UPRIGHT, 0, LJ_TF_COUR,
    "Courier" },

    /* I04 (COUR-B-12): Courier 8U 10pitch 12pt. upright bold */
  { LJ_FS_INTERNAL, 4,    0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_8U, LJ_SP_FIXED,        10.0, 12.0, LJ_ST_UPRIGHT, 3, LJ_TF_COUR,
    "Courier-Bold" },

    /* I05 (COUR-B-12): Courier 10U 10pitch 12pt. upright bold */
  { LJ_FS_INTERNAL, 5,    0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_10U, LJ_SP_FIXED,        10.0, 12.0, LJ_ST_UPRIGHT, 3, LJ_TF_COUR,
    "Courier-Bold" },

    /* I06 (COUR-B-12): Courier 11U 10pitch 12pt. upright bold */
  { LJ_FS_INTERNAL, 6,    0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_11U, LJ_SP_FIXED,        10.0, 12.0, LJ_ST_UPRIGHT, 3, LJ_TF_COUR,
    "Courier-Bold" },

    /* I07 (COUR-B-12): Courier 0N 10pitch 12pt. upright bold */
  { LJ_FS_INTERNAL, 7,    0, 0.00, 0.00, /* Note: scaling is OFF */
    LJ_SS_0N, LJ_SP_FIXED,        10.0, 12.0, LJ_ST_UPRIGHT, 3, LJ_TF_COUR,
    "Courier-Bold" },

    /* I08 (LP-8.5): Line-Printer 8U 16.6pitch 8.5pt. upright normal */
  { LJ_FS_INTERNAL, 8,    1, 0.845, 1.00, /* Scaling *****ON***** */
    LJ_SS_8U, LJ_SP_FIXED,        16.67,  8.5, LJ_ST_UPRIGHT, 0, LJ_TF_LP,
    "Courier" },

    /* I09 (LP-8.5): Line-Printer 10U 16.6pitch 8.5pt. upright normal */
  { LJ_FS_INTERNAL, 9,    1, 0.845, 1.00, /* Scaling *****ON***** */
    LJ_SS_10U, LJ_SP_FIXED,        16.67,  8.5, LJ_ST_UPRIGHT, 0, LJ_TF_LP,
    "Courier" },

    /* Last entry must be nulled out! */
  { 0, 0,                 0, 0.00, 0.00,
    0,         0,                    0.0,  0.0, 0,             0, 0,
    "" }
};

  /* Local macro definitions
  */

  /* Local function list
  */

  /* Function bodies
  */

  /* lj_get_font() returns in the argument font the closest match to the
  ** requested font.  The match *never* fails.
  **
  ** "Closest match" is defined as the first font in the
  ** list for which the attributes match.  The attributes are in the
  ** following priority order:
  **
  **        (1) Orientation (always correct because PS has arbitrary rotation)
  **        (2) Symbol set  (once again, ignored here!)
  **        (3) Spacing
  **        (4) Pitch       (ignored if the font is proportional)
  **        (5) Height
  **        (6) Style
  **        (7) Stroke weight
  **        (8) Typeface
  **        (9) Soft font        (ignored)
  **       (10) Cartridge font   (ignored)
  **       (11) Internal font    (ignored)
  **
  ** Priority schemes based on the source of the font (i.e. soft fonts,
  ** cartridge fonts, or internal fonts) are ignored.  These can be
  ** duplicated through careful ordering of the list.
  */
void
lj_match_font(font)
     ljfont *font;
{
  int    match_font;		/* Highest matched font so far */
  int    match_value;		/* Highest matched value so far */
  int    curr_value;		/* The current font being tested */
  int    curr_font;		/* The match value for the current font */
  double tmp_double;		/* Temporary value */

  curr_font   = 0;
  match_font  = curr_font;
  match_value = 0;
  do{
    curr_value = 0;

      /* Match the attributes */
      /* Orientation: *always* matches, because PS allows arbitrary rotation */
    curr_value += FAP_ORIENTATION;
      /* Symbol set */
    if( font->symbol_set == lj_fonts[curr_font].symbol_set ){
      curr_value += FAP_SYM_SET;
    }
      /* Spacing: match to two decimal places */
    if( font->spacing == lj_fonts[curr_font].spacing ){
      curr_value += FAP_SPACING;
    }
      /* Pitch: ignore if proportional spacing */
    if( font->spacing == LJ_SP_FIXED ){
      tmp_double = font->pitch - lj_fonts[curr_font].pitch;
      if( (-(LJ_FONT_ERROR) <= tmp_double) && (tmp_double <= LJ_FONT_ERROR) ){
	curr_value += FAP_PITCH;
      }
    }
      /* Height */
    tmp_double = font->point_size - lj_fonts[curr_font].point_size;
    if(   (-(LJ_FONT_ERROR) <= tmp_double) && (tmp_double <= LJ_FONT_ERROR) ){
      curr_value += FAP_HEIGHT;
    }
      /* Style: exact match needed */
    if( font->style == lj_fonts[curr_font].style ){
      curr_value += FAP_STYLE;
    }
      /* Stroke weight: exact match needed */
    if( font->weight == lj_fonts[curr_font].weight ){
      curr_value += FAP_STROKE;
    }
      /* Typeface: exact match needed */
    if( font->typeface == lj_fonts[curr_font].typeface ){
      curr_value += FAP_TYPEFACE;
    }
      /* Source soft font: ignored */
      /* Source cartridge font: ignored */
      /* Source internal font: ignored */

      /* Check to see if we currently have a better match */
    if( curr_value > match_value ){
      match_value = curr_value;
      match_font  = curr_font;
    }

    curr_font++;
  } while( (lj_fonts[curr_font].ps_name[0] != '\0') );

    /* There had *better* be a matched font in match_font by this point.
    ** If none of the attributes matched, match_font should at least hold
    ** the default.
    */

    /* OK. Now copy over *everything* */
  font->scale      = lj_fonts[match_font].scale;
  font->scale_x    = lj_fonts[match_font].scale_x;
  font->scale_y    = lj_fonts[match_font].scale_y;
  font->symbol_set = lj_fonts[match_font].symbol_set;
  font->spacing    = lj_fonts[match_font].spacing;
  if( font->spacing == LJ_SP_FIXED ){
    font->pitch      = lj_fonts[match_font].pitch;
  }
  font->point_size = lj_fonts[match_font].point_size;
  font->style      = lj_fonts[match_font].style;
  font->weight     = lj_fonts[match_font].weight;
  font->typeface   = lj_fonts[match_font].typeface;
  (void) strcpy(font->ps_name, lj_fonts[match_font].ps_name);
  font->width      = 1/(lj_fonts[match_font].pitch);

} /* lj_match_font() */



  /* lj_find_font() finds a font based on the name.  A 0 is returned
  ** if the search is successful, 1 otherwise.
  */
int
lj_find_font(source, number, font)
     int source, number;  ljfont *font;
{
  int found;
  int counter;

  found = 0;
  for( counter = 0;
       !found && !(lj_fonts[counter].source == 0);
       counter++ ){
    if(   (source == lj_fonts[counter].source)
       && (number == lj_fonts[counter].number) ){
      found = 1;
      font->scale      = lj_fonts[counter].scale;
      font->scale_x    = lj_fonts[counter].scale_x;
      font->scale_y    = lj_fonts[counter].scale_y;
      font->symbol_set = lj_fonts[counter].symbol_set;
      font->spacing    = lj_fonts[counter].spacing;
      if( font->spacing == LJ_SP_FIXED ){
	font->pitch    = lj_fonts[counter].pitch;
      }
      font->point_size = lj_fonts[counter].point_size;
      font->style      = lj_fonts[counter].style;
      font->weight     = lj_fonts[counter].weight;
      font->typeface   = lj_fonts[counter].typeface;
      (void) strcpy(font->ps_name, lj_fonts[counter].ps_name);
      font->width      = 1/(lj_fonts[counter].pitch);
    } /* if */
  } /* for */

  return( !found );
} /* lj_find_font() */
