/*
** Project:        lj2ps
** File:	   lj.h
**
** Author:	   Christopher Lishka
** Organization:   Wisconsin State Laboratory of Hygiene
**		   Data Processing Dept.
**
** Copyright (C) 1990 by Christopher Lishka.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef LJ_H
#define LJ_H

  /* Global constants
  */
  /* Implementation constraints */
#define LJ_STRING_SIZE  256
#define LJ_ERROR  0.00001	/* Accuracy of floating point numbers */
  /* Default scaling constants */
#define LJ_DEFAULT_SCALE_X   1.0000
#define LJ_DEFAULT_SCALE_Y   0.99
  /* Default offset constants */
#define LJ_DEFAULT_OFFSET_X  0.2500
#define LJ_DEFAULT_OFFSET_Y  0.0000
  /* Measurements */
#define in   1			/* Inches (base unit of measurement) */
#define dt 300			/* Dots per inch */
#define dp 720			/* Decipoints */
#define hi 120			/* Horizontal Index Unit */
#define vi  48			/* Vertical Index Unit */
#define pt  72			/* Points */
  /* Tab stops */
#define LJ_TAB_WIDTH 8		/* Tabs occur every eight characters */
  /* Internal codes */
#define LJ_UL_OFF             100 /* Underline codes */
#define LJ_UL_ON              101
#define LJ_UL_FLOAT           102
#define LJ_FI_RULE            200 /* Fill codes      Rule */
#define LJ_FI_2               201 /*              2% gray */
#define LJ_FI_10              202 /*             10% gray */
#define LJ_FI_15              203 /*             15% gray */
#define LJ_FI_30              204 /*             30% gray */
#define LJ_FI_45              205 /*             45% gray */
#define LJ_FI_70              206 /*             70% gray */
#define LJ_FI_90              207 /*             90% gray */
#define LJ_FI_100             208 /*            100% gray */
#define LJ_FI_LINE_H          209 /*     Horizontal lines */
#define LJ_FI_LINE_V          210 /*     Vertical lines */
#define LJ_FI_DIAG_1          211 /*     Diagonals 1 */
#define LJ_FI_DIAG_2          212 /*     Diagonals 2 */
#define LJ_FI_GRID_SQ         213 /*     Square grid */
#define LJ_FI_GRID_DIAG       214 /*     Diagonal grid */
#define LJ_LT_NORM            301 /* Line termination    normal */
#define LJ_LT_CR              302 /* cr=cr+lf lf=lf ff=ff */
#define LJ_LT_LF_FF           303 /* cr=cr lf=cr+lf ff=cr+ff */
#define LJ_LT_CR_LF_FF        304 /* cr=cr+lf lf=cr+lf ff=cr+ff */
#define LJ_PS_TRAY_1          301 /* Paper source */
#define LJ_PS_TRAY_2          302
#define LJ_PS_MANUAL          303
#define LJ_PS_MANUAL_ENVELOPE 304
#define LJ_OR_PORTRAIT        401 /* Orientation */
#define LJ_OR_LANDSCAPE       402
#define LJ_SS_0B              501 /* Symbol set: bit vector entries */
#define LJ_SS_0N              502
#define LJ_SS_0U              503
#define LJ_SS_1U              504
#define LJ_SS_8U              505
#define LJ_SS_10U             506
#define LJ_SS_11U             507
#define LJ_TF_HELV            601 /* Typefaces */
#define LJ_TF_TIMES           602
#define LJ_TF_PRES            603
#define LJ_TF_LP              604
#define LJ_TF_COUR            605
#define LJ_SP_PROPORTIONAL    701 /* Spacing */
#define LJ_SP_FIXED           702
#define LJ_ST_UPRIGHT         801 /* Font style */
#define LJ_ST_ITALIC          802
#define LJ_FT_PRIMARY         901 /* Primary/secondary current font */
#define LJ_FT_SECONDARY       902
#define LJ_PS_EXECUTIVE       1001 /* Page size codes */
#define LJ_PS_LETTER          1002
#define LJ_PS_LEGAL           1003
#define LJ_PS_A4              1004
#define LJ_PS_MONARCH         1005
#define LJ_PS_COMMERCIAL10    1006
#define LJ_PS_INTERNATIONALDL 1007
#define LJ_PS_INTERNATIONALC5 1008
#define LJ_FS_INTERNAL        1101 /* Font sources */
#define LJ_FS_LEFT            1102
#define LJ_FS_RIGHT           1103
#define LJ_FS_SOFT            1104

  /* Global structure and type definitions
  */

  /* Data required for a laserjet font */
typedef struct {
    /* Identifier fields */
  int     source;		/* Font source: where the font is located */
  int     number;		/* Font number */
     /* Scaling fields */
  int     scale;		/* True if scaling should be performed */
  double  scale_x;		/* % to scale in x direction */
  double  scale_y;		/* % to scale in y direction */
    /* LaserJet font ID fields */
  int     symbol_set;		/* The symbol set of the font */
  int     spacing;		/* Proportional/fixed */
  double  pitch;		/* (Only valid for fixed width fonts!) */
  double  point_size;		/* Height of the character */
  int     style;		/* Upright/italic */
  int     weight;		/* Stroke weight: -3 to +3 */
  int     typeface;		/* Which typeface to use */
  char    ps_name[LJ_STRING_SIZE]; /* The postscript font used for emulation */
  double  width;		/* Width of a character */
} ljfont;
  /* Data required to determine paper size */
typedef struct {
  int     code;			/* Type of paper: letter, legal, A4, etc. */
  double  width;		/* How wide (inches) */
  double  height;		/* How high (inches) */
  int     default_form;		/* Default value for form variable */
  double  unprint_top;		/* Unprintable region - top */
  double  unprint_bottom;	/* Unprintable region - bottom */
  double  unprint_left;		/* Unprintable region - left */
  double  unprint_right;	/* Unprintable region - right */
} psize;

  /* Global variables
  */
  /* Front panel variables */
extern int    panel_copies;	/*  Front panel: Copies */
extern int    panel_manual_feed; /* Front panel: Manual Feed */
extern int    panel_font_source; /* Front panel: Font Source */
extern int    panel_font_number; /* Front panel: Font Number */
extern int    panel_orientation; /* Front panel: Font Source, Font Number */
extern int    panel_form;	/*  Front panel: Form (i.e. lpi) */
  /* Postscript specific variables */
extern double ps_scale_x;	/* PS: scale in x direction */
extern double ps_scale_y;	/* PS: scale in y direction */
extern double ps_offset_x;	/* PS: offset in x direction */
extern double ps_offset_y;	/* PS: offset in y direction */
  /* Job control */
extern int     copies;		/* Number of copies */
  /* Page control */
extern int     paper_source;	/* Where the paper is coming from */
extern int     orientation;	/* Portrait/landscape	      (code)	  */
extern psize   page_size;	/* Type of paper being used   (structure) */
extern double  page_height;	/* Height of physical page    (inches)	  */
extern double  page_width;	/* Width of physical page     (inches)	  */
extern double  char_height;	/* Char height (~= VMI)	      (inches)	  */
extern double  char_width;	/* Char width  (~= HMI)	      (inches)	  */
extern double  margin_top;	/* Top margin                 (inches)	  */
extern int     text_length;	/* Text length		      (lines)	  */
extern double  text_height;	/* Text height                (inches)	  */
extern double  margin_left;	/* Left margin                (inches)    */
extern double  text_width;	/* Text width                 (inches)	  */
extern int     perf_skip;	/* 1 if perforation skip is on */
extern int     line_term;	/* Current line termination mode */
  /* Fonts */
extern ljfont  font_p;		/* Primary font */
extern ljfont  font_s;		/* Secondary font */
extern int     underline;	/* Underline mode */
  /* Font management */
extern int     font_ID;		/* Current font ID */
extern int     char_code;	/* Current character code */
  /* Raster graphics */
extern int     resolution;	/* Current graphics resolution */
extern int     margin_graphics;	/* Graphics left margin, in dots? */
  /* Rectangular area fill */
extern int     rect_size_h;	/* Horizontal rectangle size */
extern int     rect_size_v;	/* Vertical rectangle size */
extern int     fill_ID;		/* Current area fill ID */
  /* Macro */
extern int     macro_ID;	/* Current macro ID */
  /* Troubleshooting commands */
extern int     eol_wrap;	/* End-of-line wrap on/off */
extern int     display_func;	/* Display functions on/off */
  /* Implementation variables */
extern double  current_x;	/* Current X position         (inches)	  */
extern double  current_y;	/* Current Y position         (inches)	  */
extern int     current_line;	/* Current line being printed (lines)     */
extern int     current_font;	/* Primary or secondary	      (code)	  */
extern int     empty_line;	/* True if line is empty      (boolean)   */
extern int     empty_text;	/* True if text is empty      (boolean)   */
extern int     empty_page;	/* True if page is empty      (boolean)	  */
extern double  compress_width;	/* Compression in x axis      (%)	  */
extern double  compress_height;	/* Compression in y axis      (%)	  */
extern double  offset_width;	/* Offset of x axis           (inches)    */
extern double  offset_height;	/* Offset of y axis           (inches)    */

  /* Global macro definitions
  */
  /* Conversion macros */
#define dt2in(x)  (x/dt)	/* Dots to inches */
#define in2dt(x)  (x*dt)	/* Inches to dots */
#define dp2in(x)  (x/dp)	/* Decipoints to inches */
#define in2dp(x)  (x*dp)	/* Inches to decipoints */
#define hi2in(x)  (x/hi)	/* Horizontal index units to inches */
#define in2hi(x)  (x*hi)	/* Inches to horizontal index units */
#define vi2in(x)  (x/vi)	/* Vertical index units to inches */
#define in2vi(x)  (x*vi)	/* Inches to vertical index units */
#define pt2in(x)  (x/pt)	/* Points to inches */
#define in2pt(x)  (x*pt)	/* Inches to points */
#define co2in(x)  (x*char_width) /* Columns to inches */
#define in2co(x)  (x/char_width) /* Inches to columns */
#define ro2in(x)  (x*char_height) /* Rows to inches */
#define in2ro(x)  (x/char_height) /* Inches to rows */
  /* Emulation macros */
  /* Text buffer macros */
#define lj_text_begin()   \
  ( empty_text = 1 )
#define lj_text_add(F, X) \
  ( (empty_text?fprintf((F),"(%s",X):fputs((X),(F))), \
   empty_text = 0, empty_line = 0, empty_page = 0 )
#define lj_text_end(F) \
  ( (!empty_text) \
   ? fputs(")S ",F) \
   : 0)
  /* Cursor positioning macros */
#define lj_cursor_abs(F, X, Y) \
  (current_x = (X), current_y = (Y), \
   fprintf(F, "%.4f %.4f M ", current_x, current_y))
#define lj_cursor_hor_abs(F, X) \
  (current_x = (X), fprintf(F, "%.4f %.4f M ", current_x, current_y))
#define lj_cursor_hor_rel(F, dX) \
  (fprintf(F, "%.4f HRM ", (dX)))
#define lj_cursor_ver_abs(F, Y) \
  (current_y = (Y), fprintf(F, "%.4f VM ", current_y))
#define lj_cursor_ver_rel(F, dY) \
  (current_y -= (dY), fprintf(F, "%.4f VM ", current_y))
  /* Font macros */
#define lj_set_font(Z, X) \
  ( char_width = (X).width, \
   ( (X).scale \
   ? fprintf((Z), "/%s %.4f %.4f FS ", (X).ps_name, \
	     pt2in((X).point_size) * (X).scale_x, \
	     pt2in((X).point_size) * (X).scale_y) \
   : fprintf((Z), "/%s %.4f F ", \
	     (X).ps_name, \
	     pt2in((X).point_size)) ) )
  /* Underlining macros */
#define lj_undl_begin(F)  (underline = LJ_UL_ON,  fputs("US ", (F)))
#define lj_undl_mark(F) \
  ( (underline == LJ_UL_ON) \
   ? (fputs("US ", (F))) \
   : 0 )
#define lj_undl_flush(F) \
  ( (underline == LJ_UL_ON) \
   ? (fputs("UE ", (F))) \
   : 0 )
#define lj_undl_end(F)    (underline = LJ_UL_OFF, fputs("UE ", (F)))
  /* Number of copies */
#define lj_copies(F, C) \
  ( fprintf((F), "/#copies %d def ", (C)) )

  /* Global functions
  */
extern void  lj_factory_setup(); /* Reset to factory defaults */
extern void  lj_startup();	/* Do before the virtual LJ is ready... */
extern void  lj_shutdown();	/* Do after the virtual LJ is shutdown... */
extern void  lj_page_begin();	/* Start a new page */
extern void  lj_page_end();	/* End the current page */
/*     macro lj_text_begin();*/ /* Start text */
/*     macro lj_text_add(); */  /* Add text to the buffer */
/*     macro lj_text_end(); */  /* End text */
/*     macro lj_cursor_abs();*/ /* Absolute X,Y cursor positioning */
/*     macro lj_cursor_hor_abs(); */ /* Absolute X cursor positioning */
/*     macro lj_cursor_hor_rel(); */ /* Relative X cursor positioning */
/*     macro lj_cursor_ver_abs(); */ /* Absolute Y cursor positioning */
/*     macro lj_cursor_ver_rel(); */ /* Relative Y cursor positioning */
/*     macro lj_set_font(); */  /* Use a new font to print the text */
/*     macro lj_undl_begin();*/	/* Start underlining */
/*     macro lj_undl_mark();*/  /* Mark the start of a new underline */
/*     macro lj_undl_flush();*/	/* Flush the current underline buffer */
/*     macro lj_undl_end(); */	/* End underlining */
/*     macro lj_copies(); */    /* Set the number of copies */
extern void  lj_nl();		/* Print a newline */
extern void  lj_reset();	/* Reset the printer to default state */
extern int   lj_paper_size();	/* Get the dimensions of paper and envelopes */

#endif

