/*
        HPGL to PostScript converter
   Copyright (C) 1988 (and following) Federico Heinz

yahp2ps is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY.  No author or distributor accepts responsibility to anyone
for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.
Refer to the Free Software Foundation's General Public License for full details.

Everyone is granted permission to copy, modify and redistribute yahp2ps,
but only under the conditions described in the GNU General Public
License.  A copy of this license is supposed to have been given to you
along with yahp2ps so you can know your rights and responsibilities.  It
should be in a file named COPYING.  Among other things, the copyright
notice and this notice must be preserved on all copies.

In other words, go ahead and share yahp2ps, but don't try to stop
anyone else from sharing it farther.  Help stamp out software hoarding!

yahp2ps is TOTALLY unrelated to GNU or the Free Software Foundation,
it is only released under the same conditions.

    For bug reports, wishes, etc. send e-mail to

    ...!mcvax!unido!tub!actisb!federico  (from Europe)
    ...!uunet!pyramid!actisb!federico    (from anywhere else)

    For Physical mail:

    Federico Heinz
    Beusselstr. 21
    1000 Berlin 21

    Tel. (+49 30) 396 77 92

*/
/***************************************************************************
   Mapping of plotter primitives onto PostScript. All coordinates are
   considered to be in plotter space.

***************************************************************************/

#include <stdio.h>
#include "defs.h"
#include "penctrl.h"
#include "io.h"
#include "mchinery.h"
#include "circle.h"
#include "shade.h"


static Number PenWidth;		/* Current pen's width in plotter units */
static Number PenColor;		/* Current Pen's color			*/
static Boolean StrokePending;   /* Should a stroke be done on penUp?    */

static Boolean PenWidthChange;	/* Pen width changed (for delayed change) */
static Boolean PatternChange;	/* Pattern changed (for delayed change)   */
static Boolean ColorChange;	/* Color changed (for delayed change)	  */


/************************************************************************

   Different line patterns.

***********************************************************************/



static void pattern1(length)

Number length;

{ if (length <= PenWidth) return;
  writeNumber(Zero);
  writeNumber(length);
}



static void pattern2(length)

Number length;

{ Number halfLength;

  halfLength = length/2;
  if (halfLength <= PenWidth) return;
  writeNumber(halfLength);
  writeNumber(halfLength);
}



static void pattern3(length)

Number length;

{ Number threeQuarterLength;

  if (length/4 <= PenWidth) return;
  threeQuarterLength = (3 * length) / 4;
  writeNumber(threeQuarterLength);
  writeNumber(length - threeQuarterLength);
}



static void pattern4(length)

Number length;

{ Number fourFifthLength, whiteSpace;

  fourFifthLength = (4 * length) / 5;
  whiteSpace = (length - fourFifthLength - PenWidth) / 2;
  if (whiteSpace <= PenWidth) return;
  writeNumber(fourFifthLength);
  writeNumber(whiteSpace);
  writeNumber(Zero);
  writeNumber(length - (fourFifthLength + whiteSpace));
}



static void pattern5(length)

Number length;

{ Number threeQuarterLength, dashLength;

  threeQuarterLength = (3 * length) / 4;
  dashLength = (length - threeQuarterLength) / 3;
  if (dashLength <= PenWidth) return;
  writeNumber(threeQuarterLength);
  writeNumber(dashLength);
  writeNumber(dashLength);
  writeNumber(length - (threeQuarterLength + 2 * dashLength));
}



static void pattern6(length)

Number length;

{ Number halfLength, dashLength;

  halfLength = length/2;
  dashLength = halfLength/5;
  if (dashLength <= PenWidth) return;
  writeNumber(halfLength);
  writeNumber(dashLength);
  writeNumber(dashLength);
  writeNumber(dashLength);
  writeNumber(dashLength);
  writeNumber(length - (halfLength + 4 * dashLength));
}



typedef void (*DashFnct)();

static DashFnct DashPattern[] =
	{ pattern1, pattern2, pattern3,
	  pattern4, pattern5, pattern6 };


static DashFnct NewDashFnct;	/* Pointer to function to change pattern */
static Number PatternLength;	/* Total length of a cicle		 */


/*

  Change the line pattern.

*/

void setPattern(pattern, patternLength)

Number pattern, patternLength;

{ if ((pattern > Zero) && (pattern != FullLine))
    NewDashFnct = DashPattern[(pattern/One)-1];
  else
    NewDashFnct = NULL;
  PatternLength = patternLength;
  PatternChange = True;
}



/***************************************************************************

   PostScript pen control.

***************************************************************************/



/*

  Draw a line for internal purposes. Call only after doing a stroke()!

*/

static void invisibleLine(toX, toY)

{ writeNumber(toX);
  writeNumber(toY);
  writeString("L\n");
}



/*

  Do the real ink-dropping if needed.

*/

void stroke()

{ if (StrokePending)
  { if (PenWidthChange)
    { writeNumber(PenWidth);
      writeString("W\n");
      PenWidthChange = False;
    }
    if (ColorChange)
    { writeNumber(PenColor);
      writeString("G\n");
      ColorChange = False;
    }
    if (PatternChange)
    { writeString("[");
      if (NewDashFnct != NULL)
        (*NewDashFnct)(PatternLength);
      writeString("] D\n");
      PatternChange = False;
    }
    writeString("S\n");
    StrokePending = False;
  }
}



/*

  Draw a line from the current pen position to the target.

*/

void drawLine(targetX, targetY)

Number targetX, targetY;

{ writeNumber(targetX);
  writeNumber(targetY);
  writeString("L\n");
  StrokePending = True;
}


/*

  Draw a dot at current pen position.

*/

void drawDot()
{ writeString("CD\n");
}


/*

  Move the PostScript pen to the specified coordinates.

*/

void setCurrentPoint(targetX, targetY)

Number targetX, targetY;

{ writeNumber(targetX);
  writeNumber(targetY);
  writeString("M\n");
}


/**************************************************************************

  Rotate/unrotate coordinate System

**************************************************************************/

/*

   Rotate the coordinate system.

*/

void doRotation()

{ writeString("POR\n"); }



/*

  Undo the rotation.

*/

void undoRotation()

{ writeString("LND\n"); }
 


/**************************************************************************

   Set/reset clipping path.

**************************************************************************/

static CoordinatePair ClipCorner1, ClipCorner2;


/*

  Set the clipping window to the given rectangle.

*/

void setClip(corner1, corner2)

CoordinatePair corner1, corner2;

{ Boolean penWasDown;

  if (penWasDown = !PenIsUp)
    liftPen();
  writeString("RC\n");
  setCurrentPoint(corner1[X], corner1[Y]);
  invisibleLine(corner1[X], corner2[Y]);
  invisibleLine(corner2[X], corner2[Y]);
  invisibleLine(corner2[X], corner1[Y]);
  invisibleLine(corner1[X], corner1[Y]);
  writeString("SC\n");
  ClipCorner1[X] = corner1[X];
  ClipCorner1[Y] = corner1[Y];
  ClipCorner2[X] = corner2[X];
  ClipCorner2[Y] = corner2[Y];
  if (penWasDown)
    lowerPen();
}



/*

  Make sure the clipping path is the one that was last set.

*/

void correctClip()

{ setClip(ClipCorner1, ClipCorner2);
}


/**************************************************************************

  Initialize/shut down the machinery

***************************************************************************/

char *PreludeFile = DEFAULT_PRELUDE;


void initializeMachinery()

{ FILE *fp;
  char c;

  if (PreludeFile == NULL) return;
  if ((fp = fopen(PreludeFile, "r")) == NULL)
    error("Cannot open prelude file.");
  while ((c = getc(fp)) != EOF)
    putChar(c);
  fclose(fp);
}



void shutdownMachinery()

{ stroke();
  writeString("showpage\n");
}



/**************************************************************************

  Change the parameters of the pen we're plotting with.

**************************************************************************/


/*

  From now on, the pen will be 'width' mm thick.

*/

void setPenWidth(width)

Number width;

{ PenWidth = width * PlotterUnitsFactor;
  PenWidthChange = True;
}



/*

  From now on, the pen has the given color.

*/

void setPenColor(color)

Number color;

{ PenColor = color;
  ColorChange = True;
}



/**************************************************************************

  Filling squares & wedges

**************************************************************************/



static void fillPath(fillType, spacing, angle)

Number fillType, spacing, angle;

{ writeString("C ");
  if (fillType < ParallelLines)
    writeString("fill\n");
  else
  { writeString("gsave clip\n");
    if (angle)
    { writeNumber(angle);
      writeString("R\n");
    }
    writeString("T ");
    writeNumber(spacing);
    if (fillType == ParallelLines)
      writeString("false ");
    else
      writeString("true ");
    writeString("H\n");
  }
}



/*

  Shade a rectangle with one corner at current position and the opposite
  at the specified absolute (but maybe User) CoordinatePair.

*/


void doShadeRectangle(oppositeCorner, fillType, spacing, angle)

CoordinatePair oppositeCorner;
Number fillType, spacing, angle;

{ CoordinatePair thisCorner;
  Boolean penWasDown;

  if (penWasDown = !PenIsUp)
    liftPen();
  setCurrentPoint(PenPosition[X], PenPosition[Y]);
  thisCorner[X] = PenPosition[X]; thisCorner[Y] = PenPosition[Y];
  invisibleLine(plotterXCoord(thisCorner[X]), 
                plotterYCoord(oppositeCorner[Y]));
  invisibleLine(plotterXCoord(oppositeCorner[X]),
                plotterYCoord(oppositeCorner[Y]));
  invisibleLine(plotterXCoord(oppositeCorner[X]),
                plotterYCoord(thisCorner[Y]));
  invisibleLine(plotterXCoord(thisCorner[X]),
                plotterYCoord(thisCorner[Y]));
  fillPath(fillType, spacing, angle);
  if (penWasDown) lowerPen();
}


/*

  Shade a circle wedge.
  The chordAngle should divide the sweepAngle and (of course)
  be != 0. If it does not divide the sweepAngle, then the last chord
  will be shorter.

*/

void doShadeWedge(center, radius, startAngle, sweepAngle, chordAngle,
		fillType, spacing, angle)

CoordinatePair center;
Number radius, startAngle, sweepAngle, chordAngle;

{ Number currentAngle, endAngle;
  CoordinatePair thisPoint;
  Boolean penWasDown;
  int times;

  if (penWasDown = !PenIsUp)
    liftPen();
  else
    setCurrentPoint(plotterXCoord(PenPosition[X]),
                    plotterYCoord(PenPosition[Y]));
  polarToCartesian(thisPoint, center, radius, startAngle);
  invisibleLine(plotterXCoord(thisPoint[X]),
                plotterYCoord(thisPoint[Y]));
  times = sweepAngle / chordAngle;
  if (times < 0)
  { chordAngle = -chordAngle;
    times = -times;
  }
  currentAngle = startAngle;
  while (times--)
  { currentAngle = currentAngle + chordAngle;
    polarToCartesian(thisPoint, center, radius, currentAngle);
    invisibleLine(plotterXCoord(thisPoint[X]),
                  plotterYCoord(thisPoint[Y]));
  }
  if (sweepAngle % chordAngle)
  { polarToCartesian(thisPoint, center, radius, startAngle + sweepAngle);
    invisibleLine(thisPoint[X], thisPoint[Y]);
  }
  invisibleLine(plotterXCoord(center[X]),
                plotterYCoord(center[Y]));
  fillPath(fillType, spacing, angle);
  if (penWasDown) lowerPen();
}



/*****************************************************************************

    Maximum resolution arc stuff

*****************************************************************************/



/*

  Distort plotter space so as to reflex the user coordinate system. This is
  intended as a means for doing arcs at maximum resolution in user space.
  IT IS NOT ALLOWED TO DO TWO distortSpace CALLS IN A ROW. The only way is
  distort, undistort, and distort again.

*/

static void distortSpace()

{ writeNumber(XScaleFactor);
  writeNumber(YScaleFactor);
  writeNumber(XOrigin);
  writeNumber(YOrigin);
  writeString("TS\n");
}



/*

  Undo the distortion caused by the last call to distortSpace.

*/

static void undistortSpace()

{ writeString("SSM\n");
}



/*

   Do an arc at maximum resolution. Center and radius are in user units.

*/

void doHRArc(center, radius, startAngle, sweepAngle)

{ if (UserMode)
    distortSpace();
  writeCoordinatePair(center);
  writeNumber(radius);
  writeNumber(startAngle);
  writeNumber(startAngle + sweepAngle);
  writeString("A\n");
  StrokePending = True;
  if (UserMode)
    undistortSpace();
}



/****************************************************************************

  Tick drawing. At list one of the legths must be != Zero.

****************************************************************************/



/*

  Draw a tick.

*/

static void drawTick(fromX, fromY, toX, toY)

Number fromX, fromY, toX, toY;

{ writeString("TB ");
  writeNumber(fromX);
  writeNumber(fromY);
  writeString("M ");
  writeNumber(toX);
  writeNumber(toY);
  writeString("TE\n");
}



/*

  Do a horizontal tick.

*/

void doYTick(positiveLength, negativeLength)

Number positiveLength, negativeLength;

{ Number plotterXPenPosition, plotterYPenPosition;

  plotterXPenPosition = plotterXCoord(PenPosition[X]);
  plotterYPenPosition = plotterYCoord(PenPosition[Y]);
  drawTick(plotterXPenPosition - negativeLength, plotterYPenPosition,
           plotterXPenPosition + positiveLength, plotterYPenPosition);
}



/*

  Do a vertical tick.

*/

void doXTick(positiveLength, negativeLength)

Number positiveLength, negativeLength;

{ Number plotterXPenPosition, plotterYPenPosition;

  plotterXPenPosition = plotterXCoord(PenPosition[X]);
  plotterYPenPosition = plotterYCoord(PenPosition[Y]);
  drawTick(plotterXPenPosition, plotterYPenPosition - negativeLength,
           plotterXPenPosition, plotterYPenPosition + positiveLength);
}

