/*
        HPGL to PostScript converter
   Copyright (C) 1988 (and following) Federico Heinz

yahp2ps is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY.  No author or distributor accepts responsibility to anyone
for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.
Refer to the Free Software Foundation's General Public License for full details.

Everyone is granted permission to copy, modify and redistribute yahp2ps,
but only under the conditions described in the GNU General Public
License.  A copy of this license is supposed to have been given to you
along with yahp2ps so you can know your rights and responsibilities.  It
should be in a file named COPYING.  Among other things, the copyright
notice and this notice must be preserved on all copies.

In other words, go ahead and share yahp2ps, but don't try to stop
anyone else from sharing it farther.  Help stamp out software hoarding!

yahp2ps is TOTALLY unrelated to GNU or the Free Software Foundation,
it is only released under the same conditions.

    For bug reports, wishes, etc. send e-mail to

    ...!mcvax!unido!tub!actisb!federico  (from Europe)
    ...!uunet!pyramid!actisb!federico    (from anywhere else)

    For Physical mail:

    Federico Heinz
    Beusselstr. 21
    1000 Berlin 21

    Tel. (+49 30) 396 77 92

*/
/**************************************************************************

  Input/Output functions.

**************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include "defs.h"
#include "dispatch.h"
#include "io.h"

int LookAhead = 0;	/* Lookahead feature buffer */

/*************************************************************************

    Error handling

*************************************************************************/



/*

  Put a non-fatal warning trough stderr.

*/

void warning(text)

char *text;

{ fprintf(stderr, "hpgl2ps WARNING: %s\n", text);
}



/*

  Report a fatal error trough stderr.

*/

void error(text)

char *text;

{ fprintf(stderr, "hpgl2ps ERROR: %s\n", text);
  exit(1);
}


/**************************************************************************

    Basic file I/O

**************************************************************************/


static FILE *input = stdin;

static FILE *output = stdout;


/*

  Get the next usable char from the file, skipping escape sequences.

*/

static int getNextC()

{ int c;

  c = getc(input);
  while (c == '\033')
  { (void)getc(input); /* Ignore dot */
    switch (getc(input))
    { case '@':
      case 'H':
      case 'I':
      case 'M':
      case 'N': while (getc(input) != ':') ;
      case '(':
      case 'Y':
      case ')':
      case 'Z':
      case 'B':
      case 'E':
      case 'J':
      case 'K':
      case 'L':
      case 'O':
      case 'R': break;
      default: warning("Illegal escape sequence.");
    }
    c = getc(input);
    while(isTerminator(c)) c = getc(input);
  }
  return(c);
}
 
          

/*

  Get the next char from the input stream with lookahead

*/

int getChar()

{ int   aChar;

  if (!LookAhead) LookAhead = getNextC();
  aChar = LookAhead;
  if (LookAhead != EOF) LookAhead = getNextC();
  return(aChar);
}




/*

  Put a char to the output stream.

*/

void putChar(c)

char c;

{ putc(c, output); }



/*************************************************************************

   HPGL syntax sensitive stuff

*************************************************************************/



/*

  Skip over optional separator.

*/

void skipSeparator()

{ while (isSeparator(LookAhead)) (void)getChar(); }



/*

  Skip over statement terminator.

*/

void skipTerminator()

{ while (isTerminator(LookAhead))
  { (void)getChar();
    skipSeparator();
  }
}


/*************************************************************************

  Number input

*************************************************************************/


/* Powers of ten */

static  long int TenToThe[] = { 1L, 10L, 100L, 1000L, 10000L };



/*

  Read a Number from the input stream.

*/


Boolean getNumber(target)

Number *target;

{ Number result;
  int    placesLeft, decimalsLeft;
  Boolean negative;

  if (LookAhead == '+')
  { negative = False;
    getChar();
  }
  else if (LookAhead == '-')
  { negative = True;
    getChar();
  }
  if (!isdigit(LookAhead) && (LookAhead != '.'))
  { warning("Number expected.");
    return(False);
  }
  result = 0; placesLeft = IntPlaces;
  while (isdigit(LookAhead) && placesLeft)
  { result = result * 10L + (Number)(getChar() - '0');
    placesLeft--;
  }
  if (isdigit(LookAhead)) 
  { warning("Number out of range.");
    return(False);
  }
  decimalsLeft = DecPlaces;
  if (LookAhead == '.')
  { (void)getChar();
    while(isdigit(LookAhead) && decimalsLeft)
    { result = result * 10L + (Number)(getChar() - '0');
      decimalsLeft--;
    }
    if (isdigit(LookAhead))
    { warning("Extra decimal places skipped.");
      while(isdigit(LookAhead)) (void)getChar();
    }
  }
  skipSeparator();
  *target = (negative ? -result : result) * TenToThe[decimalsLeft];
  return(True);
}


/*

  Convert from ASCII to Number.

*/

Boolean ASCIIToNumber(target, source)

Number *target;
char **source;

{ Number result;
  int    placesLeft, decimalsLeft;

  result = 0; placesLeft = IntPlaces;
  while (isdigit(**source) && placesLeft)
  { result = result * 10L + (Number)((*((*source)++)) - '0');
    placesLeft--;
  }
  if (isdigit(**source))
    return(False);
  decimalsLeft = DecPlaces;
  if (**source == '.')
  { (*source)++;
    while(isdigit(**source) && decimalsLeft)
    { result = result * 10L + (Number)((*((*source)++)) - '0');
      decimalsLeft--;
    }
    while(isdigit(**source)) (*source)++;
  }
  *target = result * TenToThe[decimalsLeft];
  return(True);
}



/*

  Read an Integer

*/

Boolean getInteger(target)

Number *target;

{ Number result;

  if (getNumber(&result))
  { *target = trunc(result);
    return(True);
  }
  else
    return(False);
}



/*

  Read a CoordinatePair.

*/

Boolean getCoordinatePair(target)

CoordinatePair target;

{ if (!getNumber(&target[X])) return(False);
  skipSeparator();
  if (!isNumeric(LookAhead))
  { warning("Uneven number of coordinates.");
    return(False);
  }
  return(getNumber(&target[Y]));
}


/***************************************************************************

   Number output

**************************************************************************/

   


/*

  Private routine to put a positive integer to the output stream. A zero
  produces no output at all. No space is appended at the end.

*/

static void putInt(n)

int n;

{ if (n)
  { putInt(n / 10);
    putChar((n % 10) + '0');
  }
}



/*

  Write a Number to the output stream.

*/

void writeNumber(n)

Number n;

{ int integerPart, fractionalPart;

  if (n == Zero)
    putChar('0');
  else
  { if (n < Zero)
    { putChar('-');
      n = -n;
    }
    integerPart = (int)(n / One);
    fractionalPart = (int)(n % One);
    putInt(integerPart);
    if (fractionalPart)
    { putChar('.');
      while (!(fractionalPart % 10)) fractionalPart /= 10;
      putInt(fractionalPart);
    }
  }
  putChar(' ');
}



/*

  Write a CoordinatePair to the output stream.

*/

void writeCoordinatePair(source)

CoordinatePair  source;

{ writeNumber(source[X]);
  writeNumber(source[Y]);
}



/*

  Write an integer to the output string.

*/

void writeInt(n)

int n;

{ if (n = 0)
    putChar('0');
  else
    if (n < 0)
    { putChar('-');
      n = -n;
    }
    putInt(n);
  putChar(' ');
}


/**************************************************************************

  String output.

**************************************************************************/



/*

  Write a string to the output stream.

*/

void writeString(x)

char *x;

{ while (*x) putChar(*(x++)); }


/*************************************************************************

   Command input

*************************************************************************/



/*

  Get and decode a command from the input stream.

*/

static struct CommandItem
{ int           key;
  Command       code;
} ACommands[] =
  { {'A', ArcAbsolute },
    {'R', ArcRelative },
    { 0, 0 }
  },

  CCommands[] =
  { {'A', SetAlternateChar },
    {'I', Circle },
    {'P', CharacterPlot },
    {'S', SetStandardChar },
    { 0, 0 }
  },

  DCommands[] =
  { {'C', NotImplemented },             /* Digitize Clear */
    {'F', SetDefaults },
    {'I', SetAbsoluteDirection },
    {'P', NotImplemented },             /* Digitize Point */
    {'R', SetRelativeDirection },
    {'T', SetLabelTerminator },
    { 0, 0 }
  },

  ECommands[] =
  { {'A', RectangleAbsolute },
    {'R', RectangleRelative },
    {'W', Wedge },
    { 0, 0 }
  },

  FCommands[] =
  { {'T', SetFillType },
    { 0, 0 }
  },

  ICommands[] =
  { {'M', NotImplemented },             /* Input Mask */
    {'N', Initialize },
    {'P', InputP1P2 },
    {'W', InputWindow },
    { 0, 0 }
  },

  LCommands[] =
  { {'B', PutASCIILabel },
    {'T', SetLineType },
    { 0, 0 }
  },

  OCommands[] =
  { {'A', Interactive },                /* Output Position  */
    {'C', Interactive },                /* Output Commanded */
    {'D', Interactive },                /* Output Point     */
    {'E', Interactive },                /* Output Error     */
    {'F', Interactive },                /* Output Factors   */
    {'H', Interactive },                /* Output Hard Clip */
    {'I', Interactive },                /* Output Ident     */
    {'O', Interactive },                /* Output Options   */
    {'P', Interactive },                /* Output P1 and P2 */
    {'S', Interactive },                /* Output Status    */
    {'W', Interactive },                /* Output Window    */
    { 0, 0 }
  },

  PCommands[] =
  { {'A', SetAbsolutePlot },
    {'D', PenDown },
    {'R', SetRelativePlot },
    {'S', SetPaperSize },
    {'T', PenThickness },             /* Pen Thickness */
    {'U', PenUp },
    { 0, 0 }
  },

  RCommands[] =
  { {'A', ShadeRectAbsolute },
    {'O', RotateCoordSys },
    {'R', ShadeRectRelative },
    { 0, 0 }
  },

  SCommands[] =
  { {'A', SelectAlternate },
    {'C', SetScale },
    {'I', SetAbsoluteCharSize },
    {'L', SetAbsoluteCharSlant },
    {'M', SetSymbolMode },
    {'P', SelectPen },
    {'R', SetRelativeCharSize },
    {'S', SelectStandard },
    { 0, 0 }
  },

  TCommands[] =
  { {'L', SetTickLength },
    { 0, 0 }
  },

  UCommands[] =
  { {'C', UserChar },
    { 0, 0 }
  },

  VCommands[] =
  { {'S', NotImplemented },             /* Set Velocity */
    { 0, 0 }
  },

  WCommands[] =
  { {'G', ShadeWedge },
    { 0, 0 }
  },

  XCommands[] =
  { {'T', XTick },
    { 0, 0 }
  },

  YCommands[] =
  { {'T', YTick },
    { 0, 0 }
  },

  *CommandIndex[] =
    {   ACommands,
        NULL, 
        CCommands, 
        DCommands, 
        ECommands,
        FCommands, 
        NULL, 
        NULL, 
        ICommands, 
        NULL, 
        NULL, 
        LCommands,
        NULL, 
        NULL, 
        OCommands, 
        PCommands, 
        NULL, 
        RCommands,
        SCommands, 
        TCommands, 
        UCommands, 
        VCommands,
        WCommands, 
        XCommands, 
        YCommands, 
        NULL
    };



Command getCommand()

{ int                   c;
  struct CommandItem    *tablePointer;

  c = getChar();
  if (isalpha(c) &&
     ((tablePointer = CommandIndex[islower(c) ? (c - 'a') : (c - 'A')]) != NULL))
  { skipSeparator();
    c = getChar();
    if (isalpha(c))
    { if (islower(c)) c = toupper(c);
      while (tablePointer->key)
        if (tablePointer->key == c)
        { skipSeparator();
          return(tablePointer->code);
        }
        else
          tablePointer++;
    }
  }
  error("Illegal HPGL command.");
}



/*

  Handle extra parameters to a command.

*/

void endCommand()

{ Number dummy;

  if (!isTerminator(LookAhead))
  { warning("Extra parameters ignored");
    while (!isTerminator(LookAhead))
      (void)getChar();
  }
  skipTerminator();
}



/*************************************************************************

  Internal I/O redirection

*************************************************************************/


/*

  Get input chars from specified file.

*/

Boolean setInput(fileName)

char *fileName;

{ return((input = fopen(fileName,"r")) != NULL); }



/*

  Put output chars to specified file.

*/

Boolean setOutput(fileName)

char *fileName;

{ return((output = fopen(fileName,"w")) != NULL); }

