/*
        HPGL to PostScript converter
   Copyright (C) 1988 (and following) Federico Heinz

yahp2ps is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY.  No author or distributor accepts responsibility to anyone
for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.
Refer to the Free Software Foundation's General Public License for full details.

Everyone is granted permission to copy, modify and redistribute yahp2ps,
but only under the conditions described in the GNU General Public
License.  A copy of this license is supposed to have been given to you
along with yahp2ps so you can know your rights and responsibilities.  It
should be in a file named COPYING.  Among other things, the copyright
notice and this notice must be preserved on all copies.

In other words, go ahead and share yahp2ps, but don't try to stop
anyone else from sharing it farther.  Help stamp out software hoarding!

yahp2ps is TOTALLY unrelated to GNU or the Free Software Foundation,
it is only released under the same conditions.

    For bug reports, wishes, etc. send e-mail to

    ...!mcvax!unido!tub!actisb!federico  (from Europe)
    ...!uunet!pyramid!actisb!federico    (from anywhere else)

    For Physical mail:

    Federico Heinz
    Beusselstr. 21
    1000 Berlin 21

    Tel. (+49 30) 396 77 92

*/
/**************************************************************************

   Arithmetical support for circle/arc stuff

**************************************************************************/

#include <ctype.h>
#include "defs.h"
#include "circle.h"
#include "io.h"


/**************************************************************************

  Trigonometry stuff, where I can give free fly to my floating-point
  hate and my love for letting the compiler do the job.

**************************************************************************/



/* Table to convert the normalized angle returned by octantAngle to
   it's "equivalent" in the specified octant */


typedef struct
  { int     sign;
    Number  base;
    Boolean swappedFncts;
    int     cosineFactor;
    int     sineFactor;
  } Octant;



static Octant OctantTable[]=
  { { 1, Zero,         False,  1,  1},
    {-1, OneSquare,    True,   1,  1},
    { 1, OneSquare,    True,  -1,  1},
    {-1, TwoSquares,   False, -1,  1},
    { 1, TwoSquares,   False, -1, -1},
    {-1, ThreeSquares, True,  -1, -1},
    { 1, ThreeSquares, True,   1, -1},
    {-1, FullCircle,   False,  1, -1}
  };


/*

  Map an angle to the first octant and return a pointer to the octant
  conversion structure that enables to do inter-octant mapping. The
  mapped angle is in INTEGER degrees, not Number.

*/

static Octant *classifyAngle(angle)

Number *angle;

{ Number result;
  Octant *octant;

  *angle = *angle % FullCircle;
  if (*angle < Zero)
    *angle = FullCircle + *angle;
  octant = OctantTable + *angle / (OneSquare / 2);
  result = (*angle % OneSquare) / One;
  if (result > ((OneSquare / One) / 2))
    *angle = (OneSquare / One) - result;
  else
    *angle = result;
  return(octant);
}



/*

  Normalize the cathetes such that:
                                    they're both positive
                                    the X cathete is longer than the Y.
  Return also a pointer to the octant conversion structure that enables
  to map an angle from the 1st octant into the cathetes' octant.

*/

static Octant *classifyCathetes(target, cathetes)

CoordinatePair target, cathetes;

{ Number dummy;
  int    octant;

  if (cathetes[Y] >= Zero)
  { target[Y] = cathetes[Y];
    octant = 0;
  }
  else
  { target[Y] = -cathetes[Y];
    octant = 4;
  }
  if (cathetes[X] >= Zero)
  { target[X] = cathetes[X];
    if (octant) octant |= 3;
  }
  else
  { target[X] = -cathetes[X];
    if (!octant) octant |= 3;
  }
  if (target[X] < target[Y])
  { octant ^= 1;
    dummy = target[X];
    target[X] = target[Y];
    target[Y] = dummy;
  }
  return(OctantTable + octant);
}



/*

  These tables contain the values for the cosine, sine and tangent of
  all integer angles between 0 and 45 degrees (or at least so says my
  math library).

*/

static Number CosineTable[] = {
  10000L,  9998L,  9993L,  9986L,  9975L,  9961L,  9945L,  9925L,
   9902L,  9876L,  9848L,  9816L,  9781L,  9743L,  9702L,  9659L,
   9612L,  9563L,  9510L,  9455L,  9396L,  9335L,  9271L,  9205L,
   9135L,  9063L,  8987L,  8910L,  8829L,  8746L,  8660L,  8571L,
   8480L,  8386L,  8290L,  8191L,  8090L,  7986L,  7880L,  7771L,
   7660L,  7547L,  7431L,  7313L,  7193L,  7071L };

static Number SineTable[] = {
      0L,   174L,   348L,   523L,   697L,   871L,  1045L,  1218L,
   1391L,  1564L,  1736L,  1908L,  2079L,  2249L,  2419L,  2588L,
   2756L,  2923L,  3090L,  3255L,  3420L,  3583L,  3746L,  3907L,
   4067L,  4226L,  4383L,  4539L,  4694L,  4848L,  4999L,  5150L,
   5299L,  5446L,  5591L,  5735L,  5877L,  6018L,  6156L,  6293L,
   6427L,  6560L,  6691L,  6819L,  6946L,  7071L };

static int TangentTable[] = {
      0,   174,   349,   524,   699,   874,  1051,  1227,
   1405,  1583,  1763,  1943,  2125,  2308,  2493,  2679,
   2867,  3057,  3249,  3443,  3639,  3838,  4040,  4244,
   4452,  4663,  4877,  5095,  5317,  5543,  5773,  6008,
   6248,  6494,  6745,  7002,  7265,  7535,  7812,  8097,
   8390,  8692,  9004,  9325,  9656, 10000 };


/*

  Search the angle corresponding to a certain value in the table.

*/

static int search(value, table)

int value, table[];

{ int lo, hi, mid;

  lo = 0; hi = 45;
  while (lo <= hi)
  { mid = (lo + hi) / 2;
    if (value < table[mid])
      hi = mid - 1;
    else if (value > table[mid])
      lo = mid + 1;
    else
      return(mid);
  }
  return((value-table[hi] < table[lo]-value) ? hi : lo);
}



/*

  Compute the only 1st-octant angle that can generate these cathetes.

*/

static Number octantAngle(cathetes)

CoordinatePair cathetes;

{ int tangent;

  tangent = (int)divNum(cathetes[Y], cathetes[X]);
  return(search(tangent, TangentTable) * One);
}



/*

  Compute the only angle that can generate these cathetes.

*/

static Number arcTangent(cathetes)

CoordinatePair cathetes;

{ Octant *octant;
  CoordinatePair normalizedCathetes;

  octant = classifyCathetes(normalizedCathetes, cathetes);
  return(octantAngle(normalizedCathetes) * octant->sign + octant->base);
}



/*

  Compute the cosine of an angle.

*/

static Number cosine(angle)

Number angle;

{ Octant *octant;

  octant = classifyAngle(&angle);
  return((octant->swappedFncts ? SineTable[angle] : CosineTable[angle])
          * octant->cosineFactor);
}

   

/*

  Compute the sine of an angle.

*/

static Number sine(angle)

Number angle;

{ Octant *octant;

  octant = classifyAngle(&angle);
  return((octant->swappedFncts ? CosineTable[angle] : SineTable[angle])
          * octant->sineFactor);
}

   

/*************************************************************************

   Coordinate computing stuff

*************************************************************************/


/*

  Set target to the coordinates of the point that is located at
  polar coordinates (radius, angle) from center.

*/

void polarToCartesian(target, center, radius, angle)

CoordinatePair target, center;
Number radius, angle;

{ target[X] = center[X] + mulNum(cosine(angle), radius);
  target[Y] = center[Y] + mulNum(sine(angle), radius);
}




/*

   Compute radius and angle given an offset to the center.

*/

void cartesianToPolar(radius, angle, offset)

Number *radius, *angle;
CoordinatePair offset;

{ Number cosineOfAngle;

  *angle = arcTangent(offset);
  if (cosineOfAngle = cosine(*angle))	/* Watch out for cos(a) == 0!!! */
    *radius = divNum(offset[X], cosineOfAngle);
  else
    *radius = offset[X] ? offset[X] : offset[Y];
  if (*radius < 0) *radius = -*radius;
}



/*

  Return an angle suitable for a curve operation. It must divide the
  sweep angle, and must be between the minimum and maximum values specified.
  In case the sweep angle is a prime number, then compromise must be
  done. I then take the maxValue angle and make the last step
  shorter. This is not all compatible with the HP plotters, but then
  they enter an ENDLESS LOOP when this constellation occurs (at least
  mine does)
  
*/


Number correctedChordAngle(maxValue, sweepAngle, minValue)

Number maxValue, sweepAngle, minValue;

{ Number i;
  
  if (maxValue < 0)     /* Ignore sign */
    maxValue = -maxValue;
  maxValue = maxValue % FullCircle;
  if (maxValue < minValue)
    return(minValue);
  if (!(maxValue && (sweepAngle % maxValue)))
    return(maxValue);	/* High-resolution arc or exact division */
  i = maxValue - One;
  while(sweepAngle % i)
  { i -= One;
    if (i < minValue) return(maxValue);
  }
  return(i);
}



/*

  Twist the start and the sweep angles so as to make sense.

*/

void fixUpStartAndSweep(startAngle, sweepAngle)

Number *startAngle, *sweepAngle;

{ *startAngle = *startAngle % FullCircle;
  if ((*sweepAngle > FullCircle) || (*sweepAngle < -FullCircle))
    *sweepAngle = FullCircle;
}



/*

  Get the optional chord angle parameter.

*/

void getChordAngle(target, sweepAngle, minValue)

Number *target;
Number sweepAngle, minValue;

{ if (isNumeric(LookAhead))
    (void)getInteger(target);
  else
    *target = DefaultChordAngle;
  *target = correctedChordAngle(*target, sweepAngle, minValue);
}

