/*
 *	Style selection default table and related functions.
 *	See "cif2ps.c" for authors' names and addresses.
 *	Please honor the authors by not removing their attributions.
 */

#include "define.h"
#include <string.h>
#include <ctype.h>

/*
 *	Style table encoding:
 *	The strings are pairs: LAYER, STYLE and
 *	STYLE strings are merely PostScript except for the "G.n"
 *	code which is special (and must be first in the string).
 *	A spaces is required between components.
 *
 *	Drawing styles implemented:
 *
 *	"Gn"	fill with opaque gray stipple with level n where
 *		n is a real number immediately after the 'G'.
 *		The G must be the beginning of the string!
 *
 *	"n a L"	fill with lines spaced by n and rotated by a (degrees).
 *
 *	"X"	Put an X across the figure.
 *
 *	"stroke" Draw a box around the figure.
 *			(this must be last if used).
 */

char *default_styles[] = {	/* pairs of strings */

	/* MOSIS SCMOS layers */
	"CPG", "G.3",
	"CCP", "G0",
	"CAA", "G.5",
	"CCA", "G0",
	"CMF", "8 45 L",
	"CMS", "8 135 L",
	"CVA", "G.2",
	"CSP", "16 45 L",
	"CSN", "16 135 L",
	"CWP", "32 45 L",
	"CWN", "32 135 L",
	"COG", "G.7",

	/* Oct Symbolic layers */
	"POLY", "G.3",
	"COPO", "G.3 X stroke",		/* poly contact */
	"PDIF", "G.7 24 45 L",
	"COPD", "G.7 X stroke",		/* pdiff contact */
	"NDIF", "G.7 24 135 L",
	"COND", "G.7 X stroke",		/* ndiff contact */
	"MET1", "8 45 L",
	"COM2", "8 45 L 8 135 L stroke",
	"MET2", "8 135 L",
	"PWEL", "32 45 L",
	"COPS", "G.9 32 45 L X stroke",		/* psubs contact */
	"NWEL", "32 135 L",
	"CONS", "G.9 32 135 L X stroke",	/* nsubs contact */
	"GLAS", "G.7 X stroke",

	/* MOSIS Bulk CMOS layers */
	"CP", "G.3",
	"CD", "G.5",
	"CM", "8 45 L",
	"CQ", "8 135 L",	/* metal 2 */
	"CM2","8 135 L",	/* metal 2, alternate name */
	"CC", "G0",
	"CV", "G.2 stroke",	/* via */
	"CC2","G.2 stroke",	/* via, alternate name */
	"CW", "23 45 L",
	"CS", "16 45 L",
	"CG", "G.7",

	/* end of table marker -- MAKE SURE THIS STAYS! */
	0, 0
};


char *style_lookup(name)	/* returns style string for layer */
char *name;
{
	char **p;

	for (p = default_styles; *p; p++) {
		if (!strcmp(*(p++),name)) return(*p);
	}
	fprintf(stderr,"%s: layer style not found (ignoring)\n", name);
	return("");
}

style_sort()	/* determines drawing order for layers */
{
	int i,j,tmp;
	float grays[MAXLAYERS];

	/* the lighter (higher value) gray levels are drawn first,
	 * so sort into decreasing order.
	 */

	/* Read the gray levels and initialize order[] */
	/* Also, make the ones starting with "G" come first */
	i = 0; j = numlayers;
	while (--j >= 0) {
		if (!layers[j].style)
			layers[j].style = style_lookup(layers[j].name);
		if (1 == sscanf(layers[j].style,
				"G%f",&(grays[j]))) {
			order[i++] = j;
		} else {
			order[j+i] = j;
		}
	}
	/* Now only the first (i) elements need sorting */

	/* Sort the array into decreasing gray levels.
	 * An exchange sort is used on indexes in order[].
	 * (The number of elements is always small.)
	 */
	while (--i >= 0) {
		for (j=0; j<i; j++) {
			if (grays[order[j]] >= grays[order[i]])
				continue;
			/* swap (put in decreasing order) */
			tmp = order[i];
			order[i] = order[j];
			order[j] = tmp;
		}
	}
}

int style_gray(s)	/* returns true if style uses gray fill */
	char *s;
{
	if (*s != 'G') return(0);
	++s;
	while ( isdigit(*s) || (*s=='.') ) putchar(*s++);
	puts(" setgray");
	cur_style = "fill";
	return(1);
}


int style_rest(s)	/* the rest can be done in any order */
	char *s;
{
	if (*s == 'G') {	/* skip a gray fill code (already done)  */
		++s;
		while ( isdigit(*s) || *s=='.' ) s++;
		while ( isspace(*s) ) s++;
	}
	if (*s == '\0') return(0);
	cur_style = s;
	return(1);
}


set_style(s)		/* set style, i.e. "CSP=stroke" */
char *s;
{
	int i;
	char *p;

	p = s;
	while (isalnum(*p)) p++;	/* find non-alnum */
	if (*p) *(p++) = '\0';			/* zap it */
	i = layer_lookup(s);
	layers[i].style = p;
}
