/* UNFSD - copyright Mark A Shand, May 1988.
 * This software maybe be used for any purpose provided
 * the above copyright notice is retained.  It is supplied
 * as is, with no warranty expressed or implied.
 */

#include "unfsd.h"

static struct timeval TIMEOUT = { 25, 0 };

/* ====================================================================== */
#ifdef DEBUG
FILE *debuglog = NULL;
static char *pname = "unfsd";
static char argbuf[1024];

logcall(name, arg, rqstp)
char	*name;
char	*arg;
struct svc_req	*rqstp;
{
	int	i;

	if (debuglog == NULL)
	{
		unsigned long tloc;
		if ((debuglog = fopen("/tmp/unfsd.log", "w")) == NULL)
			return;
		setlinebuf(debuglog);
		time(&tloc);
		fprintf(debuglog, "\n\nstarting %s at %s\n", pname, ctime(&tloc));
	}
	fprintf(debuglog, "%s [%d ", name, rqstp->rq_cred.oa_flavor);
	if (rqstp->rq_cred.oa_flavor == AUTH_UNIX)
	{
		struct authunix_parms *unix_cred;
		struct tm *tm;
		unix_cred = (struct authunix_parms *) rqstp->rq_clntcred;
		tm = localtime(&unix_cred->aup_time);
		fprintf(debuglog, "%d/%d/%d %02d:%02d:%02d %s %d.%d",
			tm->tm_year, tm->tm_mon+1, tm->tm_mday,
			tm->tm_hour, tm->tm_min, tm->tm_sec,
			unix_cred->aup_machname,
			unix_cred->aup_uid,
			unix_cred->aup_gid);
		if (unix_cred->aup_len > 0)
		{
			fprintf(debuglog, "+%d", unix_cred->aup_gids[0]);
			for (i = 1; i < unix_cred->aup_len; i++)
				fprintf(debuglog, ",%d",unix_cred->aup_gids[i]);
		}
	}
	fprintf(debuglog, "]\n\t%s\n", arg);
}

#else
#define logcall(name, arg, client)
#define fh_pr(x)	""
#endif DEBUG
/* ====================================================================== */

extern int errno;
#ifdef DEBUG
extern char *sys_errlist[];
#endif DEBUG

extern char	*malloc();



/* ====================================================================== */

void *
nfsproc_null_2(argp, rqstp)
	void *argp;
	struct svc_req *rqstp;
{
	static char res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_null", "", rqstp);
	return ((void *)&res);
}


static void
inner_getattr(fh, status, attr, stat_optimize, cp, rqstp)
	nfs_fh		*fh;
	nfsstat		*status;
	fattr		*attr;
	struct stat	*stat_optimize;
	clnt_param	*cp;
	struct svc_req	*rqstp;
{
	char	*path;
	struct stat *s;
	struct stat sbuf;

#ifdef DEBUG
	if (debuglog)
		fprintf(debuglog, " inner_getattr");
#endif
	if ((path = fh_path(fh, status)) != 0)
	{
		if (stat_optimize != NULL)
			s = stat_optimize;
		else
		{
			s = &sbuf;
			if (lstat(path, s) < 0)
				*status = (nfsstat) errno;
		}
			
		attr->type = ft_map[ft_extr(s->st_mode)];
		attr->mode = s->st_mode;
		attr->nlink = s->st_nlink;
		attr->uid = ruid(s->st_uid, cp, rqstp->rq_xprt);
		attr->gid = rgid(s->st_gid, cp, rqstp->rq_xprt);
		attr->size = s->st_size;
		attr->blocksize = s->st_blksize;
		attr->rdev = s->st_rdev;
		attr->blocks = s->st_blocks;
		attr->fsid = 1;
		attr->fileid = fh_psi(fh);
		attr->atime.seconds = s->st_atime;
		attr->mtime.seconds = s->st_mtime;
		attr->ctime.seconds = s->st_ctime;
#ifdef DEBUG
		if (debuglog)
		{
			if (*status == NFS_OK)
			{
				fprintf(debuglog, " t=%d, m=%o, lk=%d, u/g=%d/%d, sz=%d, bsz=%d",
					attr->type, attr->mode, attr->nlink,
					attr->uid, attr->gid, attr->size,
					attr->blocksize);
				if (attr->type == NFCHR || attr->type == NFBLK)
					fprintf(debuglog, " rdev=%d/%d",
						(attr->rdev>>8)&0xff, attr->rdev&0xff);
				fprintf(debuglog, "\n  blks=%d, fsid=%d, psi=%d, at=%d, mt=%d, ct=%d\n",
					attr->blocks, attr->fsid, attr->fileid,
					attr->atime.seconds,
					attr->mtime.seconds,
					attr->ctime.seconds);
			}
			else
				fprintf(debuglog, " >>> %s\n", sys_errlist[(int) *status]);
		}
#endif
	}
#ifdef DEBUG
	else if (debuglog)
		fprintf(debuglog, " failed!\n");
#endif
}


attrstat *
nfsproc_getattr_2(argp, rqstp)
	nfs_fh *argp;
	struct svc_req *rqstp;
{
	static attrstat res;
	clnt_param	*cp;

	bzero(&res, sizeof(res));
	logcall("nfsproc_getattr", fh_pr(argp), rqstp);
	if ((cp = knownclient(rqstp)) == NULL)
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
	inner_getattr(argp, &(res.status),
		&(res.attrstat_u.attributes), NULL, cp, rqstp);
	return (&res);
}


attrstat *
nfsproc_setattr_2(argp, rqstp)
	sattrargs *argp;
	struct svc_req *rqstp;
{
	static attrstat res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_setattr", "", rqstp);
	if (knownclient(rqstp) == NULL)
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
#ifdef READ_ONLY
	res.status = NFSERR_ROFS;
#else
	not implemented
#endif
	return (&res);
}


void *
nfsproc_root_2(argp, rqstp)
	void *argp;
	struct svc_req *rqstp;
{
	static char res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_root", "", rqstp);
	return ((void *)&res);
}


diropres *
nfsproc_lookup_2(argp, rqstp)
	diropargs *argp;
	struct svc_req *rqstp;
{
	static diropres res;
	clnt_param	*cp;
	struct stat	sbuf;
	struct stat	*sbp = &sbuf;

	bzero(&res, sizeof(res));
	logcall("nfsproc_lookup", sprintf(argbuf, "fh=%s n=%s", fh_pr(&(argp->dir)), argp->name), rqstp);
	if ((cp = knownclient(rqstp)) == NULL)
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
	res.status = fh_compose(argp, &(res.diropres_u.diropres.file), &sbp);
	if (res.status == NFS_OK)
	{
		inner_getattr(&(res.diropres_u.diropres.file), &(res.status),
			&(res.diropres_u.diropres.attributes), sbp, cp,rqstp);
#ifdef DEBUG
		if (debuglog && res.status == NFS_OK)
			fprintf(debuglog, "\tnew_fh = %s\n", fh_pr(&(res.diropres_u.diropres.file)));
#endif /* DEBUG */
	}
	return (&res);
}


readlinkres *
nfsproc_readlink_2(argp, rqstp)
	nfs_fh *argp;
	struct svc_req *rqstp;
{
	static readlinkres res;
	clnt_param	*cp;
	char	*path;

	bzero(&res, sizeof(res));
	logcall("nfsproc_readlink", fh_pr(argp), rqstp);
	if ((cp = knownclient(rqstp)) == NULL)
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
	if ((path = fh_path(argp, &(res.status))) != 0)
	{
		int	cc;
		static char	linkbuf[NFS_MAXPATHLEN];

		errno = 0;
		if ((cc = readlink(path, linkbuf, NFS_MAXPATHLEN)) < 0)
			res.status = (nfsstat) errno;
		else
		{
			res.status = NFS_OK;
			linkbuf[cc] = '\0';
			res.readlinkres_u.data = linkbuf;
			if (cp->o.link_relative && linkbuf[0] == '/')
			{
				/* prepend ../ sequence.  Note: relies that
				 * fh_path returns a path containing real
				 * directories.
				 */
				int	slash_cnt = 0;
				char	*p, *q;
				for (p = index(path+1, '/'); p != NULL; p = index(p+1, '/'))
					slash_cnt++;
				p = linkbuf + strlen(linkbuf);
				q = p + 3 * slash_cnt - 1;
				if (q >= linkbuf + NFS_MAXPATHLEN)
					res.status = NFSERR_NAMETOOLONG;
				else
				{
					while (p >= linkbuf)
						*q-- = *p--;
					p = linkbuf;
					while (slash_cnt-- > 0)
					{
						*p++ = '.';	
						*p++ = '.';	
						*p++ = '/';	
					}
				}
			}
		}
	}
#ifdef DEBUG
	if (debuglog)
	{
		if (res.status != NFS_OK)
			fprintf(debuglog, " >>> %s\n", sys_errlist[(int) res.status]);
		else
			fprintf(debuglog, " %s\n", res.readlinkres_u.data);
	}
#endif /* DEBUG */
	return (&res);
}


static char iobuf[NFS_MAXDATA];

readres *
nfsproc_read_2(argp, rqstp)
	readargs *argp;
	struct svc_req *rqstp;
{
	static readres res;
	clnt_param	*cp;
	int	fd;

	bzero(&res, sizeof(res));
	logcall("nfsproc_read", sprintf(argbuf, "%s @%d for %d", fh_pr(&(argp->file)), argp->offset, argp->count), rqstp);
	if ((cp = knownclient(rqstp)) == NULL)
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
	if ((fd = fh_fd(&(argp->file), &(res.status), O_RDONLY)) >= 0)
	{
		errno = 0;
		lseek(fd, (long) argp->offset, L_SET);
		res.readres_u.reply.data.data_val = iobuf;
		if (!errno)
			res.readres_u.reply.data.data_len =
				read(fd, res.readres_u.reply.data.data_val, argp->count);
		fd_idle(fd);
		res.status = (nfsstat) errno;
		if (!errno)
			inner_getattr(&(argp->file), &(res.status),
				&(res.readres_u.reply.attributes), NULL, cp, rqstp);
	}
	return (&res);
}


void *
nfsproc_writecache_2(argp, rqstp)
	void *argp;
	struct svc_req *rqstp;
{
	static char res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_writecache", "", rqstp);
	return ((void *)&res);
}


attrstat *
nfsproc_write_2(argp, rqstp)
	writeargs *argp;
	struct svc_req *rqstp;
{
	static attrstat res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_write", sprintf(argbuf, "%s @%d for %d", fh_pr(&(argp->file)), argp->offset, argp->data.data_len), rqstp);
	if (knownclient(rqstp) == NULL)
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
#ifdef READ_ONLY
	res.status = NFSERR_ROFS;
#else /* READ_ONLY */
	not implemented
#endif /* READ_ONLY */
	return (&res);
}


diropres *
nfsproc_create_2(argp, rqstp)
	createargs *argp;
	struct svc_req *rqstp;
{
	static diropres res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_create", sprintf(argbuf, "fh=%s n=%s m=%0o u/g=%d/%d sz=%d", fh_pr(&(argp->where.dir)), argp->where.name, argp->attributes.mode, argp->attributes.uid, argp->attributes.gid, argp->attributes.size), rqstp);
	if (knownclient(rqstp) == NULL)
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
#ifdef READ_ONLY
	res.status = NFSERR_ROFS;
#else /* READ_ONLY */
	not implemented
#endif /* READ_ONLY */
	return (&res);
}


nfsstat *
nfsproc_remove_2(argp, rqstp)
	diropargs *argp;
	struct svc_req *rqstp;
{
	static nfsstat res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_remove", "", rqstp);
	if (!knownclient(rqstp))
	{
		res = NFSERR_ACCES;
		return (&res);
	}
	res = NFSERR_ROFS;
	return (&res);
}


nfsstat *
nfsproc_rename_2(argp, rqstp)
	renameargs *argp;
	struct svc_req *rqstp;
{
	static nfsstat res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_rename", "", rqstp);
	if (!knownclient(rqstp))
	{
		res = NFSERR_ACCES;
		return (&res);
	}
	res = NFSERR_ROFS;
	return (&res);
}


nfsstat *
nfsproc_link_2(argp, rqstp)
	linkargs *argp;
	struct svc_req *rqstp;
{
	static nfsstat res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_link", "", rqstp);
	if (!knownclient(rqstp))
	{
		res = NFSERR_ACCES;
		return (&res);
	}
	res = NFSERR_ROFS;
	return (&res);
}


nfsstat *
nfsproc_symlink_2(argp, rqstp)
	symlinkargs *argp;
	struct svc_req *rqstp;
{
	static nfsstat res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_symlink", "", rqstp);
	if (!knownclient(rqstp))
	{
		res = NFSERR_ACCES;
		return (&res);
	}
	res = NFSERR_ROFS;
	return (&res);
}


diropres *
nfsproc_mkdir_2(argp, rqstp)
	createargs *argp;
	struct svc_req *rqstp;
{
	static diropres res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_mkdir", "", rqstp);
	if (!knownclient(rqstp))
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
	res.status = NFSERR_ROFS;
	return (&res);
}


nfsstat *
nfsproc_rmdir_2(argp, rqstp)
	diropargs *argp;
	struct svc_req *rqstp;
{
	static nfsstat res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_rmdir", "", rqstp);
	if (!knownclient(rqstp))
	{
		res = NFSERR_ACCES;
		return (&res);
	}
	res = NFSERR_ROFS;
	return (&res);
}

static int
dpsize(dp)
struct direct *dp;
{
#define DP_SLOP	16
#define MAX_E_SIZE sizeof(entry) + MAXNAMLEN + DP_SLOP
	return sizeof(entry) + strlen(dp->d_name) + DP_SLOP;
}

readdirres *
nfsproc_readdir_2(argp, rqstp)
	readdirargs *argp;
	struct svc_req *rqstp;
{
	static readdirres res;
	entry **e;
	char	*path;

	/*
	 * Free previous result
	 */
	xdr_free(xdr_readdirres, &res);

	bzero(&res, sizeof(res));
	logcall("nfsproc_readdir", fh_pr(&(argp->dir)), rqstp);
	if (!knownclient(rqstp))
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
	if ((path = fh_path(argp, &(res.status))) != 0)
	{
		long	dloc;
		DIR	*dirp;
		struct direct *dp;
		struct	stat	sbuf;

		errno = 0;
		stat(path, &sbuf);
		if ((dirp = opendir(path)) == NULL)
		{
			if (errno != 0)
				res.status = (nfsstat) errno;
			else
				res.status = NFSERR_NAMETOOLONG;
		}
		else
		{
			int	res_size = 0;

			res.status = NFS_OK;
			bcopy(argp->cookie, &dloc, sizeof(dloc));
			if (dloc != 0)
				seekdir(dirp, dloc);
			e = &(res.readdirres_u.reply.entries);
			while (((res_size + MAX_E_SIZE) < argp->count
					 || e == &(res.readdirres_u.reply.entries))
				 && (dp = readdir(dirp)) != NULL)
			{
				if ((*e = (entry *) malloc(sizeof(entry))) == NULL)
					mallocfailed();
				(*e)->fileid = pseudo_inode(dp->d_ino, sbuf.st_dev);
				if (((*e)->name = malloc(strlen(dp->d_name)+1)) == NULL)
					mallocfailed();
				strcpy((*e)->name, dp->d_name);
				dloc = telldir(dirp);
				bcopy(&dloc, ((*e)->cookie), sizeof(nfscookie));
				e = &((*e)->nextentry);
				res_size += dpsize(dp);
			}
			*e = NULL;
			res.readdirres_u.reply.eof = (dp == NULL);
			closedir(dirp);
		}
	}
	return (&res);
}


statfsres *
nfsproc_statfs_2(argp, rqstp)
	nfs_fh *argp;
	struct svc_req *rqstp;
{
	static statfsres res;

	bzero(&res, sizeof(res));
	logcall("nfsproc_statfs", fh_pr(argp), rqstp);
	if (!knownclient(rqstp))
	{
		res.status = NFSERR_ACCES;
		return (&res);
	}
	/* no easy way to do this */
	res.status = NFS_OK;
	res.statfsres_u.reply.tsize = 4096;
	res.statfsres_u.reply.bsize = 4096;
	res.statfsres_u.reply.blocks = 100000;
	res.statfsres_u.reply.bfree = 80000;
	res.statfsres_u.reply.bavail = 71000;
	return (&res);
}
