%{
/*
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Lawrence Berkeley Laboratory,
 * Berkeley, CA.  The name of the University may not be used to
 * endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#ifndef lint
static char rcsid[] =
    "@(#) $Header: tcplex.l,v 1.7 90/02/24 23:11:10 mccanne Exp $ (LBL)";
#endif

#include <ctype.h>
#include "interface.h"
#include "tcpgram.h"
#include "tokdefs.h"
#include "pstr.h"

static void nettotemplate ();
static void lex_err ();
static void lookup_id ();

#ifdef FLEX_SCANNER
static int argv_stream ();
#undef YY_INPUT
#define YY_INPUT(buf, result, max)   (result = (argv_stream (buf, max)))
#else
#undef getc
#define getc(fp)  (*yyargv ? (**yyargv ? (*(*yyargv)++)\
			      : (++yyargv, ' ')) : EOF)
#endif

YYSTYPE yylval;
static char **yyargv;

/*
 * Some state is being maintained that allows the scanner to 
 * remember the last keyword sequence seen.  This allows identifiers 
 * to be looked up in the right place.
 *
 * The lexer state is initialized to HOST whenever a new keyword
 * sequence is seen.  It should only be done once for each sequence,
 * and this is determined by counting the intervening identifiers.
 */
#define RESET { if (id_count != 0)\
		{ id_count = 0; lexer_addr = HOST; lexer_qual = HOST; } }

#define RETURN_ID { id_count += 1; return ID; }

static int id_count;
static int lexer_addr = HOST;
static int lexer_qual = HOST;

%}

N		([0-9]+|(0X|0x)[0-9A-Fa-f]+)
B		([0-9A-Fa-f][0-9A-Fa-f]?)

%a 3000

%%
dst			{ RESET; return (DST); }
src			{ RESET; return (SRC); }

ether			{ RESET; lexer_qual = ETHER; return (ETHER); }
arp			{ RESET; lexer_qual = ARP; return (ARP); }
rarp			{ RESET; lexer_qual = RARP; return (RARP); }
ip			{ RESET; lexer_qual = IP; return (IP); }
tcp			{ RESET; lexer_qual = TCP; return (TCP); }
udp			{ RESET; lexer_qual = UDP; return (UDP); }

host			{ RESET; lexer_addr = HOST; return (HOST); }
net			{ RESET; lexer_addr = NET; return (NET); }
port			{ RESET; lexer_addr = PORT; return (PORT); }
proto			{ RESET; lexer_addr = PROTO; return (PROTO); }

between 		{ RESET; lexer_addr = HOST; return (BETWEEN); }
gateway			{ RESET; lexer_addr = GATEWAY; return (GATEWAY); }

less			return LESS;
greater			return GREATER;
byte			return BYTE;
broadcast		return BROADCAST;

and			return AND;
or			return OR;
not			return '!';

[ \n\t]			;
[+\-*/:\]!<>()&|=]	return yytext[0];
\[			{ /* undo pname */ lexer_qual = HOST; return '['; }
">="			return GEQ;
"<="			return LEQ;
"!="			return NEQ;

\-?{N}			{
				switch (lexer_addr) {

				default: 
					yylval.i = strtol (yytext,
							   (char **) NULL, 0);
					return (NUM);

				case NET:
					nettotemplate (inet_network (yytext),
						       &yylval.id.n);

					RETURN_ID;

				case PORT:
					yylval.id.pp.pr = -1;
					yylval.id.pp.pt = 
						strtol (yytext, (char **) NULL,
							0);
					RETURN_ID;
				case PROTO:
					yylval.id.pr = strtol (yytext,
						       (char **) NULL, 0);
					RETURN_ID;
				}
			}

{N}\.{N}		{ 
				if (lexer_addr != NET)
					lex_err (yytext);
	                        nettotemplate (inet_network (yytext),
					       &yylval.id.n);
				RETURN_ID;
			}

{N}\.{N}\.{N}		{ 
				if (lexer_addr != NET)
					lex_err (yytext);
				nettotemplate (inet_network (yytext),
					       &yylval.id.n);

				RETURN_ID;
			}

{N}\.{N}\.{N}\.{N}	{ 
	                        switch (lexer_addr) {

				case NET:
					nettotemplate (inet_network (yytext),
						       &yylval.id.n);
					RETURN_ID;

				case HOST:
					if (lexer_qual != HOST)
						lex_err (yytext);
					yylval.id.h = inet_addr (yytext);
					RETURN_ID;

				default:
					lex_err (yytext);
				}
			}

{B}:{B}:{B}:{B}:{B}:{B} {
			if (lexer_addr == HOST && 
			    lexer_qual == ETHER) {
				yylval.id.e = s_numstrtoeaddr (yytext);
				RETURN_ID;
			}
			else 
				lex_err (yytext);
			}

{B}:+({B}:+)+		{
			fprintf (stderr, 
				 "bogus ethernet address: %s\n",
				 yytext);
			exit (-1);
			}

[-_.A-Za-z0-9]+		{ lookup_id (yytext); RETURN_ID; }
"\\"[^ !()\n\t]+	{ lookup_id (yytext + 1); RETURN_ID; }

[^ \t\n\-_.A-Za-z0-9!<>()&|=]+    { fprintf (stderr, "bad token: %s\n",
					     yytext);
				    n_errors += 1; }
.			{ /* this shouldn't happen */ }
%%
void
lex_init (argv)
	char **argv;
{
	yyargv = argv;
}

static void
lex_err (id)
	char *id;
{
	char *s;

	switch (lexer_addr) {
		
	case HOST:
		s = lexer_qual == ETHER ? "ether" : "host";
		break;
		
	case GATEWAY:
		s = "gateway";
		break;
		
	case NET:
		s = "network";
		break;
		
	case PORT:
		s = "port";
		break;
		
	case PROTO:
		s = "protocol";
		break;
		
	default:
		usage ();
		exit (-1);
	}
	fprintf (stderr, "bad %s: %s\n", s, id);
	exit (-1); 
}

#ifdef FLEX_SCANNER
/*
 * Fill buffer BP with up to MAX characters from command line concatenating
 * arguments with spaces.  Return number of characters actually read.
 */
static int 
argv_stream (bp, max)
	register char *bp;
	register int max;
{
	register int t = max;

	if (!*yyargv)
		return YY_NULL;

	while (max--) {
		if (!(*bp++ = *(*yyargv)++)) {
			bp[-1] = ' ';
			if (!*++yyargv)
				break;
		}
	}
	
	return t - max - 1;
}
#else
int 
yywrap ()
/* so we don't need -ll */
{
	return 1;
}				
#endif

/*
 *  Left justify addr and compute its network mask.  Place results in *net.
 */
static void
nettotemplate (addr, net)
	unsigned long addr;
	register struct net_id *net;
{
	register unsigned int m = 0xffffffff;

	if (addr)
		while ((addr & 0xff000000) == 0)
			addr <<= 8, m <<= 8;

	net->m = m;
	net->n = addr;
}

static void
lookup_id (s)
	char *s;
{
	switch (lexer_addr) {

	default:
		punt ("unknown state mapping id");
		
	case HOST:
		if (lexer_qual == ETHER) {
			if (!(yylval.id.e = s_nametoeaddr (s))) 
				lex_err (s);
		}
		else if (!(yylval.id.h = s_nametoaddr (s)))
			lex_err (s);
		return;

	case GATEWAY:
		if (!(yylval.id.g.h = s_nametoaddr (s)))
			lex_err (s);
		if (!(yylval.id.g.e = s_nametoeaddr (s)))
			lex_err (s);
		return;
		
	case NET:
		if (!(yylval.id.n.n = s_nametonetaddr (s)))
			lex_err (s);
		nettotemplate (yylval.id.n.n, &yylval.id.n);
		return;
		
	case PORT:
		if (!s_nametoport (s, &yylval.id.pp))
			lex_err (s);
		return;
		
	case PROTO:
		switch (lexer_qual) {
		case ETHER:
			/* ether proto */
			yylval.id.pr = s_nametoeproto (s);
			if (yylval.id.pr == -1)
				lex_err (s);
			return;
		case IP:
			/* ip proto */
			
			yylval.id.pr = s_nametoproto (s);
			if (yylval.id.pr == -1)
				lex_err (s);
			return;
		default:
			lex_err (s);
		}
	}
}
