/*
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Lawrence Berkeley Laboratory,
 * Berkeley, CA.  The name of the University may not be used to
 * endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#ifndef lint
static  char rcsid[] =
	"@(#)$Header: tcpdump.c,v 1.17 90/01/12 18:12:33 mccanne Exp $ (LBL)";
static char copyright[] =
	"Copyright (C) 1987, Van Jacobson, Lawrence Berkeley Laboratory";
#endif

/*
 * tcpdump - monitor tcp/ip traffic on an ethernet.
 *
 * Copyright (c) 1987 by Van Jacobson, Lawrence Berkeley Laboratory
 *
 * First written in 1987 by Van Jacobson, Lawrence Berkeley Laboratory.
 * Mercilessly hacked and occasionally improved since then via the
 * combined efforts of Van, Steve McCanne and Craig Leres of LBL.
 *
 * Modifications made to accomodate the new SunOS4.0 NIT facility by
 * Micky Liu, micky@cunixc.cc.columbia.edu, Columbia University in May, 1989.
 * Major changes were in tcpdump.c to use the new STREAMS based NIT.
 */

#include <stdio.h>
#include <netdb.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <net/nit.h>

#ifdef SUNOS4
#include <sys/dir.h>
#include <net/nit_if.h>
#include <net/nit_pf.h>
#include <net/nit_buf.h>
#include <sys/stropts.h>
#endif

#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#include <netinet/ip_var.h>
#include <netinet/udp.h>
#include <netinet/udp_var.h>
#include <netinet/tcp.h>
#include <netinet/tcpip.h>

#ifdef SUNOS4
#include "sunos4.map.h"
#endif

#include "interface.h"
#include "filter.h"
#include "sample.h"

#define BUFSPACE        4*8192
#define CHUNKSIZE       2048
#define SNAPLEN         (max(sizeof (struct sample),96))
#define NFDS 32

int nflag;			/* leave addresses as numbers */
int pflag;			/* don't go promiscuous */
int qflag;			/* quick (shorter) output */
int cflag;			/* only print 'cnt' packets */
int tflag = 1;			/* print packet arrival time */
int eflag;			/* print ethernet header */
int vflag;
int xflag;			/* print packet in hex */
int oflag = 1;			/* run filter code optimizer */

int dflag;			/* print filter code */

#ifdef YYDEBUG
extern int yydebug;
#endif YYDEBUG

int thiszone;			/* gmt to local correction */

u_long netmask;
u_long localnet;

void cleanup ();
void timestampinit ();
void ether_if_print();
void (*printit)() = ether_if_print;

int snaplen = SNAPLEN;

void
main (argc, argv)
	int argc;
	char **argv;
{
	int *parse ();
	unsigned char buf[BUFSPACE];

	int cc, i, cnt, if_fd = -1, flen;
	struct timeb zt;
	u_long localhost;
	char thishost[256];
	register int *fcode;

	static char *device;
	static int bufspace = BUFSPACE;
	static int chunksize = CHUNKSIZE;

	/*
	 *
	 */

	while (argc > 1 && argv[1][0] == '-') {

		switch (argv[1][1]) {
		case 'c':
			if (argc < 3)
				usage();
			cflag++;
			cnt = atoi(argv[2]);
			argc--;
			argv++;
			break;
		case 'd':
#ifdef YYDEBUG
			if (argv[1][2] == 'p')
				yydebug = 1;
			else 
#endif YYDEBUG
			{
				dflag++;
				break;
			}
			break;
		case 'e':
			eflag++;
			break;
		case 'f':
			gethostname (thishost, sizeof(thishost));
			localhost = s_nametoaddr (thishost);
			netmask = (localhost < 0x80000000)? 0xff000000 :
				(localhost < 0xc0000000)? 0xffff0000 :
					0xffffff00;
			localnet = localhost & netmask;
			break;
		case 'i':
			if (argc < 3)
				usage();
			device = argv[2];
#ifdef CSLIP
			if (strncmp(device, "sl", 2) == 0) {
				printit = sl_if_print;
				pflag = 1;
			}
#endif
			argc--;
			argv++;
			break;
		case 'l':
			setlinebuf(stdout);
			break;
		case 'n':
			nflag++;
			break;
		case 'O':
			oflag = 0;
			break;
		case 'p':
			pflag++;
			break;
		case 'q':
			qflag++;
			break;
		case 's':
			if (argc < 3)
				usage();
			snaplen = max(atoi(argv[2]), SNAPLEN);
			argc--;
			argv++;
			break;
		case 't':
			tflag = 0;
			break;
		case 'v':
			vflag++;
			break;
		case 'x':
			xflag++;
			break;
		default:
			fprintf (stderr, "unrecognized option: '%s'\n", 
				 argv[1]);
			break;
		}
		argc--;
		argv++;
	}

	if (tflag) {
		int now;
		
		ftime (&zt);
		thiszone = zt.timezone * -60;
		
		now = time (0);
		if (localtime (&now)->tm_isdst)
			thiszone += 3600;

		/* Machine specific timestamp initialization */
		timestampinit();
	}

	initservarray();

	fcode = parse (&argv[1], &flen, oflag);

	if (! fcode)
		punt ("specification rejects all packets");
	
	if (dflag) {
		show_code (fcode, flen);
		exit (0);
	}

	if_fd = initdevice (device, pflag, bufspace, chunksize, snaplen);
	(void)signal (SIGINT, cleanup);
	(void)signal (SIGHUP, cleanup);
	
	/*
	 * Let user own process after socket has been opened.
	 */
	setuid (getuid ());

	i = 0;

#ifndef SUNOS4
	while ((cc = read (if_fd, (char *)buf, sizeof (buf))) > 0) {
		register u_char *bp, *bstop;
		register u_char *sp;
		register struct nit_hdr *nh;
		register int datalen = 0;
		
		/*
		 * Loop through each packet.  The increment expression
		 * rounds up to the next int boundary past the end of
		 * the previous packet.
		 */
		bstop = buf + cc;
		for (bp = buf; bp < bstop;
		     bp += ((sizeof(struct nit_hdr)+datalen+sizeof(int)-1)
			    & ~(sizeof (int)-1))) {

			nh = (struct nit_hdr *)bp;
			sp = bp + sizeof (*nh);
			
			switch (nh->nh_state) {
			case NIT_CATCH:
				datalen = nh->nh_datalen;
				break;
			case NIT_SEQNO:
			case NIT_NOMBUF:
			case NIT_NOCLUSTER:
			case NIT_NOSPACE:
				datalen = 0;
				continue;
			default:
				(void)fprintf(stderr,
				      "bad nit state %d\n", nh->nh_state);
				exit (1);
			}
			
			if (execute (sp, nh->nh_wirelen, fcode)) {
				(*printit)(sp, nh);
				i++;
				if (cflag && i >= cnt)
					exit (0);
			}
		}
	}
#else
	while ((cc = read(if_fd, (char*)buf, chunksize)) >= 0) {
		register u_char *bp, *cp, *bufstop;
		struct nit_hdr nh;
		struct nit_bufhdr *hdrp;
		struct nit_iftime *ntp;
		struct nit_iflen  *nlp;

		bp = buf;
		bufstop = buf + cc;
		/*
		* loop through each snapshot in the chunk
		*/
		while (bp < bufstop) {
			cp = bp;
			/* get past NIT buffer  */
			hdrp = (struct nit_bufhdr *)cp;
			cp += sizeof(*hdrp);

			/* get past NIT timer   */
			ntp = (struct nit_iftime *)cp;
			cp += sizeof(*ntp);

			/* get past packet len  */
			nlp = (struct nit_iflen *)cp;
			cp += sizeof(*nlp);

			/* next snapshot        */
			bp += hdrp->nhb_totlen;

			/* for printit()        */
			nh.nh_wirelen = nlp->nh_pktlen;
			nh.nh_timestamp.tv_sec = ntp->nh_timestamp.tv_sec;
			nh.nh_timestamp.tv_usec = ntp->nh_timestamp.tv_usec;

			if (execute (cp, nlp->nh_pktlen, fcode)) {
				(*printit)(cp, &nh);
				i++;
				if (cflag && i >= cnt)
					exit (0);
			}
		}
	}
#endif
	perror ("read");
	exit (-1);
}

/* make a clean exit on interrupts */
void
cleanup ()
{
	exit (0);
}

/* These defaults display timestamps to hundreths */
char *timestamp_fmt = "%02d:%02d:%02d.%02d ";
long timestamp_scale = 10000;

/* The sun includes are arranged too stupidly to help us */
#ifdef sun
#define ARCH_MASK	0xf0000000
#define MACH_MASK	0x0f000000

#define ARCH_SUN4C	0x50000000
#define MACH_60		0x01000000	/* Sparcstation 1 */

#define ARCH_SUN4	0x20000000
#define MACH_260	0x01000000
#define MACH_110	0x02000000
#define MACH_330	0x03000000
#define MACH_460	0x04000000
#endif

void
timestampinit()
{
#ifdef sun
	u_long id = gethostid();
	int good = 0;
#endif

#ifdef sun
	if ((id & ARCH_MASK) == ARCH_SUN4) {
		if ((id & MACH_MASK) == MACH_330)
			good++;
		else if ((id & MACH_MASK) == MACH_460)
			good++;
	}
	if ((id & ARCH_MASK) == ARCH_SUN4C)
		good++;
#endif

#ifdef sun
	/* On machines with "good" clocks, timestamps to microseconds */
	if (good) {
		timestamp_fmt = "%02d:%02d:%02d.%06d ";
		timestamp_scale = 1;
	}
#endif
}

void
ether_if_print(sp, nh)
	u_char *sp;
	register struct nit_hdr *nh;
{
	int length;
	register int i;
	struct in_addr inet_makeaddr();

	if (tflag) {
		i = (nh->nh_timestamp.tv_sec + thiszone) % 86400;
		(void)printf (timestamp_fmt,
			i / 3600, (i % 3600) / 60, i % 60,
			nh->nh_timestamp.tv_usec / timestamp_scale);
	}
	length = nh->nh_wirelen;
	if (eflag)
		ether_print (sp, length);

	length -= sizeof (struct ether_header);
	switch (TYPE (sp)) {

	case ETHERPUP_IPTYPE:
		ip_print (Ip (sp), length);
		break;

	case ETHERPUP_ARPTYPE:
	case ETHERPUP_REVARPTYPE:
		arp_print (Arp (sp), length);
		break;

	default:
		if (! eflag)
			ether_print (sp, length);
		if (!xflag && !qflag)
			default_print ((u_short *)(sp + sizeof (struct ether_header)), length);
		break;
	}
	if (xflag)
		default_print ((u_short *)(sp + sizeof (struct ether_header)),
			       length);
	putchar ('\n');
}

void
default_print (sp, length)
	register u_short *sp;
	register int length;
{
	register int i, j;

	j = (min(snaplen,length) + (sizeof(u_short) - 1)) / sizeof(u_short);
	i = 0;
	(void)printf("\n\t\t\t");
	while (--j >= 0) {
		i++;
		(void)printf(" %04x", *sp++);
		if (j && i == 8) {
			i = 0;
			(void)printf("\n\t\t\t");
		}
	}
}

void
usage()
{
	(void)fprintf(stderr,
"Usage: tcpdump [-e|f|l|n|p|q|t] [-c cnt] [-i interface] [options]\n");
	(void)fprintf(stderr,
"options are:\n");
	(void)fprintf(stderr,
"       [ether|ip|arp|rarp] [dst|src] [host] id\n");
	(void)fprintf(stderr,
"       [tcp|udp] [dst|src] [port] id\n");
	(void)fprintf(stderr,
"       [ether|ip] [proto] id\n");
	(void)fprintf(stderr,
"       between gateway less greater byte broadcast\n");

	exit(-1);
}
