/*
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Lawrence Berkeley Laboratory,
 * Berkeley, CA.  The name of the University may not be used to
 * endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *  Optimization module for tcpdump virtual machine code.
 */
#ifndef lint
static char rcsid[] =
    "@(#) $Header: optimize.c,v 1.6 90/01/12 18:12:26 mccanne Exp $ (LBL)";
#endif

#include <stdio.h>

#include "interface.h"
#include "filter.h"

#define max(a,b) ((b)>(a)?(b):(a))

/*
 * The code array and length for this module.
 */
static int olen;
static int *ocode;

/*
 * A flag to indicate that further optimization is needed.
 * Iterative passes are continued until a given pass yields no 
 * branch movement.
 */
static int not_done;

/*
 * An instruction is marked if only if its mark equals the current mark.
 * Rather than traverse the code array, marking each item, CUR_MARK is
 * incremented.  This automatically makes each element unmarked.
 *
 * The marks array is indexed by the same index as the code array.
 */
static int *marks;
static int cur_mark;
#define Mark(index)		(marks[index])
#define isMarked(index)		(marks[index] == cur_mark)

/*
 * Data accesors to the code array.
 */
#define OpCode(index)		(((struct insn *)&ocode[index])->opcode)
#define JumpT(index)		(((struct insn *)&ocode[index])->jt)
#define JumpF(index)		(((struct insn *)&ocode[index])->jf)
#define Data(index,i)		(((struct insn *)&ocode[index])->data[i])
#define Aux(index)		(((struct insn *)&ocode[index])->aux)

/*
 * The numbers of times a given instruction is a target of a jump.
 * This array is indexed by the same index as the code array.
 */
static int *edges;
#define Edges(index)		(edges[index])

/*
 * A data structure to house information that is accumulated along a
 * particular thread of execution.  The index of the instruction is stored
 * in the stack along with the type of branch taken (i.e. true or false).
 * By looking in this stack, we can often know the outcome of the current
 * instruction, and thereby delete it.
 */
static int tos;
struct st_elem { int index, type; } *stack;

#define Push(i, t)		(stack[tos].index = (i), \
				 stack[tos++].type = (t))
#define Pop()			(tos--)
#define Index(i)		(stack[i].index)
#define Type(i)			(stack[i].type)

#define NULL_SET		(tos)

static void remove_edge ();
static void optimize0 ();
static void make_marks ();
static void mark_code ();
static void count_edges ();
static void compress ();

static int redundant ();
static int eq_operand_p ();
static int rm_unused_side_effects ();


/*
 * REMOVE_EDGE	Decrement the edge count of node at INDEX.  Also, recursively
 *		decrement children edge counts if edge count of current node
 *		reaches zero (i.e. is unreachable).
 */
static void
remove_edge (index)
	int index;
{
	if (! --Edges (index) && ! isLeaf (index)) {
		remove_edge (index + JumpT (index));
		remove_edge (index + JumpF (index));
	}
}

/*
 * REDUNANT	Determine if an instruction is unnecessary based on the
 *		information that has been accumulated by traversing
 *		the filter code tree.
 *
 *		Return true if the instruction at INDEX is not needed.
 *		START is the stack index of the current generated set.
 *		*BOOL contains the result that this operation will
 *		always give.
 */
static int
redundant (index, bool, start)
	register int index, start;
	int *bool;
{
	register int i;

	if (isLeaf (index))
		return FALSE;

	if (JumpT (index) == JumpF (index) && ! HasSideEffects (index)) {
		/*
		 * Common branch targets can be eliminated.
		 */

		*bool = TRUE;	/* this could just as well be FALSE */
		
		return TRUE;
	}
	
	if (OpCode (index) > EtherProtoOp) {
		/*
		 * XXX 
		 * 
		 * Currently, only packet protocol operations are
		 * optimized.  These all have opcodes less than
		 * EtherProtoOp.
		 */
		return FALSE;
	}
	
	for (i = start; i < tos; i += 1)
		/*
		 *  Traverse current generated set, looking for instructions
		 *  with identical opcodes.
		 */
		if (OpCode (index) == OpCode (Index (i))) {
			
			if (eq_operand_p (index, Index (i))) 
				/*
				 * The operands are the same, so the result
				 * is true if a true branch was taken to
				 * get here, otherwise false.
				 */
				if (Type (i))
					*bool = TRUE;
				else
					*bool = FALSE;
			
			else
				if (Type (i))
					/*
					 * This operation is true, but the
					 * current operand is different;
					 * the result must be false.
					 */
					*bool = FALSE;
			
				else
						continue;
			return TRUE;
		}
	
	return FALSE;
}


static void
optimize0 (index, set)
	register int index, set;
{
	register int target;
	int bool;

	if (! isLeaf (index) && ! isMarked (index)) {

		Mark (index) = cur_mark;

		if (Edges (index) != 1)
			/*
			 * Since more than one path exists to the current node
			 * and it is uncertain which path has been taken,
			 * all generated information is killed.
			 */
			set = NULL_SET;

		/*
		 * Add the current node to the generated set as true,
		 * and optimize its true branch.
		 * Optimize the child first so we won't have to do
		 * as much work at the current node.
		 */
		Push (index, TRUE);
		optimize0 (JumpT (index) + index, set);

		/*
		 * Target is the absolute index of the jump destination.
		 */
		target = JumpT (index) + index;

		/*
		 * Now optimize the current node.
		 */
		while (redundant (target, &bool, set)) {

			not_done = 1; /* tell outer loop we're not done */

			if (bool)
				target += JumpT (target);
			else
				target += JumpF (target);

		}
		Edges (target) += 1;
		remove_edge (index + JumpT (index));
		JumpT (index) = target - index;

		Pop ();

		/*
		 * Do same thing for the false branch.
		 */

		Push (index, FALSE);
		optimize0 (JumpF (index) + index, set);

		target = JumpF (index) + index;

		while (redundant (target, &bool, set)) {

			not_done = 1;
			
			if (bool)
				target += JumpT (target);
			else
				target += JumpF (target);

		}
		Edges (target) += 1;
		remove_edge (index + JumpF (index));
		JumpF (index) = target - index;

		Pop ();

	}
}
			
/*
 * OPTIMIZE		Optimize the filter code, FCODE.
 *			*FLEN is the length of the code array.
 *			The array is modified and its new length is 
 *			returned in *FLEN.
 */
void
optimize (fcode, flen)
	int *fcode;
	int *flen;
{
	ocode = fcode;
	olen = *flen;

	stack = (struct st_elem *) malloc (olen * sizeof (*stack));
	marks = (int *) malloc (olen * sizeof (*marks));
	edges = (int *) malloc (olen * sizeof (*edges));

	if (! stack || ! marks || ! edges)
		punt ("memory allocation failed");

	count_edges ();

	not_done = 1;
	while (not_done) {
		not_done = 0;
		cur_mark += 1;
		optimize0 (0, NULL_SET);
#define SE_OPTIMIZE
#ifdef SE_OPTIMIZE
		rm_unused_side_effects (0);
#endif
	}

	compress ();

	free ((char *) stack);
	free ((char *) marks);
	free ((char *) edges);

	*flen = olen;
}

/* This code isn't too useful. */
#ifdef SE_OPTIMIZE
/*
 * RM_UNUSED_SIDE_EFFECTS	Traverse the code array eliminating 
 *				instructions whose only purpose is to
 *				perform a side effect that is never used.
 */
static int
rm_unused_side_effects (i)
{
	int target;

	if (! isLeaf (i)) {
		if (HasSideEffects (i)) {
			target = i + JumpT (i);
			if (HasSideEffects (target)) {
				/*
				 * Since this insn's target has side effects
				 * as well, we don't know whether they are
				 * used until after the recursive call.
				 */
				int new = rm_unused_side_effects (target);

				if (new != target) {
					/* 
					 * The new address is different
					 * from what we had, meaning the
					 * side effect of this insn is
					 * never used.  In this case, the
					 * new target is propagated back.
					 */
					not_done = 1;
					return new;
				}
				else
					/*
					 * The results will be used so
					 * return the old address.
					 */
					return i;
			}
			/* This optimization is dependent on the way the 
			   code is generated.  E.g. that the result of an 
			   arithmetic sequence is immediately used,
			   except for interspersed protocol ops. */

			else if (! UsesSideEffects (target)
				 /*
				  * Protocol ops can be imbedded inside
				  * a side arithmetic sequence.
				  */
				 && OpCode (target) != IpProtoOp
				 && OpCode (target) != EtherProtoOp)
				/*
				 * The immediate target does not use
				 * our side effect, so propagate back
				 * its address, effectively removing
				 * this insn from the code.
				 */
				return target;
		}
		else if (isJump (i)) {
			JumpT (i) = rm_unused_side_effects (i + JumpT (i)) - i;
			JumpF (i) = rm_unused_side_effects (i + JumpF (i)) - i;
		}
	}
	return i;
}
#endif

/*
 * EQ_OPERAND_P			True iff instructions at X and Y 
 *				perform the same operation.
 */
static int
eq_operand_p (x, y)
	int x, y;
{
	if (OpCode (x) != OpCode (y))
		return FALSE;

	switch (OpCode (x)) {

	default:
		punt ("unknown op in eq_operand_p");

	case ArpDstOp:
	case ArpSrcOp:
	case RarpDstOp:
	case RarpSrcOp:
	case IpDstOp:
	case IpSrcOp:
		return Data (x, 0) == Data (y, 0);

	case EDstOp:
	case ESrcOp:

	case AndByteOp:
	case OrByteOp:
	case EqByteOp:
	case LTByteOp:
	case GTByteOp:
		return Data (x, 0) == Data (y, 0) 
			&& Aux (x) == Aux (y);
			      
	case ArpDstNetOp:
	case ArpSrcNetOp:
	case RarpDstNetOp:
	case RarpSrcNetOp:
	case IpDstNetOp:
	case IpSrcNetOp:
		return Data (x, 0) == Data (y, 0) &&
			Data (x, 1) == Data (y, 1);

	case TcpDstPortOp:
	case TcpSrcPortOp:
	case UdpDstPortOp:
	case UdpSrcPortOp:
	case EtherProtoOp:
	case LessOp:
	case GreaterOp:
	case IpProtoOp:
		return Aux (x) == Aux (y);

	case AcceptOp:
	case RejectOp:
		return TRUE;
	}
}

static void
make_marks (i)
	register int i;
{
	if (! isMarked (i)) {

		Mark (i) = cur_mark;

		if (! isLeaf (i)) {
			make_marks (i + JumpT (i));
			make_marks (i + JumpF (i));
		}
	}
}
	
/*
 *  MARK_CODE		Mark code array such that isMarked(i) is true
 *			only for nodes that are alive.
 *			Mark fields must have been previously initialized 
 *			(by count_edges).
 */
static void
mark_code ()
{
	cur_mark += 1;
	make_marks (0);
}

/*
 * COUNT_EDGES		Determine number of "in edges" at each node.
 *			Initialize mark field.
 *			Must be called before optimize0.
 */
static void
count_edges ()
{
	register int i = 0;

	while (i < olen) {
		Mark (i) = cur_mark;
		Edges (i) = 0;
		i += insn_size[OpCode (i)];
	}

	mark_code ();

	i = 0;
	
	while (i < olen) {

		if (isMarked (i) && ! isLeaf (i)) {
			Edges (i + JumpT (i)) += 1;
			/*
			 * Both jumps to the same insn can be
			 * considered a single edge.
			 */
			if (JumpT (i) != JumpF (i))
				Edges (i + JumpF (i)) += 1;
		}
		i += insn_size[OpCode (i)];
	}
	Edges (0) = 1;
}

/*
 *  COMPRESS		Remove dead code from the code array.
 *			The MARKS and EDGES array are invalidated.
 */
static void
compress ()
{
	register int i, delta, src, dst;

	mark_code ();

	if (isJump (0) && JumpT (0) == JumpF (0))
		/*
		 * This is a special case optimization that 
		 * OPTIMIZE misses.  I guess it really doesn't
		 * belong here, but this is a convenient way to
		 * delete the instruction at index 0.
		 */
		Mark (0) -= 1;
	
	/*
	 * At this point the EDGES array is no longer needed, so it is
	 * used to keep information which tells us how to fix up jumps.
	 * We remove code in two passes.  First, the offset for each 
	 * jump is placed in EDGES.  Then, the actual code movement and
	 * jump altering is done.
	 */
#define Offset(i) (edges[i])

	i = 0;
	delta = 0;

	while (i < olen) {

		Offset (i) = delta;

		if (! isMarked (i))
			delta += insn_size[OpCode (i)];

		i += insn_size[OpCode (i)];
	}

	src = 0;
	dst = 0;

	while (src < olen) {

		while (! isMarked (src))
			src += insn_size[OpCode (src)];

		if (isJump (src)) {
			JumpT (src) -= Offset (src + JumpT (src)) - 
				Offset (src);
			JumpF (src) -= Offset (src + JumpF (src)) -
				Offset (src);
		}

		bcopyi (&ocode[src], &ocode[dst], insn_size[OpCode (src)]);

		src += insn_size[OpCode (dst)];
		dst += insn_size[OpCode (dst)];
	}
	olen = dst;
}

/*
 * BCOPYI	Copy LEN integers from SRC to DST.
 */
bcopyi (src, dst, len)
	register int *src, *dst, len;
{
	while (len--)
		*dst++ = *src++;
}
