%{
/*
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Lawrence Berkeley Laboratory,
 * Berkeley, CA.  The name of the University may not be used to
 * endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Grammar for tcpdump.
 */
#ifndef lint
static char rcsid[] =
    "@(#) $Header: tcpgram.y,v 1.10 90/02/24 23:14:39 mccanne Exp $ (LBL)";
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>

#ifdef SUNOS4
#include "sunos4.map.h"
#endif

#include "interface.h"
#include "tcpgram.h"
#include "filter.h"

static int new_op ();
static int genArithOp ();
static int alloc_reg ();
static int genLoadImm ();
static int genLoad ();
static int genOpcode ();

static struct blist And ();
static struct blist Or ();
static struct blist Not ();
static struct blist genCode ();
static struct blist genProtoAbbrev ();
static struct blist genRelOp ();
static struct blist amerge ();

static void pass2 ();
static void backpatch ();
static void yyerror ();
static void fc_error ();

static int reg_pair_to_offset ();
static int alloc_reg ();
static void free_reg ();
static void init_reg ();

static struct blist null_list = { -1, -1, -1 };

int n_errors = 0;

/*
 * The length of each type of instruction in int's.  
 */
const int insn_size[] =  {
	3, 3,			/* ether */
	3, 3, 4, 4,		/* ip */
	3, 3, 4, 4,		/* arp */
	3, 3, 4, 4,		/* rarp */
	2, 2,			/* tcp ports */
	2, 2,			/* udp ports */
	2,			/* ip proto */
	2,			/* ether proto */
	2, 2,			/* less/greater */
	3, 3, 3, 3, 3,		/* byte ops */

	2, 2, 2, 2, 2, 2,	/* rel ops */
	2, 2, 2, 2, 2, 2,	/* rel ops */
	2, 2, 2, 2, 2, 2,	/* rel ops */
	2, 2, 2, 2, 2, 2,	/* rel ops */
	2, 2, 2, 2, 2, 2,	/* rel ops */
	2, 2, 2, 2, 2, 2,	/* rel ops */

	2, 2, 2, 2, 2, 2,	/* arith ops XXX size == 1? */
	2, 2, 2, 2, 2, 2,	/* arith ops */
	2, 2, 2, 2, 2, 2,	/* arith ops */
	2, 2, 2, 2, 2, 2,	/* arith ops */
	2, 2, 2, 2, 2, 2,	/* arith ops */
	2, 2, 2, 2, 2, 2,	/* arith ops */

	3,			/* load imm */
	3,			/* load imm */
	3,			/* load imm */

	3, 3, 3, 3, 3, 3,	/* load proto */
	3, 3, 3, 3, 3, 3,	/* load proto */
	3, 3, 3, 3, 3, 3,	/* load proto */

	1, 1,			/* accept/reject */
	1			/* undef */
};

/*
 * The parser generates jump destinations of the form:
 * "jump on true/false to x", where x is an absolute index.
 * A second pass converts them to the form:
 * "jump if false to y, jump if true to z", where y and z are relative indices
 *
 * The jt field of the instructions data structure is used for jump (per
 * the definitions below), while the jf field supplie the absolute
 * destination.
 */
#define JF   0			/* jump on false */
#define JT   1			/* jump on true */

#define Jump(index)		(((struct insn *)&fcode[index])->jf)
#define invertJmp(index)	(((struct insn *)&fcode[index])->jt =\
				!((struct insn *)&fcode[index])->jt)

#define JumpT(index)		(((struct insn *)&fcode[index])->jt)
#define JumpF(index)		(((struct insn *)&fcode[index])->jf)
#define Data(index,i)		(((struct insn *)&fcode[index])->data[i])
#define Aux(index)		(((struct insn *)&fcode[index])->aux)
#define OpCode(index)		(((struct insn *)&fcode[index])->opcode)
#define Src(index)		(((struct insn *)&fcode[index])->u.reg.src)
#define Dst(index)		(((struct insn *)&fcode[index])->u.reg.dst)

static int *fcode;

#define Q_Prim(x)	((x) & 0xf)

#define Q_Host		0x02
#define Q_Net		0x03
#define Q_Port		0x04
#define Q_Gateway	0x05
#define Q_Proto		0x06

#define Q_Protocol(x)	((x) & 0xf0)

#define Q_Ether		0x10
#define Q_Ip		0x20
#define Q_Arp		0x30
#define Q_Rarp		0x40
#define Q_Tcp		0x50
#define Q_Udp		0x60

#define Q_Dir(x)	((x) & 0x700)

#define Q_Src		0x0100
#define Q_Dst		0x0200
#define Q_Or		0x0300
#define Q_And		0x0400

#define Q_Default	0
#define Q_Undef		-1

%}

%union {
	int i;
	struct {
		int type;
		struct blist list;	/* index of current op, true list,
					   and false list */
	} list;
	union id id;
}

%type	<list>	expr and or not id idlist spec thing proto 
%type	<list>	pqual pname dqual aqual ndaqual other lp aname arith 
%type	<i>	byte.op relop 

%token  DST SRC HOST BETWEEN GATEWAY
%token  NET PORT LESS GREATER PROTO BYTE
%token  ARP RARP IP TCP UDP
%token  BROADCAST
%token  NUM ID
%token  ETHER
%token	GEQ LEQ NEQ

%type	<id> ID
%type	<i> NUM

%left OR AND
%nonassoc  '!'
%left '&' '|'
%left '+' '-'
%left '*' '/'
%%
prog:	  expr		{ 
				int out;
				
				backpatch ($1.list.t, new_op (AcceptOp));
				backpatch ($1.list.f, out = new_op (RejectOp));
				Jump ($1.list.op) = out;
			}

	| /* null */		{ (void) new_op (AcceptOp); }
	;
expr:	  spec
	| expr and spec		{ $$.list = And ($1.list, $3.list);
				  $$.type = $3.type; }
	| expr or spec		{ $$.list = Or ($1.list, $3.list); 
				  $$.type = $3.type; }
	| expr and id		{ $$.list = And ($1.list, $3.list);
				  $$.type = $3.type; }
	| expr or id		{ $$.list = Or ($1.list, $3.list); 
				  $$.type = $3.type; }
	;
id:	  ID			{ $$.type = $<list>0.type;
				  $$.list = genCode ($1, $$.type); }
	| not id		{ $$.list = Not ($2.list); $$.type = $2.type; }
	| lp idlist ')'		{ $$ = $2; }
	;
lp:	  '('			{ $$.type = $<list>0.type; }
	;
not:	  '!'			{ $$.type = $<list>0.type; }
	;
idlist:	  id
	| idlist and id		{ $$.list = And ($1.list, $3.list);
				  $$.type = $3.type; }
	| idlist or id		{ $$.list = Or ($1.list, $3.list);
				  $$.type = $3.type; }
	;
and:	  AND			{ $$.type = $<list>0.type; }
	;
or:	  OR			{ $$.type = $<list>0.type; }
	;
spec:	  thing
	| not spec		{ $$.list = Not ($2.list); $$.type = $2.type; }
	;
thing:	  pqual dqual aqual id	{ $$ = $4; }
	| pqual dqual id	{ $$ = $3; }
	| pqual aqual id	{ $$ = $3; }
	| pqual proto id	{ $$ = $3; }
	| pqual ndaqual id	{ $$ = $3; }
	| lp expr ')'		{ $$ = $2; }
	| pname			{ $$.list = genProtoAbbrev ($1.type);
				  $$.type = Q_Undef; }
	| arith relop arith	{ $$.list = genRelOp ($2, $1.type, $3.type);
				  /* Check to see if the arith's have
				     protocol info in them.  If so we
				     must AND them with the relop that
				     was just made. */
				  if ($1.list.op >= 0) {
					  if ($3.list.op >= 0)
						  $$.list = And (And ($1.list,
								      $3.list),
								 $$.list);
					  else
						  $$.list = And ($1.list, 
								 $$.list);
				  } else if ($3.list.op >= 0)
					  $$.list = And ($3.list, $$.list);

				  $$.type = Q_Undef; }
	| other
	;
proto:	  PROTO			{ $$.type = $<list>0.type | Q_Proto; }
	;
/* protocol level qualifiers */
pqual:	  pname
	|			{ $$.type = 0; }
	;
pname:	  ETHER			{ $$.type = Q_Ether; }
	| IP			{ $$.type = Q_Ip; }
	| ARP			{ $$.type = Q_Arp; }
	| RARP			{ $$.type = Q_Rarp; }
	| TCP			{ $$.type = Q_Tcp; }
	| UDP			{ $$.type = Q_Udp; }
	;
/* 'direction' qualifiers */
dqual:	  SRC			{ $$.type = $<list>0.type | Q_Src; }
	| DST			{ $$.type = $<list>0.type | Q_Dst; }
	| SRC OR DST		{ $$.type = $<list>0.type | Q_Or; }
	| DST OR SRC		{ $$.type = $<list>0.type | Q_Or; }
	| SRC AND DST		{ $$.type = $<list>0.type | Q_And; }
	| DST AND SRC		{ $$.type = $<list>0.type | Q_And; }
	;
/* address type qualifiers */
aqual:	  HOST			{ $$.type = $<list>0.type | Q_Host; }
	| NET			{ $$.type = $<list>0.type | Q_Net; }
	| PORT			{ $$.type = $<list>0.type | Q_Port; }
	;
/* non-directional address type qualifiers */
ndaqual:  BETWEEN 		{ $$.type = $<list>0.type | Q_Host; }
	| GATEWAY 		{ $$.type = $<list>0.type | Q_Gateway; }
	;
other:   BROADCAST		{
				$$.list.op = new_op (EDstOp);
				Aux ($$.list.op) = 0xffff;
				Data ($$.list.op, 0) = 0xffffffff;
				$$.list.t = $$.list.f = -1;
				}
	| LESS NUM		{
				$$.list.op = new_op (LessOp);
				Aux ($$.list.op) = $2;
				$$.list.t = $$.list.f = -1; 
				}
	| GREATER NUM		{
				$$.list.op = new_op (GreaterOp);
				Aux ($$.list.op) = $2;
				$$.list.t = $$.list.f = -1;
				}
	| BYTE NUM byte.op NUM
		{
			switch ($3) {
			case '&':  $$.list.op = new_op (AndByteOp); break;
			case '|':  $$.list.op = new_op (OrByteOp);  break;
			case '=':  $$.list.op = new_op (EqByteOp);  break;
			case '<':  $$.list.op = new_op (LTByteOp);  break;
			case '>':  $$.list.op = new_op (GTByteOp);  break;
			}
			Data ($$.list.op, 0) = $2;	/* index */
			Aux ($$.list.op) = $4;	/* value */
			$$.list.t = $$.list.f = -1;
		}
	;
relop:	  '>'				{ $$ = GTOp; }
	| GEQ				{ $$ = GEOp; }
	| '='				{ $$ = EQOp; }
	| LEQ				{ $$ = LEOp; }
	| '<'				{ $$ = LTOp; }
	| NEQ				{ $$ = NEOp; }
	;

/* The register number is in the type field, while the list is 
   formed in the list field.  This expressions are chained together
   in a logical and fashion so the boolean combining can be maintained
   and not have to special case these. */

arith:	  aname '[' arith ']'		{ $$.type = genLoad ($1.type, 
							     $3.type, 1);
					  $$.list = amerge ($1.list, $3.list);}
	| aname '[' arith ':' NUM ']'	{ $$.type = genLoad ($1.type, 
							     $3.type, $5);
					  $$.list = amerge ($1.list, $3.list);}
	| NUM				{ $$.type = genLoadImm ($1);
					  $$.list = null_list; }
	| arith '+' arith	{ $$.type = genArithOp (AddOp,$1.type,$3.type);
				  $$.list = amerge ($1.list, $3.list); }
	| arith '-' arith	{ $$.type = genArithOp (SubOp,$1.type,$3.type);
				  $$.list = amerge ($1.list, $3.list); }
	| arith '*' arith	{ $$.type = genArithOp (MulOp,$1.type,$3.type);
				  $$.list = amerge ($1.list, $3.list); }
	| arith '/' arith	{ $$.type = genArithOp (DivOp,$1.type,$3.type);
				  $$.list = amerge ($1.list, $3.list); }
	| arith '&' arith	{ $$.type = genArithOp (AndOp,$1.type,$3.type);
				  $$.list = amerge ($1.list, $3.list); }
	| arith '|' arith	{ $$.type = genArithOp (OrOp,$1.type,$3.type);
				  $$.list = amerge ($1.list, $3.list); }
	| lp arith ')'		{ $$ = $2; }
	;
aname:	  pname			{ $$.type = $1.type;
				  if ($1.type == Q_Ether)
					  /* Don't have to check for "ether" */
					  $$.list = null_list;
				  else
					  $$.list = genProtoAbbrev ($1.type); }
	;
byte.op:  '&'			{ $$ = '&'; }
	| '|'			{ $$ = '|'; }
	| '<'			{ $$ = '<'; }
	| '>'			{ $$ = '>'; }
	| '='			{ $$ = '='; }
	;
%%
static void
yyerror ()
{
	n_errors++;
}

static void
fc_error (s)
	char *s;
{
	fprintf (stderr, "filter expression: %s\n", s);
	exit (-1);
}

void
punt (s)
	char *s;
{
	fprintf (stderr, "*** tcpdump error: %s\n", s);
	exit (-1);
}

static int maxlen = 0;
static int curlen;

struct insn *
parse (argv, len, optimize_flag)
	char **argv;
	int *len, optimize_flag;
{
	init_reg ();
	lex_init (argv);

	maxlen = 0x40;
	curlen = 0;

	fcode = (int *) malloc (maxlen * sizeof (*fcode));

	if (! fcode) {
		perror ("malloc");
		exit (-1);
	}

	yyparse ();
	pass2 ();

	if (n_errors)
		fc_error ("syntax error");

	if (optimize_flag) {
		optimize (fcode, &curlen);
		if (((struct insn *)fcode)->opcode == RejectOp) {
			free (fcode);
			fcode = (int *) 0;
		}
	}

	*len = curlen;

	return (struct insn *) fcode;
}

/*
 * new_op	Initialize and return index of next available instruction.
 * 		OP is the opcode.  Return index of that instruction.
 */
static int 
new_op (op)
	int op;
{
	register struct insn *ip;
	register int oldlen;

	oldlen = curlen;
	curlen += insn_size[op];

	if (curlen > maxlen) {
		maxlen *= 2;
		fcode = (int *) realloc (fcode, maxlen * sizeof (*fcode));
		if (! fcode) {
			perror ("malloc");
			exit (-1);
		}
	}
	ip = (struct insn *) &fcode[oldlen];
	ip->opcode = op;

	if (insn_size[op] > 1) {
		ip->jf = -1;
		ip->jt = JF;
	}

	return oldlen;
}

/*
 *  backpatch	
 *
 *	Set the jf field of each instruction in the list starting at INDEX,
 *	to ADDR.  The list begins at INDEX.  The next element in this list
 *	is indicated by the original jf value.  The list is terminates
 *	with -1.
 */
static void
backpatch (index, addr)
	register int index, addr;
{
	register int temp;

	while (index >= 0) {
		temp = Jump (index);
		Jump (index) = addr;
		index = temp;
	}
}

/*
 *  merge
 *
 *	Merge list I1 to I2.
 *	The new list is returned.
 */
static int
merge (i1, i2)
	int i1, i2;
{
	if (i1 >= 0) {
		register int T = i1;

		/* Find end of list. */
		while (Jump (T) >= 0)
			T = Jump (T);

		/* Concatenate the lists. */
		Jump (T) = i2;

		return i1;
	} else
		return i2;
}

/*
 *  pass2
 *
 *	Convert absolute jumps to relative jumps.
 *
 *	Before pass2, the all jumps contain a jump type in jt, and an
 *	absolute jump target in jf.
 */
static void
pass2 ()
{
	register int i = 0;

	while (i < curlen) {

		register int size = insn_size[OpCode (i)];

		/* 
		 * If the size is <= 1, the instruction does not
		 * have jump fields associated with it.
		 */
		if (size > 1) {
			if (JumpT (i) == JF) {
				JumpT (i) = size;
				JumpF (i) -= i;
			} else {
				JumpT (i) = JumpF (i) - i;
				JumpF (i) = size;
			}
			/* 
			 * If it is not a jump, but it's size is > 1,
			 * it doesn't use its jump fields.  They are
			 * set to the next insn to eliminate these
			 * special cases in the optimiztion code.
			 */
			if (! isJump (i))
				JumpT (i) = JumpF (i) = size;
		}
		i += size;
	}
}

static struct blist 
Or (e1, e2)
	struct blist e1, e2;
{
	struct blist result;

	if (e1.f >= 0) 
		backpatch (e1.f, e1.op + insn_size[OpCode (e1.op)]);

	Jump (e1.op) = e1.t;
	invertJmp (e1.op);
	result.t = merge (e2.t, e1.op);
	result.f = e2.f;
	result.op = e2.op;

	return result;
}

static struct blist
And (e1, e2)
	/* e1 must have been generated before e2 */
	struct blist e1, e2;
{
	struct blist result;

	if (e1.t >= 0)
		backpatch (e1.t, e1.op + insn_size[OpCode (e1.op)]);

	Jump (e1.op) = e1.f;
	result.f = merge (e2.f, e1.op);
	result.t = e2.t;
	result.op = e2.op;

	return result;
}

static struct blist
Not (e)
	struct blist e;
{
	struct blist result;

	result.t = e.f;
	result.f = e.t;
	result.op = e.op;
	invertJmp (result.op);

	return result;

}

/*
 * This array maps a Q_Proto value to the protocol types defined
 * in <netinet/if_ether.h>.
 */
static const int proto_type[] = {
	-1, -1, ETHERPUP_IPTYPE, ETHERPUP_ARPTYPE, ETHERPUP_REVARPTYPE,
	IPPROTO_TCP, IPPROTO_UDP };

#define QUAL_TO_PROTO(q)  (proto_type[(q) >> 4])


static struct blist
genPortOp (id, type)
	union id id;
	int type;
	/*
	 * Q_Protocol(type) is the protocol specified by the user
	 * id.pp.pr is the protocol specified by /etc/services
	 */
{
	struct blist r;
	int op = UndefOp;
	int t;

	if (id.pp.pr == -1) {
		/*
		 * Protocol was specified by user.
		 */
		if (Q_Protocol (type) == Q_Tcp)
			id.pp.pr = IPPROTO_TCP;

		else if (Q_Protocol (type) == Q_Udp)
			id.pp.pr = IPPROTO_UDP;

		else {
			/*
			 * Don't know which protocol to check,
			 * so check them both.
			 */
			r = genPortOp (id, Q_Prim(type)|Q_Dir(type)|Q_Tcp);
			r = Or (r, genPortOp (id, Q_Prim(type)|Q_Dir(type)
					      |Q_Udp));
			
			return r;
		}
			
		op = genOpcode (type);
	}

	else {
		/*
		 * A port has been specified by /etc/services so the
		 * protocol is known.
		 */
		if (id.pp.pr == IPPROTO_TCP)

			op = genOpcode (Q_Prim(type)|Q_Dir(type)|Q_Tcp);

		else if (id.pp.pr == IPPROTO_UDP)

			op = genOpcode (Q_Prim(type)|Q_Dir(type)|Q_Udp);
		else
			/*
			 * Something is wrong if /etc/services gives us
			 * a non-udp/tcp port
			 */
			punt ("bogus port info from /etc/services");

	}

	Aux (t = new_op (EtherProtoOp)) = ETHERPUP_IPTYPE;
	Aux (r.f = new_op (IpProtoOp)) = id.pp.pr;
	Jump (r.f) = t;
	Aux (r.op = new_op (op)) = id.pp.pt;
	r.t = -1;

	return r;
}
	
static struct blist
genPort (id, type)
	union id id;
	int type;
{
	struct blist r;

	switch (Q_Dir (type)) {

	case Q_Src:
		return genPortOp (id, Q_Src|Q_Prim(type)|Q_Protocol(type));

	case Q_Dst:
		return genPortOp (id, Q_Dst|Q_Prim(type)|Q_Protocol(type));

	case Q_And:
		r = genPortOp (id, Q_Src|Q_Prim(type)|Q_Protocol(type));
		return And (r, genPortOp (id, Q_Dst|Q_Prim(type)
					  |Q_Protocol(type)));

	case Q_Or:
	case Q_Default:
		r = genPortOp (id, Q_Src|Q_Prim(type)|Q_Protocol(type));
		return Or (r, genPortOp (id, Q_Dst|Q_Prim(type)
					 |Q_Protocol(type)));

	default:
		punt ("illegal direction in genPort");
	}
	return r;
}

static struct blist
genHostOp (id, type)
	union id id;
	int type;
{
	struct blist result;

	switch (Q_Protocol (type)) {
		
	case Q_Ether:
		result.op = new_op (genOpcode (type));
		Data (result.op, 0) = *(int *)id.e;
		Aux (result.op) = *(short *)((int *)id.e + 1);
		result.f = result.t = -1;
		break;

	case Q_Ip:
	case Q_Arp:
	case Q_Rarp:
		Aux (result.f = new_op (EtherProtoOp)) = 
			QUAL_TO_PROTO (Q_Protocol (type));

		result.op = new_op (genOpcode (type));

		if (Q_Prim (type) == Q_Net) {
			Data (result.op, 0) = id.n.n;
			Data (result.op, 1) = id.n.m;
		}
		else
			Data (result.op, 0) = id.h;
			
		result.t = -1;
		break;

	case Q_Default:
		result = genHostOp (id, Q_Dir(type)|Q_Prim(type)|Q_Ip);
		result = Or (result, genHostOp (id, Q_Dir(type)|Q_Prim(type)
						|Q_Arp));
		result = Or (result, genHostOp (id, Q_Dir(type)|Q_Prim(type)
						|Q_Rarp));
			break;
		
	case Q_Tcp:
	case Q_Udp:
		fc_error ("bad modifier of host");

	default:
		punt ("illegal protocol in genHostOp");
	}
	return result;
}

static struct blist
genHost (id, type)
	union id id;
	int type;
{
	struct blist list;

	switch (Q_Dir (type)) {

	case Q_Src:
	case Q_Dst:
		return genHostOp (id, type);

	case Q_And:
		list = genHostOp (id, Q_Src|Q_Prim(type)|Q_Protocol(type));
		return And (list, genHostOp (id, Q_Dst|Q_Prim(type)
					     |Q_Protocol(type)));

	case Q_Or:
	case Q_Default:
		list = genHostOp (id, Q_Src|Q_Prim(type)|Q_Protocol(type));
		return Or (list, genHostOp (id, Q_Dst|Q_Prim(type)
					    |Q_Protocol(type)));

	default:
		punt ("illegal direction in genHost");

	}
}
		
static struct blist
genGateway (id, type)
	union id id;
	int type;
{
	struct blist r;
	union id new;

	switch (Q_Protocol (type)) {

	case Q_Default:
	case Q_Ip:
	case Q_Arp:
	case Q_Rarp:
		new.e = id.g.e;
		r = genHost (new, Q_Or | Q_Ether | Q_Host);
		new.h = id.g.h;
		r = And (r, Not (genHost (new, Q_Or | Q_Protocol (type)
					  | Q_Host)));
		return r;

	default:
		fc_error ("bad modifier to gateway");
	}
}

static struct blist
genProto (id, type)
	union id id;
	int type;
{
	struct blist r;

	switch (Q_Protocol (type)) {

	case Q_Default:
	case Q_Ip:
		
		Aux (r.f = new_op (EtherProtoOp)) = ETHERPUP_IPTYPE;
		Aux (r.op = new_op (IpProtoOp)) = id.pr;
		r.t = -1;
		break;

	case Q_Ether:
		Aux (r.op = new_op (EtherProtoOp)) = id.pr;
		r.f = r.t = -1;
		break;
		
	default:
		fc_error ("bad modifier of proto");
	}
	return r;
}

static struct blist
genProtoAbbrev (proto)
	int proto;
{
	union id id;
	int type;

	switch (proto) {
	case Q_Udp:
		/* ip proto udp */
		type = Q_Ip | Q_Proto;
		id.pr = QUAL_TO_PROTO (Q_Udp);
		break;

	case Q_Tcp:
		/* ip proto tcp */
		type = Q_Ip | Q_Proto;
		id.pr = QUAL_TO_PROTO (Q_Tcp);
		break;

	case Q_Ip:
		/* ether proto ip */
		type = Q_Ether | Q_Proto;
		id.pr = QUAL_TO_PROTO (Q_Ip);
		break;

	case Q_Arp:
		/* ether proto arp */
		type = Q_Ether | Q_Proto;
		id.pr = QUAL_TO_PROTO (Q_Arp);
		break;
	case Q_Rarp:
		/* ether proto rarp */
		type = Q_Ether | Q_Proto;
		id.pr = QUAL_TO_PROTO (Q_Rarp);
		break;
	case Q_Ether:
		/* ether */
		return null_list;
	default:
		punt ("bad case in genProtoAbbrev");
	}
	return genCode (id, type);
}	

static struct blist
genCode (id, type)
	union id id;
	int type;
{
	struct blist r;

	switch (Q_Prim (type)) {

	case Q_Default:
	case Q_Net:
	case Q_Host:		r = genHost (id, type); break;

	case Q_Port:		r = genPort (id, type); break;
	case Q_Gateway:		r = genGateway (id, type); break;
	case Q_Proto:		r = genProto (id, type); break;

	default:
		punt ("illegal primary in genCode");
	}

	return r;
}

static int
genOpcode (type)
	int type;
{
	int dir = (Q_Dir (type) >> 8) - 1;

	switch (Q_Prim (type)) {

	case Q_Net:
		dir += 2;
	case Q_Default:
	case Q_Host:
		switch (Q_Protocol (type)) {

		case Q_Ether:	return dir + ESrcOp;
		case Q_Ip:	return dir + IpSrcOp;
		case Q_Arp:	return dir + ArpSrcOp;
		case Q_Rarp:	return dir + RarpSrcOp;
		default:	break;
		}

	case Q_Port:
		switch (Q_Protocol (type)) {

		case Q_Udp:	return dir + UdpSrcPortOp;
		case Q_Tcp:	return dir + TcpSrcPortOp;
		default:	break;
		}
	case Q_Proto:
		switch (Q_Protocol (type)) {

		case Q_Ether:	return EtherProtoOp;
		case Q_Ip:	return IpProtoOp;
		default:	break;
		}
	default:
		break;
	}
	return UndefOp;
}

static int
genLoad (protocol, reg, size)
	int protocol, reg, size;
{
	int index;
	int opcode;

#ifdef DEBUG
	if (protocol < Q_Ether || protocol > Q_Udp)
		punt ("bad protocol passed to genLoad");
#endif
	opcode = (protocol - Q_Ether) >> 4;
	opcode += LoadEtherOp + (reg * N_PROTOCOLS);

	index = new_op (opcode);
	Aux (index) = size;

	return reg;
}	

static struct blist
genRelOp (opcode, r1, r2)
	int opcode, r1, r2;
{
	struct blist result;
	int index = new_op (opcode + reg_pair_to_offset (r1, r2));

	result.op = index;
	result.t = result.f = -1;

	free_reg (r1);
	free_reg (r2);

	return result;
}

static int
genLoadImm (val)
	int val;
{
	int reg = alloc_reg ();
	int index = new_op (LoadImmOp + reg);

	Data (index, 0) = val;
	return reg;
}

static int
genArithOp (opcode, dst, src)
	int opcode, dst, src;
{
	new_op (opcode + reg_pair_to_offset (dst, src));

	free_reg (src);

	return dst;
}

/*
 * Here we handle simple allocation of 3 registers.
 * If more than 3 registers are alloc'd, the allocator punts.
 */
#define N_REGISTERS 3
static int regnum[N_REGISTERS];

/*
 * Must be called before alloc_reg.
 */
static void
init_reg ()
{
	int i;

	for (i = 0; i < N_REGISTERS; i += 1)
		regnum[i] = i;
}

/*
 * Index into register table.
 */
static int next_reg = 0;

/*
 * Return the next free register.
 */
static int
alloc_reg ()
{
	int n;

	if (next_reg == N_REGISTERS)
		punt ("out of registers");

	n = regnum[next_reg++];

	return n;
}

/*
 * Return a register to the table so it can
 * be used later.
 */
static void
free_reg (n)
	int n;
{
	regnum[--next_reg] = n;
}

/*
 * Translates source and destination registers into an opcode offset.
 * The opcodes are numbered so that this offset can be added to the
 * base opcode to get opcode that corresponds to the appropriate registers.
 */
static int
reg_pair_to_offset (dst, src)
	int dst, src;
{
	static int tab[3][3] = {
		{ -1, 0, 1 },
		{ 2, -1, 3 },
		{ 4, 5, -1 }
	};

	int reg;

#ifdef DEBUG
	if (src > N_REGISTERS || dst > N_REGISTERS)
		abort ();
#endif
	reg = tab[dst][src];
#ifdef DEBUG
	if (reg < 0)
		abort ();
#endif
	return reg;
}

/*
 * amerge	Merge backlists that form arithmetic groups.  This are
 *		special since they may contain an op = -1 (i.e. no 
 *		jump insns).  The reason we have jumps at all is
 *		because the packet accessors generate protocol checks.
 */
static struct blist
amerge (a, b)
	struct blist a, b;
{
	struct blist r;

	if (b.op >= 0) {
		/*
		 * B has at least one jump.  Let the result op be B's op,
		 * and merge the false lists, along with A's op.
		 * Do nothing for the true lists since these must always
		 * be null.
		 */
		r.op = b.op;
		Jump (a.op) = a.f;
		r.f = merge (a.op, b.f);
		r.t = -1;
	}
	else
		/* B is null so just return A. */
		r = a;

	return r;
}
