#include "common.h"
#include <sys/file.h>
#include <pwd.h>

/*
 * SubServer main module.
 */

#define	RCFILE	"/.services"	/* File to grab services from */

char *getenv();

/*
 * This linked list structure is used to keep track of the
 * services we're offering.
 */
struct service {
	char	name[20];		/* Service name */
	char	path[MAXPATHLEN];	/* Service path */
	struct service *next;		/* Next element... */
} *list;

/*
 * Read in a services file and set up the linked list of
 * services.  Test each service to be sure we can offer it.
 * Returns 0 if there are no services offered.
 */
getservices()
{
	FILE	*fp;
	char	filename[MAXPATHLEN], *home;

	home = getenv("HOME");
	if (! home)
	{
		fprintf(stderr, "no home directory\n");
		return(0);
	}

	strcpy(filename, home);
	strcat(filename, RCFILE);

	fp = fopen(filename, "r");
	if (! fp)
	{
		perror("couldn't open services file");
		return(0);
	}

	list = NULL;

	while (! feof(fp))
	{
		char	servname[20], format[20];

		servname[0] = 0;
		sprintf(format, "%%20s\t%%%d[^\n]", MAXPATHLEN);
		fscanf(fp, format, servname, filename);
		getc(fp);
		
		if (servname[0] && filename[0])
		{
			struct service *temp;

			if (access(filename, X_OK))
			{
				fprintf(stderr, "warning: can't execute %s\n",
					filename);
				continue;
			}

			temp = (struct service *) malloc(sizeof(struct service));
			strcpy(temp->name, servname);
			strcpy(temp->path, filename);
			temp->next = list;
			list = temp;
		}
	}
	fclose(fp);
	return 1;
}


/*
 * Reap children.
 */
sigchld()
{
	return wait(0);
}

int	fd;

/*
 * Reload the database.
 * We do this by killing the old list element by element then calling
 * getservices(), and closing the socket file descriptor so the main
 * loop will have to reregister our services.
 */
reload()
{
	struct service *cur, *next;

	cur = list;
	while (cur)
	{
		next = cur->next;
		free(cur);
		cur = next;
	}

	if (! getservices())
		exit(-1);
	close(fd);
	fd = -1;	/* prevent another close... */
}

main(argc, argv, envp)
char **argv, **envp;
{
	int thirty;
	struct service *cur;
	struct passwd *pw;
	char	service[80], user[16];

	if (! getservices())
		exit(0);

	if (fork())
		exit(0);

	close(0);
	close(1);
	close(2);

	fd = open("/dev/tty", O_RDWR);
	if (fd >= 0)
	{
		ioctl(fd, TIOCNOTTY, 0);
		close(fd);
	}
	else
		printf("warning: couldn't disassociate from tty\n");

/*
 * Handle signals.  We want to reap children, so we should handle SIGCHLDs;
 * we also want to let the user reload his services database, which we do
 * with SIGHUP.
 */
	signal(SIGCHLD, sigchld);
	signal(SIGHUP, reload);

	thirty = 30;

	pw = getpwuid(getuid());
	if (pw == NULL)
	{
		printf("warning: couldn't get UID\n");
		user[0] = '\n';
		user[1] = '\0';
	}
	else
	{
		strncpy(user, pw->pw_name, sizeof(user)-2);
		user[sizeof(user)-1] = '\0';
		strcat(user, "\n");
	}
	
	while (1)
	{
		char c;

		fd = clientsock("localhost", SUPERPORT);
		if (fd < 0)
			if (errno == ECONNREFUSED)
			{
				start_super(argv[0], envp);
/*
 * Give the superserver time to fire up.
 */
				sleep(5);
				continue;
			}
			else
			{
				perror("superserver connect");
				exit(-1);
			}

		thirty = 30;
		setsockopt(fd, SOL_SOCKET, SO_LINGER, &thirty, sizeof(thirty));
		c = 0;
		do
			if (read(fd, &c, 1) < 0 && errno != EINTR)
				break;
		while (c != '\n');

		if (c != '\n')
			continue;

		write(fd, "REGISTER\n", 9);
		write(fd, user, strlen(user));

		for (cur = list; cur; cur = cur->next)
		{
			write(fd, cur->name, strlen(cur->name));
			write(fd, "\n", 1);
		}
		write(fd, "\n", 1);

		service[0] = 0;
		if (read(fd, service, 20) < 0 && errno != EBADF && errno != EINTR)
		{
			perror("read");
			exit(-1);
		}

		if (service[0])
			do_service(service, fd, envp);
		if (fd >= 0)
			close(fd);
	}
}


/*
 * Provide the service.  Fork off and keep reading parameters until
 * they are terminated by an empty line, then pass them to the program
 * specified by the service.
 */
do_service(service, fd, envp)
char *service;
int fd;
char **envp;
{
	struct service *cur;
	char	*argv[256], input[256];
	int	curarg = 0, index = 0, thirty = 60;

	if (fork())
		return;

	argv[curarg++] = service;

	while (1) {
		read(fd, &input[index], 1);
		if (input[index] == '\r')
			continue;
		if (input[index] == '\n')
		{
			if (! index)
				break;

			argv[curarg] = (char *)malloc(index+1);
			bcopy(input, argv[curarg], index);
			argv[curarg][index] = 0;

			index = 0;
			curarg++;
		}
		else
			index++;
	}

	dup2(fd, 0);
	dup2(fd, 1);
	dup2(fd, 2);
	setsockopt(0, SOL_SOCKET, SO_LINGER, &thirty, sizeof(thirty));
	setsockopt(1, SOL_SOCKET, SO_LINGER, &thirty, sizeof(thirty));
	setsockopt(2, SOL_SOCKET, SO_LINGER, &thirty, sizeof(thirty));
	if (fd > 2)
		close(fd);

	argv[curarg] = NULL;

	for (cur = list; cur; cur = cur->next)
		if (! strcmp(cur->name, service))
			break;

	if (! cur)		/* service not there */
		exit(0);

	execve(cur->path, argv, envp);
	perror("execve");
	exit(0);
}

char *superv[] = { "/bin/sh", "-c", "supersrv", NULL };

start_super(argv0, envp)
char *argv0, **envp;
{
	char *lastslash, argcopy[MAXPATHLEN], *rindex();

	strcpy(argcopy, argv0);
/*
 * If a path was given, try to find the superserver in the
 * same directory as the subserver...
 */
	if (lastslash = rindex(argcopy, '/'))
	{
		char path[MAXPATHLEN];

		*lastslash = 0;
		strcpy(path, argcopy);
		strcat(path, "/supersrv");
		if (! access(path, X_OK))
		{
			if (! fork())
				execve(path, superv+2, envp);
			return;
		}
	}

/*
 * Otherwise, start up a shell to scan along the user's
 * $PATH.
 */
	if (! fork())
		execve(superv[0], superv, envp);
}

