/*
 *	Serial Line interface
 *
 *	Rick Adams
 *	Center for Seismic Studies
 *	1300 N 17th Street, Suite 1450
 *	Arlington, Virginia 22209
 *	(703)276-7900
 *	rick@seismo.ARPA
 *	seismo!rick
 *
 *	Some things done here could obviously be done in a better way,
 *	but they were done this way to minimize the number of files
 *	that had to be changed to accomodate this device.
 *	A lot of this code belongs in the tty driver.
 */

#ifndef lint
static char rcsid[] = "$Header: /ai/car/src/slip/slip-3.x/tty_sl.c,v 1.12 89/05/31 23:49:02 rayan Exp $";
#endif

#include "sl.h"
#if NSL > 0

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/mbuf.h"
#include "../h/buf.h"
#include "../h/protosw.h"
#include "../h/socket.h"
#include "../h/vmmac.h"
#include "../h/ioctl.h"
#include "../h/tty.h"
#include "../h/errno.h"
#include "../h/conf.h"

#include "../net/if.h"
#include "../net/netisr.h"
#include "../net/route.h"
#include "../netinet/in.h"
#include "../netinet/in_systm.h"
#include "../netinet/ip.h"
#include "../netinet/ip_var.h"
#include "../netpup/pup.h"

/* Comment required or this screws up sun's make depend
#ifdef vax
include "../vax/cpu.h"		had to take out the `#' too
include "../vax/mtpr.h"
#endif
*/
#include "../h/kernel.h"

#ifdef sun
#define	t_slsoftc	t_linep
#else	/* upper case on 4.2BSD */
#define t_slsoftc	T_LINEP
#endif

struct sl_stats {
	int	st_badesc;		/* bad character after escape */
	int	st_zerolen;		/* zero length packet received */
	int	st_rqfull;		/* rcvr queue full, packet dropped */
	int	st_rtoobig;		/* long packet received */
	int	st_ipqfull;		/* ip queue full, packet dropped */
	int	st_sndqfull;		/* send queue full, packet dropped */
	int	st_norfreebufs;		/* no buffers on receiver free list */
	int	st_hungbug;		/* restarted from hang */
	int	st_tsched;		/* softints scheduled via SLMAXTFREE */
	int	st_rsched;		/* softints scheduled via SLRFREELOW */
	int	st_memdenied;		/* memory denied by system */
} sl_stats[NSL];

struct sl_softc {
	char	sc_escaped;		/* flags */
	char	sc_isscheduled;		/* 1 when software interrupt scheduled*/
	char	sc_oactive;		/* active or not */
	short	sc_len;			/* length of buffer remaining */
	short	sc_totlen;		/* sanity limit on rcv buff size */
	short	sc_nrfreem;		/* number of free mbufs on rfree list */
	struct mbuf *sc_rfreem;		/* rcv free mbuf list */
	struct sl_stats *sc_stat;	/* pointer to stat structure */
	struct mbuf *sc_m;		/* head of mbuf chain */
	struct mbuf *sc_mt;		/* tail of mbuf chain */
	u_char *sc_mp;			/* ptr to next available buffer char */
	struct mbuf *sc_tfreem;		/* txmt free mbuf list */
	u_short	sc_tfreebcnt;		/* approx. number of bytes on tfree */
	u_short	sc_outbytcnt;		/* number of bytes xmted from current */
	struct mbuf *sc_outm;		/* current mbuf being transmitted */
	struct mbuf *sc_actoutm;	/* head of mbuf chain currently xmted */
	struct	tty *sc_ttyp;		/* pointer to tty structure */
	struct ifqueue sc_rxqueue;	/* receiver packet queue */
	struct	ifnet sc_if;		/* network-visible interface */
}sl_softc[NSL];

#define FRAME_END	 0300			/* Frame End */
#define FRAME_ESCAPE	 0333			/* Frame Esc */
#define TRANS_FRAME_END	 0334			/* transposed frame end */
#define TRANS_FRAME_ESCAPE 0335			/* transposed frame esc */

#define SLTU	1500
#define SLLOWAT	  50
#define SLHIWAT	2000

/*
 * Schedule a software interrupt when:
 * 1) There is about SLMAXTFREE bytes worth of mbufs on the output free list
 * 2) There are fewer than SLRFREELOWAT mbufs on the input free list
 * 3) There is a packet on slrxqueue
 *
 * A software interrupt will free everything on the output free list,
 * top up the input free list to the high water mark and queue any packets
 * on the rx queue to ip.  Since a software interrupt is always scheduled
 * for condition (3), the idea is to make the other limits high enough
 * that they schedule an interrupt only in pathological situations, letting
 * the reception of packets initiate the cleanup under normal circumstances.
 *
 * It is assumed in the following that the software interrupt will occur
 * for sure within 1/2 second of scheduling (i.e. 1000 bytes at 19200 baud).
 */

#define	SLMAXTFREE	(4000)
#define	SLRFREELOWAT	(1000 / MLEN)
#define	SLRFREEHIWAT	(((SLTU + 200 + 1000) / MLEN) + 1)  /* 200 for slop */

int slsoftintr();

#define	SLSOFTSCHED(sc)	\
	{ if ((sc)->sc_isscheduled == 0) { \
		softcall(slsoftintr, (caddr_t)(sc)); \
		(sc)->sc_isscheduled++; } }

#define	SLMFREE(sc, m, bcnt) \
	{ (m)->m_act = (sc)->sc_tfreem; \
	(sc)->sc_tfreem = (m); \
	(sc)->sc_tfreebcnt += (bcnt); \
	if ((sc)->sc_tfreebcnt > SLMAXTFREE) { \
		(sc)->sc_stat->st_tsched++; \
		SLSOFTSCHED(sc); } }

#define	SLMGET(sc, m) \
	{ if ((m) = (sc)->sc_rfreem) { \
		(sc)->sc_rfreem = (m)->m_next; \
		(m)->m_next = 0; \
		(sc)->sc_nrfreem--; \
		if ((sc)->sc_nrfreem < SLRFREELOWAT) { \
			(sc)->sc_stat->st_rsched++; \
			SLSOFTSCHED(sc) } \
	} else { if ((sc)->sc_nrfreem > 0) panic ("slmget"); \
		SLSOFTSCHED(sc); } }

/*
 * The following should be the interrupt priority of the terminal interface.
 */

#define	splsl	spl5

int sloutput(), slioctl();

slopen(dev,tp)
	dev_t dev;
	register struct tty *tp;
{
	register struct sl_softc *sc = sl_softc;
	register int nsl;
	extern int ifqmaxlen;
	struct ifnet *loop_if;

#ifdef lint
	dev = dev;
#endif
	if (tp->t_line == SLIPDISC)
		return EBUSY;

	for(nsl=0;sc->sc_ttyp;){
		if(nsl++ >= NSL)
			return ENOSPC;
		sc++;
	}

	tp->t_slsoftc = (caddr_t)sc;
	sc->sc_ttyp = tp;
	sc->sc_stat = &sl_stats[nsl];
	bzero((caddr_t)&sl_stats[nsl], sizeof(struct sl_stats));
	sc->sc_if.if_unit = nsl;
	sc->sc_if.if_mtu = SLTU;
	sc->sc_if.if_output = sloutput;
	sc->sc_if.if_ioctl = slioctl;
	sc->sc_if.if_flags = IFF_POINTOPOINT;
	sc->sc_if.if_snd.ifq_maxlen = ifqmaxlen;
	sc->sc_rxqueue.ifq_maxlen = ifqmaxlen;
	slsoftintr(sc);			/* initialize rfree list */
	if(sc->sc_if.if_name)
		/* already previously attached */
		return 0;
	sc->sc_if.if_name = "sl";
	if_attach(&sc->sc_if);

	/* This is a real kludge. Since we do the attach in the rc file,
	   the loopback driver gets configured first. Now Berkeley, in its
	   infinite wisdom, when connecting to a machine through a gateway,
	   chooses the FIRST interface on its list. Guess what that is when you
	   don't have an ethernet? That's right! It's the loopback driver!
	   The following crock is an attempt to get around that.
	 */

	if ( ifnet && ifnet->if_name[0] == 'l' && ifnet->if_name[1] == 'o'
		&& ifnet->if_name[2] == '\0') {
		/*
		 * The loopback device is configured in first: detach and
		 * re-attach it to move it to the end of the linked list.
		 */
		loop_if = ifnet;
		ifnet = loop_if->if_next;
		loop_if->if_next = ifnet->if_next;
		ifnet->if_next = loop_if;
	}

	return 0;
}

/*ARGSUSED*/
sltioctl(tp, cmd, data, flag)
	struct tty *tp;
	caddr_t data;
{
	register struct sl_softc *sc = (struct sl_softc *)tp->t_slsoftc;

	/*
	 * This is done this way, because we want these atributes to
	 * be present AFTER theuser program closes the line.
	 *
	 */
	if (cmd == TIOCGETD){
		*(int *)data = (int)sc->sc_if.if_unit;
		tp->t_state &= ~TS_HUPCLS;
		tp->t_flags |= NOHANG;
		return 0;
	}
	if(cmd == TIOCSDTR || cmd == TIOCCDTR)
		return(-1);
	return EINVAL;
}

/*
 * Always called from above at low priority.
 */
sloutput(ifp, m, dst)
	register struct ifnet *ifp;
	register struct mbuf *m;
	struct sockaddr *dst;
{
	register struct sl_softc *sc = &sl_softc[ifp->if_unit];
	int s, error = 0;

	if (dst->sa_family != AF_INET) {
		printf("sl%d: af%d not supported\n", ifp->if_unit,
			dst->sa_family);
		m_freem(m);
		return (EAFNOSUPPORT);
	}
	s = splsl();
	if (IF_QFULL(&ifp->if_snd)) {
		IF_DROP(&ifp->if_snd);
		(void) splx(s);
		m_freem(m);
		sc->sc_if.if_oerrors++;
		sc->sc_stat->st_sndqfull++;
		error = ENOBUFS;
		(void) splsl();
		if (sc->sc_oactive && sc->sc_ttyp &&
				(sc->sc_ttyp->t_state & TS_BUSY) == 0) {
			/*
			 * There seems to be a bug somewhere whereby
			 * we end up thinking that we are busy transmitting
			 * while the terminal interface is not doing
			 * anything.  Make us agree with the terminal
			 * interface.  It would be better if someone
			 * figured the real bug out.
			 */
			printf("sl%d: output hung, restarting\n", sc->sc_if.if_unit);
			sc->sc_stat->st_hungbug++;
			sc->sc_oactive = 0;
		}
	} else {
		IF_ENQUEUE(&ifp->if_snd, m);
	}
	if (sc->sc_oactive == 0 && sc->sc_ttyp) {
		sc->sc_oactive = 1;
		slstart(sc->sc_ttyp);
	}
	splx(s);
	return (error);
}

/*
 * Start output on interface.  Get another datagram
 * to send from the interface queue and map it to
 * the interface before starting output.
 * Called from sloutput() and also from below as line-specific
 * (re)start routine.
 *
 * Always called at the interrupt priority of the terminal
 * driver.
 */
slstart(tp)
	register struct tty *tp;
{
	register struct sl_softc *sc = (struct sl_softc *)tp->t_slsoftc;

	if(tp->t_outq.c_cc < SLLOWAT && (sc->sc_outm || sc->sc_if.if_snd.ifq_len)) {
		register struct mbuf *m;

		m = sc->sc_outm;
		while(tp->t_outq.c_cc < SLHIWAT) {
			register u_char *cp;
			register n;
			int res;

			if(m == 0) {
				IF_DEQUEUE(&sc->sc_if.if_snd, m);
				if (m == 0) {
					sc->sc_oactive = 0;
					break;
				}
				/* save beginning of chain */
				sc->sc_actoutm = m;
				sc->sc_outbytcnt = m->m_len;
			}

			n = m->m_len;
			cp = mtod(m, u_char *);
			while(n > 0) {
				if(*cp==FRAME_END || *cp==FRAME_ESCAPE)
					break;
				cp++;
				n--;
			}

			res = b_to_q(mtod(m, caddr_t), m->m_len-n, &tp->t_outq);
			if(res) {
				m->m_off += m->m_len-n-res;
				m->m_len = n+res;
				break;
			}

			if(n) {
				m->m_off += m->m_len-n;
				m->m_len = n;
				if(putc(FRAME_ESCAPE, &tp->t_outq))
					break;
				if(putc(*cp==FRAME_ESCAPE ? TRANS_FRAME_ESCAPE: TRANS_FRAME_END, &tp->t_outq)) {
					(void)unputc(&tp->t_outq);
					break;
				}
				if(--m->m_len > 0)
					/*
					 * Careful not to bump data pointer
					 * into next MBCLUSTER page, otherwise
					 * we would free the wrong one.
					 */
					m->m_off++;
				continue;
			}
			if(m->m_next == 0) {
				if(putc(FRAME_END, &tp->t_outq)) {
					/* Will get it next time 'round */
					m->m_len = 0;
					break;
				}
				sc->sc_if.if_opackets++;
				/*
 				 * Free the entire chain
				 */
				SLMFREE(sc, sc->sc_actoutm, sc->sc_outbytcnt);
				sc->sc_actoutm = m = 0;
			} else {
				m = m->m_next;
				sc->sc_outbytcnt += m->m_len;
			}
		}
		sc->sc_outm = m;
	}

	if(tp->t_oproc)
		(*tp->t_oproc)(tp);
}

/*
 * tty interface receiver interrupt.
 *
 * Called at ipl of device.
 */
slinput(c, tp)
register c;
register struct tty *tp;
{
	register struct sl_softc *sc;
	int s;

	sc = (struct sl_softc *)tp->t_slsoftc;
	if ( sc == NULL ){
		printf("t_slsoftc NULL\n");
		return;
	}
	c &= 0xff;
	if (sc->sc_escaped) {
		sc->sc_escaped = 0;
		switch(c) {
		case TRANS_FRAME_ESCAPE:
			c = FRAME_ESCAPE;
			break;
		case TRANS_FRAME_END:
			c = FRAME_END;
			break;
		default:
			sc->sc_if.if_ierrors++;
			sc->sc_stat->st_badesc++;
			return;
		}
	} else {
		switch(c) {
		case FRAME_END:
			if (sc->sc_mt == 0) {
				sc->sc_if.if_ierrors++;
				sc->sc_stat->st_zerolen++;
				return;
			}
			sc->sc_if.if_ipackets++;
			sc->sc_mt->m_len -= sc->sc_len;

			if (IF_QFULL(&sc->sc_rxqueue)) {
				IF_DROP(&sc->sc_rxqueue);
				sc->sc_if.if_ierrors++;
				sc->sc_stat->st_rqfull++;
				slrxdrop(sc);
			} else {
				IF_ENQUEUE(&sc->sc_rxqueue, sc->sc_m);
				SLSOFTSCHED(sc);
				sc->sc_mt = 0;
				sc->sc_len = 0;
				sc->sc_totlen = 0;
			}
			return;
		case FRAME_ESCAPE:
			sc->sc_escaped = 1;
			return;
		}
	}
	if (sc->sc_len <= 0){	/* have to get more buffer space */
		struct mbuf *mm;
		SLMGET(sc, mm);
		if (mm == 0){
			slrxdrop(sc);
			sc->sc_if.if_collisions++;
			sc->sc_stat->st_norfreebufs++;
			return;
		}
		if (sc->sc_mt == 0){
			sc->sc_mt = sc->sc_m = mm;
		} else {
			sc->sc_mt->m_next = mm;
			sc->sc_mt = mm;
		}
		sc->sc_len = mm->m_len = MLEN;
		sc->sc_mp = mtod(mm,u_char *);
	}

	*sc->sc_mp++ = c;
	sc->sc_len--;
	if(++sc->sc_totlen > SLTU*2) {
		slrxdrop(sc);
		sc->sc_if.if_ierrors++;
		sc->sc_stat->st_rtoobig++;
		printf("sl%d: rcvd pckt too big\n", sc->sc_if.if_unit);
	}
}

/*
 * Free the current rcvr mbuf chain.
 *
 * It would be much cleaner if SLMFREE were used to free the
 * mbufs to the tfreem list.  Instead the mbufs are put back on
 * the rfreem list from whence they came.  This requires some
 * illicit knowledge of the workings of MGET and MFREE, but is
 * more efficient, I guess.
 */
slrxdrop(sc)
register struct sl_softc *sc;
{
	register struct mbuf *m;
	register int n;

	if (sc->sc_mt) {
		m = sc->sc_m;
		n = 0;
		do {
			m->m_off = MMINOFF;
			m->m_act = 0;
			n++;
			m = m->m_next;
		} while (m);
		sc->sc_mt->m_next = sc->sc_rfreem;
		sc->sc_rfreem = sc->sc_m;
		sc->sc_nrfreem += n;
	}
	sc->sc_mt = 0;
	sc->sc_len = 0;
	sc->sc_totlen = 0;
}


/*
 * Software interrupt routine.
 *
 * Clean out the tfreem list, allocate more buffers for the
 * rfreem list, transfer packets from the rcvr queue to
 * the ipintrq and call ip.
 */
slsoftintr(sc)
register struct sl_softc *sc;
{
	register struct mbuf *m;
	int callip = 0;
	int s;

	/*
	 * Clean up after transmitter.
	 */
	s = splsl();
	sc->sc_tfreebcnt = 0;
	do {
		if (m = sc->sc_tfreem) {
			sc->sc_tfreem = m->m_act;
			m->m_act = 0;
		}

		(void) splx(s);
		if (m)
			m_freem(m);
		(void) splsl();

	} while (m);
	(void) splx(s);

	/*
	 * Top up the free mbuf list for the receiver.
	 */
	while (sc->sc_nrfreem < SLRFREEHIWAT) {
		MGET(m, M_DONTWAIT, MT_DATA);
		if (m == 0) {
			sc->sc_stat->st_memdenied++;
			break;
		}
		(void) splsl();
		m->m_next = sc->sc_rfreem;
		sc->sc_rfreem = m;
		sc->sc_nrfreem++;
		(void) splx(s);
	}

	/*
	 * Reset scheduled flag
	 */
	sc->sc_isscheduled = 0;

	/*
	 * Move received packets from queue at splsl()
	 * to ip interrupt queue at splimp()
	 */
	for (;;) {
		(void) splsl();
		IF_DEQUEUE(&sc->sc_rxqueue, m);
		(void) splx(s);

		if (m == 0)
			break;
		callip = 1;

		(void) splimp();
		if (IF_QFULL(&ipintrq)) {
			IF_DROP(&ipintrq);
			sc->sc_if.if_ierrors++;
			sc->sc_stat->st_ipqfull++;
			m_freem(m);
		}
		else
			IF_ENQUEUE(&ipintrq, m);
		(void) splx(s);
	}

	/*
	 * If something was put on the ip queue, call ip.
	 */
	if (callip)
		ipintr();
}


/*
 * Process an ioctl request.
 */
slioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	struct tty *tp;
	struct sockaddr_in *sin;
	struct rtentry route;
	int s = splimp(), error = 0;


	switch (cmd) {

	case SIOCSIFADDR:
		tp = sl_softc[ifp->if_unit].sc_ttyp;
		if(tp == 0) {
			error = EINVAL;
			break;
		}
		tp->t_line = SLIPDISC;
		tp->t_flags = RAW|EVENP|ODDP;
		(*cdevsw[major(tp->t_dev)].d_ioctl)(tp->t_dev, TIOCSDTR, 0, 0);
		if (ifp->if_flags & IFF_RUNNING) {
			bzero((caddr_t)&route, sizeof (route));
			route.rt_dst = ifp->if_dstaddr;
			route.rt_gateway = ifp->if_addr;
			route.rt_flags = RTF_HOST | RTF_UP;
			(void) rtrequest((int) SIOCDELRT, &route);
		}
		ifp->if_addr = *(struct sockaddr *)data;
		sin = (struct sockaddr_in *)&ifp->if_dstaddr;
		ifp->if_net = in_netof(sin->sin_addr);
		ifp->if_flags |= IFF_UP | IFF_RUNNING;
		/* set up routing table entry */
		rtinit(&ifp->if_dstaddr, &ifp->if_addr, RTF_HOST|RTF_UP);
		ifp->if_flags |= IFF_ROUTE;
		break;

#ifndef sun
	case SIOCSIFDSTADDR:
		ifp->if_dstaddr = ((struct ifreq *)data)->ifr_dstaddr;
		break;
#endif sun

#ifdef SIOCIFDETACH
	case SIOCIFDETACH:
		{
			register struct sl_softc *sc;
			register struct mbuf *m;
			int s;

			if(!suser()) {
				error = u.u_error;
				break;
			}
			bzero((caddr_t)&route, sizeof (route));
			route.rt_dst = ifp->if_dstaddr;
			route.rt_gateway = ifp->if_addr;
			route.rt_flags = RTF_HOST | RTF_UP;
			(void) rtrequest((int) SIOCDELRT, &route);
			sc = &sl_softc[ifp->if_unit];
			tp = sc->sc_ttyp;
			if(tp) {
				tp->t_slsoftc = 0;
				sc->sc_ttyp = 0;
				ttyclose(tp);
			}

			/*
			 * Free up memory holdings (note tty is off)
			 */
			if (sc->sc_mt) {
				m_freem(sc->sc_m);
				sc->sc_mt = 0;
				sc->sc_len = sc->sc_totlen = 0;
			}
			if (sc->sc_actoutm) {
				m_freem(sc->sc_actoutm);
				sc->sc_actoutm = sc->sc_outm = 0;
			}
			sc->sc_tfreebcnt = 0;
			while (m = sc->sc_tfreem) {
				sc->sc_tfreem = m->m_act;
				m->m_act = 0;
				m_freem(m);
			}
			sc->sc_nrfreem = 0;
			if (m = sc->sc_rfreem) {
				m_freem(m);
				sc->sc_rfreem = 0;
			}
			do {
				IF_DEQUEUE(&sc->sc_rxqueue, m);
				if (m)
					m_freem(m);
			} while (m);
			s = splimp();
			do {
				IF_DEQUEUE(&ifp->if_snd, m);
				if (m)
					m_freem(m);
			} while (m);
			(void)splx(s);
		}
		break;
#endif

	default:
		error = EINVAL;
	}
	splx(s);
	return (error);
}
#endif
