/* top level common memory manager */

/*

this is a server process that manages common memory - or at least what
behaves like common memory.  It is actually quite simple.  For each object
that we are keeping track of we allocate a piece of memory via declare().
We write the object with writevalue() and read it with readvalue()

Initially, we sit idle waiting for messages.  Each message is processed
as follows:

forever {
	for each message in queue
		for each component in message
			declares are created or confirmed
			reads cause a new message to be built
			writes cause processes to be tagged for wakeup.
	acknowledge every message, by sending back a list of variable
	values that the process is listed as a reader which have been written
	
	for each process scheduled for wakeup {
	    and not having an outstanding read in the queue {
		create a message with all its "read" variables
		if (message built) send it, delete it
	    }
	}
}

*/

#include <stdio.h>
#include <sys/types.h>

#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "inet.h"

#include "cm_constants.h"
#include "cm_var.h"
#include "cm_sd.h"
#include "cm_slot.h"
#include "cm_msg.h"
#include "cm_man.h"
#include "cm_time.h"

#include <setjmp.h>
#include <signal.h>

struct process processes[CM_MAXPROCESSES];
struct variable variables[CM_MAXVARIABLES];
cm_value cm_variable_names = {0,0,0,1};

int connection_socket;

static struct msg *cm_imsg;
struct msg *cm_omsg;	/* shared with man_get_slot.c .. ugh! */

struct slot *nextslot();

#define CM_WRITE_TIMEOUT	5	/* after this many seconds, we give */
					/* up writing to a process, and go */
					/* on.  We will eventually retry. */
static int timeout = CM_WRITE_TIMEOUT;
jmp_buf write_context;	/* context when we are about to call write() */
void sigpipe_handler();
void sigalrm_handler();

char *malloc();

main(argc,argv)
int argc;
char **argv;
{
	init(argc,argv);
	while (TRUE) {
	    process_msg();
	}
}

process_msg()
{
	struct slot *s;
	int clientfd;
	int slots;	/* number of slots in message */

	int rc;

	clientfd = man_get_msg(cm_imsg);
	init_msg(cm_omsg);
	if (cm_imsg->version == CMM_VERSION) {
		slots = cm_imsg->slots;
		s = cm_imsg->data;
		while (TRUE) {
			if (s == NULL || slots == 0) break;
			if (0 > (rc = man_decode_slot(s,clientfd))) {
				printf("process_msg: error trying to decode slot <%d>...aborting msg\n",rc);
				break;
			}
			s = nextslot(cm_imsg,s);
			slots--;
		}
	} else {
		char error_message[80];

		if (cm_imsg->version > CMM_VERSION)
		sprintf(error_message,"cm library (v%d) is %s than cmm (v%d)",
			cm_imsg->version,
			((cm_imsg->version > CMM_VERSION)?"newer":"older"),
			CMM_VERSION);
		put_slot_error(cm_omsg,"version",0,error_message);
	}


	/* if a slot produced an explicit message (like an error), send it. */
	/* otherwise assume he just wants normal variable updates */
	/* Also, respond immediately to anyone waiting on an immediate
	/* response. */
	if ((cm_omsg->slots > 0) || cm_imsg->read_wait) {
		add_read_vars_to_msg(cm_omsg,clientfd);
		send_msg_to(cm_omsg,clientfd);
		processes[clientfd].wakeup = FALSE;
	}
/* why is this necessary, if you figure it out, comment this! */
/*	else processes[clientfd].wakeup = TRUE;*/
	wakeup();	/* any processes necessary */
}

wakeup()
{
    int i=0;

    eprintf(3,"processes to wake:\n");
    for (i=0;i<CM_MAXPROCESSES;i++) {	/* step through all processes */
	if (processes[i].wakeup) {
		eprintf(3," %s",processes[i].name);
		init_msg(cm_omsg);
		add_read_vars_to_msg(cm_omsg,i);
		if (cm_omsg->slots > 0) send_msg_to(cm_omsg,i);
		processes[i].wakeup = FALSE;
	}
    }
    eprintf(3,"\n");
}

extern char *optarg;	/* pointer to start of option argument */
extern int optind;	/* index of argv of next argument to be processed */

/* initialize the process and variable structures */
init(argc,argv)
int argc;
char **argv;
{
	int i;
	int debug_level = 0;
	int c;
	int port = CM_PORT;
	extern int optind;

	while (EOF != (c = getopt(argc,argv,"d:p:t:"))) {
		switch (c) {
		case 'd':
			debug_level = atoi(optarg);
			break;
		case 'p':
			port = atoi(optarg);
			break;
		case 't':	/* timeout */
			timeout = atoi(optarg);
			break;
		}
	}

	set_cm_debug_level(debug_level);
	set_cm_process_name(CM_MANAGER_NAME);

	eprintf(1,"cmm version %d\n",CMM_VERSION);
	eprintf(1,"timeout on writes = %d\n",timeout);

	if (-1 == (connection_socket = initport(PORT_NUMBER(port),SERVER,SOCK_STREAM,
		(char *)0))) {
		fprintf(stderr,"failed to initialize connection socket\n");
		exit(-1);
	}
	cm_time_init();

	if (!(cm_imsg = (struct msg *)malloc(CM_MSGSIZE))) {
		fprintf(stderr,"init: failed malloc(imsg)\n");
		exit(-1);
	}
	if (!(cm_omsg = (struct msg *)malloc(CM_MSGSIZE))) {
		fprintf(stderr,"init: failed malloc(omsg)\n");
		exit(-1);
	}

	for (i=0;i<CM_MAXPROCESSES;i++) processes[i].inuse = FALSE;

	for (i=0;i<CM_MAXVARIABLES;i++) {
		cm_sd_clear(&variables[i].data);
		variables[i].readers = 0;
		variables[i].writers = 0;
		variables[i].xwriter = 0;
		variables[i].count = 0;
	}

#if 0
	/* grab a variable for maintaining the names of all other variables */
	variables[0].writers = 1;
	variables[0].xwriter = connection_socket;	/* reserved to us */

	/* stick in the first value */
	variables[0].count = 1;
	cm_variable_names.size = cm_variable_name.msize =
		strlen("cm_variable_names"+1);
	cm_variable_names.data = new_string("cm_variable_names");
#endif

	signal(SIGPIPE,sigpipe_handler);
	signal(SIGALRM,sigalrm_handler);
}

/* if process died while we were writing to it, abort the write() */
void sigpipe_handler()
{
	longjmp(write_context,SIGPIPE);	/* abort the write() */
}

/* if process hangs and we write to it enough, eventually we will block */
/* abort the write(), when the timer goes off */
void sigalrm_handler()
{
	longjmp(write_context,SIGALRM);
}

struct variable *
get_variable(name)
char *name;
{
	int i;
	int j;
	int free_var = -1;

	for (i=0;i<CM_MAXVARIABLES;i++) {
		if (!var_inuse(&variables[i])) {
			free_var = i;
		} else if (!strcmp(variables[i].name,name)) {
			return(&variables[i]);
		}
	}

	if (free_var == -1) {
		/* no old definition and no space for a new one! */
		/* this is really an exceptional condition, so let's */
		/* print out some helpful info */

		fprintf(stderr,"get_variable(%s) failed\n",name);
		for (i=0;i<CM_MAXVARIABLES;i++) {
			printf("%d: ",i);
			if (var_inuse(&variables[i])) {
				printf("<%s>",variables[i].name);
				printf(" r=%d  w=%d\n  inuse=%d",
					variables[i].readers,
					variables[i].writers,
					var_inuse(&variables[i]));
			} else printf("<not in use>\n");
		}
		return(NULL);
	}


	strcpy(variables[free_var].name,name);
	variables[free_var].count = 0;
	time_now(&variables[free_var].timestamp);
	variables[free_var].command_association = 0;
	variables[free_var].xwriter = CM_NULL_PROCESS;
	for (j=0;j<CM_MAXPROCESSES;j++) {
		variables[free_var].role[j].reader = FALSE;
		variables[free_var].role[j].writer = FALSE;
		variables[free_var].role[j].wakeup = FALSE;
		variables[free_var].role[j].new = FALSE;
	}
	return(&variables[free_var]);
}

send_msg_to(m,to)
struct msg *m;
int to;
{
	int rc;

	switch (rc = setjmp(write_context)) {
	case 0:
		/* assume write() succeeds */
		alarm(timeout);
		sized_write(to,(char *)m,m->size);
		alarm(0);
		print_msg(m);
		break;
	case SIGPIPE:
		alarm(0);
		/* if write() interrupted due to SIGPIPE */
		eprintf(3,"broken pipe\n");
		eprintf(3,"detected death of %s on fd %d\n",
			(processes[to].inuse?
				processes[to].name:"unknown"),
				to);
		kill_client(to);
		break;
	case SIGALRM:
		fprintf(stderr,
			"process %s is being antisocial on fd %d\n",
			(processes[to].inuse?
				processes[to].name:"unknown"),
				to);
		break;
	default:
		alarm(0);
		printf("setjmp(write_context) = %d?!\n",rc);
		abort();	/* should never happen */
	}
}

/* The philosophy coded here is that the cmm always responds to a msg
with the data values that are marked for read and have changed since the
last time it was read */
add_read_vars_to_msg(m,pin)
struct msg *m;
int pin;
{
	struct variable *v;

	for (v=variables;v<&variables[CM_MAXVARIABLES];v++) {
		if (!var_inuse(v)) continue;
		eprintf(5,"is_reader(%s,%s)=%d  is_new(%s,%s)=%d\n",
			processes[pin].name,v->name,is_reader(pin,v),
			processes[pin].name,v->name,is_new(pin,v));
		if (is_reader(pin,v) && is_new(pin,v)) {
                    put_slot_read_response(m,v->name,
			v->count,&v->timestamp,v->command_association,&v->data);
		    unset_new(pin,v);
		}
	}
}

static int clientfds = 0;	/* bitmap of sockets to clients */

int	/* returns file descriptor of client */
man_get_msg(msg)
struct msg *msg;
{
    int clientfd;
    int msg_length = 0;


    while (msg_length <= 0) {
	eprintf(3,"known fds = %x\n",clientfds);
	clientfd = select_server_stream(connection_socket,&clientfds);
	msg_length = sized_read(clientfd,(char *)msg,CM_MSGSIZE);
	eprintf(3,"received message from %s.  fd = %d length = %d\n",
				(processes[clientfd].inuse?
					processes[clientfd].name:"new user"),
			clientfd,
			msg_length);
	print_msg(msg);
	if (msg_length < 0) {
		eprintf(3,"detected death of %s on fd %d\n",
				(processes[clientfd].inuse?
					processes[clientfd].name:"unknown"),
					clientfd);
		kill_client(clientfd);
	}
    }

    if (!processes[clientfd].inuse) {
	processes[clientfd].inuse = TRUE;
	processes[clientfd].wakeup = FALSE;
	strcpy(processes[clientfd].name,msg->name);
    }
    eprintf(2,"received msg from %s on fd %d\n",msg->name,clientfd);

    return(clientfd);
}

/* delete any info we had about this client */
kill_client(fd)
int fd;
{
	struct variable *v;

	processes[fd].inuse = FALSE;
	close(fd);
	eprintf(3,"known fds = %x (before kill_client)\n",clientfds);
/*new*/	clientfds &= ~(1<<fd);
	eprintf(3,"known fds = %x (after kill_client)\n",clientfds);

	for (v=variables;v<&variables[CM_MAXVARIABLES];v++) {
		if (!var_inuse(v)) continue;
		unset_reader(fd,v);
		unset_writer(fd,v);
		unset_wakeup(fd,v);
		/* note that the above calls change the value of var_inuse */
		if (var_inuse(v)) continue;
		/* var had been set, free up any space taken by it */
		if (v->count) {
			cm_sd_free(&v->data);
			v->count = 0;
		}
	}
}

/* flag this variable as having a new value with respect to this process */
set_new(proc,var)
int proc;
struct variable *var;
{
	var->role[proc].new = TRUE;
}

set_reader(proc,var)
int proc;
struct variable *var;
{
	if (!var->role[proc].reader) {
		var->role[proc].reader = TRUE;
		var->readers++;
	}
}

set_nonxwriter(proc,var)
int proc;
struct variable *var;
{
	if (!var->role[proc].writer) {
		var->role[proc].writer = TRUE;
		var->writers++;
	}
}

set_xwriter(proc,var)
int proc;
struct variable *var;
{
	var->xwriter = proc;
	set_nonxwriter(proc,var);
}

set_wakeup(proc,var)
int proc;
struct variable *var;
{
	var->role[proc].wakeup = TRUE;
}

unset_wakeup(proc,var)
int proc;
struct variable *var;
{
	var->role[proc].wakeup = FALSE;
}

unset_new(proc,var)
int proc;
struct variable *var;
{
	var->role[proc].new = FALSE;
}

unset_reader(proc,var)
int proc;
struct variable *var;
{
	if (var->role[proc].reader) {
		var->role[proc].reader = FALSE;
		var->readers--;
	}
}

unset_writer(proc,var)
int proc;
struct variable *var;
{
	if (var->role[proc].writer) {
		if (var->xwriter == proc) var->xwriter = CM_NULL_PROCESS;
		var->role[proc].writer = FALSE;
		var->writers--;
	}
}

var_inuse(var)
struct variable *var;
{
	return(var->readers || var->writers);
}
