/*

These are the user functions uses in manipulating common memory variables.
The primary functions are:

cm_init(process_name,host,debug_level)
cm_declare(name,type,role,period)
cm_undeclare(variable)
cm_sync(wait)
cm_exit()
*/

/* real code begins here */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <strings.h>
#include <sys/file.h>	/* for open(/dev/null) hack */
#include "inet.h"

#include "cm_constants.h"
#include "cm_sd.h"
#include "cm_var.h"
#include "cm_interface.h"
#include "cm_sync.h"
#include "cm_slot.h"
#include "cm_msg.h"
#include "cm_time.h"

extern int errno;

static struct msg *omsg, *imsg;

int server_socket;
int maxfds;

int cm_init();
cm_variable *cm_declare();
cm_variable *firstuservar(), *nextuservar();
cm_write();
struct user_variable *cm_read();
cm_sync();
struct slot *nextslot();
cm_variable *get_variable();

char *malloc();
static composemsg();

/* real definitions begin here */

int
cm_init(name,host,debug_level)
char *name;
char *host;	/* host running cmm.  0 or "" means local host */
int debug_level;
{
	set_cm_debug_level(debug_level);
	set_cm_process_name(name);
	if (0 > (server_socket = initport(PORT_NUMBER(CM_PORT),CLIENT,
			SOCK_STREAM,host))) return(E_CM_INIT_FAILED);

	maxfds = getdtablesize();
	cm_time_init();

	if (!(imsg = (struct msg *)malloc(CM_MSGSIZE))) {
		fprintf(stderr,"init: failed malloc(imsg)\n");
		exit(-1);
	}
	if (!(omsg = (struct msg *)malloc(CM_MSGSIZE))) {
		fprintf(stderr,"init: failed malloc(omsg)\n");
		exit(-1);
	}
	return(0);
}

/* the following call is important if you plan on restarting cm connections
serveral times in one process.  It runs around deallocating memory that has
been allocated by cm_sd_copy.
*/
cm_exit()
{
	cm_variable *v;

	close(server_socket);

	for (v=next_user_variable((struct user_variable *)0);v;
					v=next_user_variable(v)) {
		if (!v->count) continue;	/* never set */
		cm_sd_free(&v->data);
	}
}

cm_variable *
cm_declare(name,role)
char *name;
unsigned int role;		/* READER, WRITER, XWRITER, WAKEUP */
{
	cm_variable *v;

	if (!(v = get_variable(name))) {
	    fprintf(stderr,"cannot get_variable(%s)\n",name);
	    return(NULL);
	}
	if (role) {
	    /* you gave no way to "or out" these bits, you twit! */
	    if (role & CM_ROLE_READER) v->role.reader = TRUE;
	    if (role & CM_ROLE_XWRITER) v->role.xwriter = TRUE;
	    if (role & CM_ROLE_NONXWRITER) v->role.nonxwriter = TRUE;
	    if (role & CM_ROLE_WAKEUP) v->role.wakeup = TRUE;
	}
/*	if (cmd_assoc != 0) v->command_association = cmd_assoc;*/

	v->status.declared = TRUE;
	return(v);
}

cm_undeclare(v)
cm_variable *v;
{
	v->status.undeclared = TRUE;
}

/* returns negative for failure or 0 for success */
cm_sync(wait_opts)
int wait_opts;
{
	int slots;	/* number of slots in incoming message */
	int query;	/* TRUE, if we are requesting an ack from the cmm */
	int rc = 0;
	int cc;
	int selector;
	struct slot *s;
	int processed_msg = FALSE;	/* true if we have received and
					   processed at least one msg in
					   this call to cm_sync */

	if ((0 == message_waiting(server_socket))
	    && (wait_opts & CM_WAIT_READ)) {
		/* no messages waiting, so provoke the cmm to give us some */
		query = TRUE;
	} else query = FALSE;

	composemsg(omsg);
        if (query) omsg->read_wait = TRUE;
	if (omsg->slots || query) {
	    print_msg(omsg);
	    cc = sized_write(server_socket,omsg,omsg->size);
	    if (cc != omsg->size) {
		fprintf(stderr,"%s%s\n","failed to send msg to cmm.  ",
			"cmm disappeared?");
		exit(0);
	    }
	    eprintf(2,"sent msg to server (%d)\n",cc);
	} /* no slots in msg, so no msg not sent to server */
	while (TRUE) {
	    /* poll first */
	    selector = 1<<server_socket;
	    if (-1 == (cc = message_waiting(server_socket))) {
		rc = E_CMM_DIED;
		break;
	    }
	    /* wait rules are as follows:
WAIT:		WAIT_AT_MOST_ONCE		WAIT_FOR_ALL
no msg waiting	select/return			select/return
   msg waiting	select/return			select/select

NO_WAIT:	WAIT_AT_MOST_ONCE		WAIT_FOR_ALL
no msg waiting	return/return			return/return
   msg waiting	select/return			select/select

Where x/y means:
Do x if no messages have been processed in this call to cm_sync, and 
Do y if at least one message has been processed in this call to cm_sync.

For example:
If user specified NO_WAIT|WAIT_AT_MOST_ONCE, no message is waiting and
at least one message has been processed, we return.
*/
	    /* catagorize by the conditions that return, which are
               implemented by "break".
	     */
	    if (wait_opts & CM_NO_WAIT) {
		if (cc == 0) break;	/* no msg waiting */
	    } else if (wait_opts & CM_WAIT_AT_MOST_ONCE) {
		if (processed_msg) break;
	    } else if (processed_msg && (cc == 0)) break;

	    /* That was amazingly simpler to code then to say!!! */
		
	    /* wait until someone responds */
	    selector = 1<<server_socket;
	    if (-1 == select(maxfds,&selector,(int *)0,(int *)0,
	    				(struct timeval *)NULL)) {
		rc = E_CMM_DIED;
		break;
	    } else {
		if (0 >= (cc = sized_read(server_socket,imsg,CM_MSGSIZE))) {
		    rc = E_CMM_DIED;
		    break;
		}
		processed_msg = TRUE;
		s = imsg->data;
		print_msg(imsg);
		if (imsg->version != CMM_VERSION) {
			if (imsg->version > CMM_VERSION) {
				fprintf(stderr,"cm library (v%d) is %s than cmm (v%d)",
					CMM_VERSION,
					((imsg->version > CMM_VERSION)?"older":"newer"),
					imsg->version);
			}
			return(E_CM_WRONG_VERSION);
		}

		slots = imsg->slots;
		s = imsg->data;
		for (;;) {
			if (s == NULL || slots == 0) break;
			if (0 > user_decode_slot(s)) {
			   printf("bad slot encountered...aborting msg \n");
			   break;
			}
			eprintf(6,"slot = %x  ",s);
			eprintf(6,"slot->name %s  ",s->s_name);
			eprintf(6,"slot->size %x\n",s->s_size);

			s = nextslot(imsg,s);
			slots--;
		}
	    }
	}
	return(rc);
}

/* returns -1 if cmm died, 0 if no message waiting, >0 if message waiting */
static int
message_waiting(fd)
int fd;
{
	int selector = 1<<fd;

	return(select(maxfds,&selector,(int *)0,(int *)0,&cm_period_zero));
}

/*
go through our variables looking for ones that we have written or declared.
Stuff them all into a buffer and send this message to the cmm.
By looking at all variables only once, we get rid of duplicates.  I.e. the
user may have written a value more than once before syncing.
*/
static composemsg(m)
struct msg *m;	/* outgoing msg */
{
	int size;
	cm_variable *v;

	init_msg(m);

	for (v=next_user_variable((struct user_variable *)0);v;
					v=next_user_variable(v)) {
		if (v->status.declared) {
			eprintf(6,"adding declare slot for %s\n",v->name);
			size = put_slot_declare(m,v->name,
/*				&v->role,v->command_association); */
				&v->role,0);
			if (size > 0) v->status.declared = FALSE;
		}
		if (v->status.written) {
			eprintf(6,"adding write slot for %s\n",v->name);
			size = put_slot_write(m,v->name,&v->data,
				v->command_association);
			if (size > 0) v->status.written = FALSE;
		}
		if (v->status.undeclared) {
			eprintf(6,"adding undeclare slot for %s\n",v->name);
			size = put_slot_undeclare(m,v->name);
			if (size > 0) {
				cm_sd_free(&v->data);
				v->status.inuse = FALSE;
			}
		}
	}
#if 0
	/* this should be last, because when the manager receives it, */
	/* any current values going into the message queue are zapped. */
	/* This way, the user will not get any duplicate values. */
	if (query) put_slot_read(m);
#endif 
}

cm_print_variable(name)
char *name;
{
	cm_variable *v;

        if (!(v = get_variable(name))) {
            fprintf(stderr,"cannot get_variable(%s)\n",name);
	}

	printf("name = %s\n",v->name);
	printf("role.reader = %d\n",v->role.reader);
	printf("role.nonxwriter = %d\n",v->role.nonxwriter);
	printf("role.wakeup = %d\n",v->role.wakeup);
	printf("role.xwriter = %d\n",v->role.xwriter);
	printf("count = %d\n",v->count);
	printf("cmd assoc = %d\n",v->command_association);
	printf("status.inuse = %d\n",v->status.inuse);
	printf("status.written = %d\n",v->status.written);
	printf("status.declared = %d\n",v->status.declared);
	printf("status.undeclared = %d\n",v->status.undeclared);
}
