/* cm_sd.c 
structured data functions

These functions will be used by the user when doing cm_set_value's

i.e.  cm_set_value(variable,value)

These will also be used when sending/receiving values to/from the cmm.

cm_sd_to_flat(sd,flat)
cm_flat_to_sd(flat,sd)
cm_sd_copy(from,to)
cm_sd_clear(sd)			- clear initially
cm_sd_free(sd)			- release old storage and clear
cm_sd_equal(value1,value2)	- returns true if value1 == value2

*/

#include <stdio.h>
#include <strings.h>
#include "cm_sd.h"
#include "cm_var.h"

#define min(x,y)	(((x)<(y))?(x):(y))

#define TRUE 1
#define FALSE 0

char *malloc();

static int ssresize();

/* return 0 if success, negative if failure */
int
cm_sd_copy(from,to)
cm_value *from;
cm_value *to;
{
	if (from->data == 0) {
		/* can happen if read before set */
		to->size = 0;
		return(0);
	}

	if (0 > ssresize(to,from->size)) return(-1);
	/* if not mallocable, it is possible to->msize < from->size */
	to->size = min(from->size,to->msize);

	/* if user supplied us with 0 size, then to->data may still == 0,
	/* so, don't call safebcopy which checks for zero pointers */
	if (to->data != 0) {
		safebcopy(from->data,to->data,to->size);
	} /* else to->size = 0; */
	return(0);
}

/* convert sd style data to flattened out data - ready for transmission */
/* return size of flattened result */
int
cm_sd_to_flat(sd,f)
struct cm_value *sd;
struct cm_flattened_data *f;
{
	safebcopy(sd->data,f->data,sd->size);
	f->size = sd->size;
	return(sd->size + sizeof(f->size));
}

/* convert transmitted data to sd style data - ready for local storage */
/* return 0 if ok, negative if problems */
int
cm_flat_to_sd(f,sd)
struct cm_flattened_data *f;
cm_value *sd;
{
	if (0 > ssresize(sd,f->size)) return(-1);

	sd->size = min(f->size,sd->msize);

	if (sd->data != NULL) {
		/* if user supplied us with zero-size, then */
		/* sd->...size may still equal zero, so don't call */
		/* safebcopy which may check for zero pointers. */
		safebcopy(f->data,sd->data,sd->size);
	}

	return(0);
}

/* resize that takes short ints */
/* return 0 if resize succeeded or not mallocable */
/* return -1 if resize failed */
static int
ssresize(s,newsize)
cm_value *s;
int newsize;
{
	if (!s->mallocable) return(0);

	eprintf(9,"ssresize(data:%x,old:%x,new:%x)  ",s->data,s->msize,newsize);
	if (s->msize >= newsize) {
	    eprintf(9,"msize >= newsize\n");
	    return(0);
	}
	if (s->data != NULL) {
	    eprintf(9,"free(data)");
	    free(s->data);
	}
	eprintf(9,"  malloc(%x)",newsize);
	if (NULL == (s->data = malloc((unsigned int)newsize))) {
		fprintf(stderr,"resized failed! - out of space\n");
		s->msize = s->size = 0;
		return(-1);
	} else s->msize = newsize;
	return(0);
}

/* zero the various fields in the sd structures */
cm_sd_clear(sd)
cm_value *sd;
{
	sd->msize = 0;
	sd->size = 0;
	sd->data = NULL;
	sd->mallocable = TRUE;
}

cm_sd_free(sd)
cm_value *sd;
{
	if (!sd->mallocable) {
		fprintf(stderr,"cm_sd_free() called on nonmallocable object?\n");
		return;
	}

	free(sd->data);
	cm_sd_clear(sd);
}

int
cm_sd_equal(x,y)
struct cm_value *x, *y;
{
	return(bcmp(x->data,y->data,x->size));
}
