/*

Common memory is local to the CMM process.  It is composed of:

An array of variables.  Each variable is a structure with elements such as
a name, type, value, etc.

Variable values can be stored in the variable structure if they are small
(such as an int) or in a piece of malloc'd memory if they are large (such
as a string).

An array of process descriptors.  Each process descriptor is a structure.
The most important element of this structure is the wakeup field.  If it is
set, the process should be informed of new variable values.
*/

#define CM_MANAGER_NAME		"cm manager"
#define CM_MAXVARIABLES		50
#define CM_MAXPROCESSES		20

#define is_new(proc,var)		(var->role[proc].new)
#define is_reader(proc,var)		(var->role[proc].reader)
#define is_wakeup(proc,var)		(var->role[proc].wakeup)
#define is_writer(proc,var)		(var->role[proc].writer)
#define is_nonxwriter(proc,var)		(var->role[proc].writer && \
					 var->xwriter != proc)
#define is_xwriter(proc,var)		(var->xwriter == proc)

#define CM_NULL_PROCESS		-1

/* process descriptors */
/* this table is indexed by corresponding file descriptors (or sockets) */
struct process {
	char name[CM_PROCESSNAMELENGTH];
	int inuse;
	int wakeup;			/* per process wakeup */
};

/* this is how variables are stored internally to the cmm */
struct variable {
	char name[CM_VARIABLENAMELENGTH];
	struct cm_value data;	/* data (and size, if necessary) */
	unsigned long count;	/* nth definition of this var */
	struct timeval timestamp;	/* when last written */
	int command_association;
	int xwriter;			/* name of exclusive writer */
	int writers;			/* number of (any type of) writers */
	int readers;			/* number of readers */
	struct {
	    unsigned reader : 1;	/* reader */
	    unsigned writer : 1;	/* writer */
	    unsigned wakeup : 1;	/* a-wake me */
	    unsigned new    : 1;	/* changed but not read */
	} role[CM_MAXPROCESSES];
};

/* internal errors specific to CMM */
#define E_GET_VARIABLE_NO_SPACE		-1
