#ifndef lint
static char *RCSid = "$Header: rsend.c,v 1.7 85/12/30 22:54:02 zap Exp $";
static char *Copyright = "rsend.c  Copyright (C) 1985 Svante Lindahl.";
#endif

/*
 * $Log:	rsend.c,v $
 * Revision 1.7  85/12/30  22:54:02  zap
 * Made lint happier. Fixed bug, replacing strcmp with strncmp.
 * 
 * Revision 1.6  85/10/06  19:01:35  zap
 * Bugfix; working directory was "wrong" when shellescape invoked, now
 * does a chdir back after checking things in /dev & /usr/spool/rwho.
 * 
 * Revision 1.5  85/08/21  17:37:12  zap
 * Now chooses to send to writeable tty with least idletime if reciever
 * is logged in more than once on the local host.
 * 
 * Revision 1.4  85/08/19  23:08:56  zap
 * Small change.
 * 
 * Revision 1.3  85/08/19  21:27:51  zap
 * When no host is specified and the reciever isn't logged in on the local
 * host the rwho database is searched to find reciever on an other machine
 * in the local area network. If logged in several times on other hosts the
 * tty (pty) with the least idletime is selected.
 * 
 * Revision 1.2  85/08/17  01:40:41  zap
 * Now checks if reciever is logged in and tty is writeable before entering
 * interactive mode.
 * 
 * Revision 1.1  85/08/16  21:41:09  zap
 * Initial revision
 * 
 * rsend.c    1.0    zap@bogart    1985-07-23 21:29:27
 *
 */

/*
** rsend - send a message to a local or remote user
**
** This program, and any documentation for it, is copyrighted by Svante
** Lindahl. It may be copied for non-commercial use only, provided that
** any and all copyright notices are preserved.
**
** Please report any bugs and/or fixes to:
**
** UUCP: {seismo,mcvax,cernvax,diku,ircam,prlb2,tut,ukc,unido}!enea!ttds!zap
** ARPA: enea!ttds!zap@seismo.ARPA
**   or  Svante_Lindahl_NADA%QZCOM.MAILNET@MIT-MULTICS.ARPA
**
*/

#include "rsend.h"

#define	TIMEOUT    2	                /* seconds */

int	Argc, remotesend, interactive = FALSE ;
char	**Argv, *my_name, *my_tty ;
char    my_cpu[HOST_NAME_SIZE] ;
char	rcv_cpu[HOST_NAME_SIZE] ;
struct	sockaddr_in sin;

char	*getlogin(), *ttyname(), *rindex() ;
struct	hostent *gethostbyname() ;
struct	servent *getservbyname() ;

extern	void err_check () ;
extern	void who_am_i () ;
extern	void who_is_rcvr () ;
extern	void set_up_msg() ;
extern	void set_up_tty() ;
extern	int  where_is_rcvr () ;
extern	void where_to () ;
extern	void check_it () ;
extern	void be_interactive () ;
extern	void do_shell ();
extern	void build_msg () ;
extern	int  build_header () ;
extern	void send_it () ;
extern	void timeout() ;
extern	void error_handle() ;

main (argc, argv)
    int argc ;
    char **argv ;
{
    Argc = argc ;
    Argv = argv ;
    err_check () ;
    who_am_i () ;
    who_is_rcvr () ;
    where_to () ;
    if (interactive) {
        check_it () ;
        be_interactive() ;
    } else {
        build_msg () ;
        send_it () ;
    }
}

void
err_check ()
{
    if (Argc < 2 ) {
        fprintf(stderr,
            "Usage: rsend user[@host] [-tty] [message...]\n") ;
        exit(1) ;
    }
    if (!isatty(0)) {
        fprintf(stderr,
               "rsend: Sorry, standard input must be a tty!\n") ;
        exit(1) ;
    }
}

void
who_am_i ()
{
    struct passwd *pwp ;

    my_name = getlogin() ;
    if (*my_name == NULL)
        if ((pwp = getpwuid(getuid())) == NULL) {
            fprintf(stderr, "You are not for real!.\n") ;
            exit(1) ;
        }
        else
            my_name = (char *)pwp->pw_name ;
    my_tty = rindex(ttyname(0), '/') + 1 ;
    (void) gethostname(my_cpu, sizeof(my_cpu)) ;
}

void
who_is_rcvr ()
{
    char *ptr ;
    int  code ;

    Argc-- ;  Argv++ ;
    *msg.rcv_tty = '\0' ;
    *msg.rcv_name = '\0' ;
    for (ptr = *Argv ; *ptr != '\0' && *ptr != '@' && *ptr != ':' 
        && *ptr != '!' && *ptr != '.' ; ptr++)
            ;
    if (*ptr == '\0') {               /* no host specified */
        set_up_msg(*Argv, my_cpu) ;
        if (*msg.rcv_tty == '\0') {
            if ((code = where_is_rcvr()) != SUCCESS)
                error_handle(code) ;
	}
    } else {
        if (*ptr == '@') {            /* user@host || -tty@host */
            *ptr++ = '\0' ;
            set_up_msg(*Argv, ptr) ;
        } else {                      /* host[.!:]user || host[.!:]-tty */
            *ptr++ = '\0' ;     
            set_up_msg(ptr, *Argv) ;
        }
    }
    remotesend = strcmp(rcv_cpu, my_cpu) ;
    Argc-- ;  Argv++ ;
    if (Argc == 0)                    /* anything left on the commandline? */
        interactive = TRUE ;
    else if (**Argv == '-') {         /* tty-specification? */
        set_up_tty(&Argv[0][1]) ;
        Argc-- ;  Argv++ ;
        if (Argc == 0)                /* still more? */
            interactive = TRUE ;
    }
}

void
set_up_msg(ptr, cpu)
    char *ptr, *cpu;
{
    if (*ptr != '-')                  /* reciever's username or tty? */
        (void) strcpy(msg.rcv_name, ptr) ;
    else
        set_up_tty(++ptr) ;
    (void) strcpy(rcv_cpu, cpu) ;
}

void
set_up_tty(ptr)
    char *ptr ;
{
    if (!strcmp("console", ptr))
        (void) strcpy(msg.rcv_tty, ptr) ;
    else {
        if (strncmp("tty", ptr, 3)) 
            (void) strcpy(msg.rcv_tty, "tty") ;
        (void) strcat(msg.rcv_tty, ptr) ;
    }
    msg.rcv_tty[TTY_SIZE-1] = '\0' ;
}

int
where_is_rcvr()
{
    int    code, f, i, idle = -1, nowd = 0 ;
    char   path[1024], *getwd() ;
    struct stat st ;
    struct utmp ut ;
    long   now ;

    if (getwd(path) == '\0')
        nowd++ ;
    if (getpwnam(msg.rcv_name) == NULL)
        code = NO_USER ;
    else if ((f = open ("/etc/utmp", 0)) < 0)
        code = OP_UTMP ;
    else if (chdir("/dev") < 0)
        code = CHDIR ;
    else {
        code = NOT_IN ;       /* search for reciver in /etc/utmp */
        (void) time(&now) ;
	while (read(f, (char *)&ut, sizeof(ut)) > 0) {
	    if (strncmp (msg.rcv_name, ut.ut_name, 8) == 0) {
		if ((stat(ut.ut_line, &st) >= 0) && (st.st_mode & 2)
		  && (((i = now - st.st_mtime) < idle) || (idle = -1))) {
		    (void) strcpy(msg.rcv_tty, ut.ut_line) ;
		    code = SUCCESS ;
                    idle = i ;
		} else if (code != SUCCESS) { 
                    (void) strcpy(msg.rcv_tty, ut.ut_line) ;
		    code = DENIED ;
		}
	    }
	}
        (void) close(f) ;
    }
#ifdef UNIQUE
    {
        int    cc, n ;
        DIR   *dirp ;
        struct direct *dp ;
        struct whoent *we ;
        struct whod wd ;
        register struct whod *wp = &wd ;

        if (   ((code == NO_USER) || (code == NOT_IN) || (code == CHDIR))
            && (chdir("/usr/spool/rwho") >= 0)
            && ((dirp = opendir(".")) != NULL)) {
            /* check other machines in local network (use rwho database) */
	    (void) time(&now) ;
	    while (dp = readdir (dirp)) {
		if ((dp->d_ino == 0) || (strncmp(dp->d_name, "whod.", 5)))
		    continue ;
		if ((f = open(dp->d_name, 0)) < 0)
		    continue ;
		cc = read(f, (char *)&wd, sizeof(struct whod)) ;
		if ((cc < WHDRSIZE) ||  (now - wp->wd_recvtime > 5 * 60) 
		  || (!strcmp (my_cpu, wp->wd_hostname))) {
		    (void) close (f) ;
		    continue ;
		}
		for(we = wp->wd_we,
                  n = (cc - WHDRSIZE) / sizeof(struct whoent);
		  n > 0; n--, we++) {
		    if (strncmp(msg.rcv_name, we->we_utmp.out_name, 8))
			continue ;
		    if ((i = we->we_idle) < idle || idle == -1) {
			code = SUCCESS ;
			idle = i ;
			(void) strcpy(rcv_cpu, wp->wd_hostname) ;
			(void) strcpy(msg.rcv_tty, we->we_utmp.out_line) ;
		    }
		}
		(void) close (f) ;
	    }
	    if (code == SUCCESS)
		fprintf(stderr, "%s not logged in on %s, sending to %s@%s\n",
		  msg.rcv_name, my_cpu, msg.rcv_name, rcv_cpu) ;
	}
    }
#endif 
    if (!nowd)
        (void) chdir(path) ;
    return(code) ;
}

void
where_to()
{
    struct    hostent *hp ;
    struct    servent *sp ;

    hp = gethostbyname(rcv_cpu) ;      /* get address of reciever's machine */
    if (hp == (struct hostent *) 0 ) {
        fprintf(stderr, "rsend: I don't know a machine named %s\n", rcv_cpu) ;
        exit(1) ;
    }
    bzero(&sin, sizeof (sin)) ;
    bcopy(hp->h_addr, (char *)&sin.sin_addr, hp->h_length) ;
    sin.sin_family = hp->h_addrtype ;
#ifndef	TEST
    sp = getservbyname("rsend", "udp") ;     /* find daemon's port */
    if (sp == NULL) {
        fprintf(stderr, "rsend isn't supported by this machine\n") ;
        exit(1) ;
    }
    sin.sin_port = sp->s_port ;
#else
    if (remotesend) {    /* remote sends not supported when testing */
        fprintf(stderr,
            "Sorry, only local sends are supported at this time\n") ;
        exit(1) ;
    }
    sin.sin_port = htons(RENDEZ_VOUS) ;      /* find daemon's port */
#endif
}

void
check_it()
{
	msg.rqst_type = CHECK ;
	send_it() ;
	msg.rqst_type = SEND ;
}

void
be_interactive()
{
    int hdr_siz, i ;
    char buf[BUF_SIZE] ;

    hdr_siz = build_header () ;   /* standard header for first message */
    msg.text[hdr_siz++] = ' ' ;
    msg.text[hdr_siz] = '\0' ;
    for(ever) {
        printf("rsend>") ;        /* prompt for input */
        (void) fflush(stdout) ;
        if (fgets(buf, sizeof(buf), stdin) != NULL) {
            if ((i = strlen(buf)) > 1) {
                buf[--i] = '\0' ;             /* nullify the \n from fgets */
                if (i == 1 && *buf == '.')    /* '.' to exit pgm */
                    break ;
                else if (*buf == '!') 
                    do_shell(i, buf) ;
                else {
                    if (hdr_siz + i > LINE_LENGTH)  /* separate header if .. */
                        (void) strcat(msg.text, "\r\n") ; /* .. long message */
                    if (hdr_siz + i + 8 > BUF_SIZE) {     /* too much... */
                        (void) strncat(msg.text, buf, BUF_SIZE-hdr_siz-i-8) ;
                        (void) strcat(msg.text, "...\r\n") ;
                        fprintf(stderr, 
                            "rsend: long message, %d word(s) truncated\n",
                                Argc + 1) ;
                    } else {
                        (void) strcat(msg.text, buf) ;
                        (void) strcat(msg.text, ">\r\n") ;
                    }
                    msg.text[BUF_SIZE-1] = '\0' ;
                    send_it() ;
                    (void) sprintf(msg.text, "\r\n<%s: ", my_name) ; 
                    hdr_siz = strlen(msg.text) ;
                }
            }
        } else {
            printf("\r\n") ;
            break ;
        }
    } 
}

void
do_shell (i, cmd)
    int i ;
    char *cmd ;
{
    int pid, w ;
    union wait status ;
    char *sh, *getenv() ;
    int (*saveintr)(), (*savequit)() ;

    if ((sh = getenv("SHELL")) == NULL)
        sh = "/bin/sh" ;
    if (i == 1) {                             /* interactive shell */
        if ((pid = vfork()) == 0)
            execl(sh, sh, (char *) 0) ;
    }
    else if ((pid = vfork ()) == 0)            /* do one commandline */
        execl(sh, sh, "-c", ++cmd, (char *) 0) ;
    saveintr = signal(SIGINT, SIG_IGN) ;
    savequit = signal(SIGQUIT, SIG_IGN) ;
    while ((w = wait(&status)) != pid && w != -1)
        ;
    if (i == 1)
        printf("\n") ;
    (void) signal(SIGINT, saveintr) ;
    (void) signal(SIGQUIT, savequit) ;
}

void
build_msg ()
{
    int     hdr_siz, too_much = FALSE ;
    char    buf[BUF_SIZE] ;

    msg.rqst_type = SEND ;
    hdr_siz = build_header() ;
    while (Argc--) {            /* copy everything from the commandline */
        (void) strcat(buf, " ") ;
        if (hdr_siz + strlen(buf) + 8 + strlen(*Argv) > BUF_SIZE ) {
            too_much = TRUE ;
            break ;
        }
        (void) strcat(buf, *Argv++) ;
    }
    if (hdr_siz + strlen(buf) > LINE_LENGTH)  /* separate heading from .. */
        (void) strcat(msg.text, "\r\n") ;     /*  ..message if it's long  */
    (void) strcat(msg.text, buf) ;
    if (too_much) {
        (void) strcat(msg.text, "...\r\n") ;
        fprintf(stderr, 
            "rsend: long message, %d word(s) truncated\n", Argc + 1) ;
    }
    else
        (void) strcat(msg.text, ">\r\n") ;
    msg.text[BUF_SIZE-1] = '\0' ;        /* just to make sure...*/
}

int
build_header ()
{
    long    clock ;
    struct  tm *tp, *localtime() ;

    (void) time(&clock) ;  /* message header: "<11:47 From foo@bar (ttyxx):" */
    tp = localtime(&clock) ;
    (void) sprintf(msg.text,"\r\n\007<%d:%02d From %s@%s (%s):", 
        tp->tm_hour, tp->tm_min, my_name, my_cpu, my_tty) ;
    return(strlen(msg.text)) ;
}

void
send_it ()
{
    struct sockaddr from ;
    int s, cc, fromlen = sizeof(from) ;

    s = socket(AF_INET, SOCK_DGRAM, 0) ;    /* get a socket */
    if (s < 0) {
        perror("rsend: socket") ;
        exit(1) ;
    }
    /* send to address in 'sin' through socket to (possibly remote) daemon */
    cc = sendto(s, (char *)&msg, sizeof(msg),
                0, (struct sockaddr *)&sin, sizeof (sin)) ;
    if (cc != sizeof(msg)) {
        perror("sendto") ;
        exit(1) ;
    }
    rsp.code = NO_ANSW ;
    (void) signal(SIGALRM, timeout) ;/* will not wait forever for a response */
    (void) alarm(TIMEOUT) ;
    (void) recvfrom(s, (char *)&rsp, sizeof(rsp), 0, &from, &fromlen) ;
    (void) close(s) ;
    (void) alarm(0) ;               /* don't time out on an interactive user */
    if (*msg.rcv_name == '\0')
        (void) strcpy(msg.rcv_name, rsp.rcv_name) ;
    if (*msg.rcv_tty == '\0')
        (void) strcpy(msg.rcv_tty, rsp.rcv_tty) ;
    if (rsp.code != SUCCESS)
        error_handle(rsp.code) ;
}

void
timeout ()
{
    fprintf(stderr, "rsend: timeout before acknowledged by rsend-daemon@%s\n",
        rcv_cpu) ;
    exit(0) ;
}

void
error_handle (errcod)
    int errcod ;
{
    char errmsg[BUF_SIZE] ;
    char recvr[NAME_SIZE + HOST_NAME_SIZE + 1] ;
    char daemon_at[HOST_NAME_SIZE + 15] ;

    if (remotesend) {
        (void) sprintf(recvr, "%s@%s", msg.rcv_name, rcv_cpu) ;
        (void) sprintf(daemon_at, "rsend-daemon@%s", rcv_cpu) ;
        (void) strcat(rcv_cpu, ":") ;
    } else {
        (void) strcpy(recvr, msg.rcv_name) ;
        (void) strcpy(daemon_at, "rsend-daemon") ;
        *rcv_cpu = '\0' ;
    }

    switch (errcod) {
    case NO_USER :
        (void) sprintf(errmsg, "%s: no such user", recvr) ;
        break ;
    case NOT_IN :
        (void) sprintf(errmsg, "%s not logged in", recvr) ;
        break ;
    case NOT_TTY :
        (void) sprintf(errmsg, "%s/dev/%s no such device",
            rcv_cpu, msg.rcv_tty) ;
        break ;
    case WRNG_TTY :
        (void) sprintf(errmsg, "%s is not on %s", recvr, msg.rcv_tty) ;
        break ;
    case NOT_USED :
        (void) sprintf(errmsg, "%s/dev/%s is not in use",
            rcv_cpu, msg.rcv_tty) ;
        break ;
    case DENIED :
        (void) sprintf(errmsg,
            "%s/dev/%s: permission denied, try mail instead", 
             rcv_cpu, msg.rcv_tty) ;
        break ;
    case OP_UTMP :
        (void) sprintf(errmsg, "cannot open %s/etc/utmp", rcv_cpu) ;
        break ;
    case OP_TTY :
        (void) sprintf(errmsg, "cannot open %s/dev/%s", rcv_cpu, msg.rcv_tty) ;
        break ;
    case WR_FAIL :
        (void) sprintf(errmsg, "couldn't write on %s/dev/%s", 
            rcv_cpu, msg.rcv_tty) ;
        break ;
    case CONFUSED :
        (void) sprintf(errmsg, "%s is confused", daemon_at) ;
        break ;
    case NO_ANSW :
        (void) sprintf(errmsg, "no answer from %s", daemon_at) ;
        break ;
    case CHDIR :
        (void) sprintf(errmsg, "couldn't chdir to /dev") ;
        break ;
    default :
        (void) sprintf(errmsg,
            "this cannot happen! (unknown error, no:%d)", errcod) ;
    }
    fprintf(stderr, "rsend: %s\n", errmsg) ;
    exit(1) ;
}
