/*
 *   Routines to manage the windows.  SOme of this is pretty ugly
 *   stuff, especially as most of the time is spent here.
 *
 *   We manage to resize windows by buffering all parts of a conversation,
 *   so we just (re)print the buffered part.  The buffering also accounts
 *   for most of the lack of speed.  A future enhancement will probably
 *   be to replace the constant buffering with a smarter routine that
 *   will just grab each window from the curses screen buffer when we call
 *   stretch() ...
 */


#include <curses.h>
#include <sys/ioctl.h>
#include <signal.h>
#undef ctrl()			/* defined in curses.h */
#include "defs.h"

#ifndef cbreak()		/* these are defined in *some* versions of curses.h */
#define cbreak() (_tty.sg_flags |= CBREAK, _rawmode = TRUE, stty(_tty_ch,&_tty))
#define nocbreak() (_tty.sg_flags &= ~CBREAK,_rawmode=FALSE,stty(_tty_ch,&_tty))
#endif  !cbreak()

#ifndef WINDOWS
#define WINDOWS 32		  /* we can only fit 23 on a sun !!! */
#endif

struct win {
	int   inuse;          /* 1 if in use, 0 if not      */
	char  *login;         /* login name of this person  */
	char  *host;          /* their host name            */
	char  *tty;           /* tty                        */
	char  *realname;      /* finger name, if any        */
	int   upper;          /* top line of this window    */
	int   lower;          /* bottom line of this window */
	int   y;              /* y coord rel to top of wind */ 
	int   x;              /* last known x coordinate    */
	int   old;            /* index of oldest ch in buf  */
	int   new;            /* where next char goes       */
	char  *buf;           /* text buffer                */
};

static	struct win windows[WINDOWS];
static	struct win *currwin;	/* pointer to current window    */
int		current = 0;     		/* currently active window      */
int		users;					/* number of users/windows      */
int		upper_y;				/* boundaries of current window */
int		lower_y;
int		stretching = 0;
int		did_screen = 0;			/* have set up curses          */
static	int y , x;				/* current screen coordinates  */
extern	int sigstop();			/* ^Z handling routine         */

static int ourwin;				/* the window # that is _ours_ */

/*
 *  Mark all of the windows as not being in use.
 */

setup ()
{
	int  i;

	for (i = 0; i < WINDOWS; i++) {
		windows[i].inuse = 0;
		windows[i].old   = 0;
		windows[i].new   = 0;
	}
	
	initscr();
	did_screen = 1;
	clear ();
	cbreak ();
	noecho ();
	nonl ();
	refresh ();

	signal (SIGTSTP, sigstop);
	geteditchars ();				/* get and save his editing characters */

	maxx = stdscr->_maxx - 1;      			/* save max screen coordinates */
	maxy = stdscr->_maxy - 1;

	windows[0].upper = upper_y = 0;				/* initial screen bounds */
	windows[0].lower = lower_y = maxy - 1;
	windows[0].y = y = 1;						/* initial window position */
	windows[0].x = x = 0;
	currwin = &windows[0];
	move (1, 0);
	users = 0;									/* not talking to anyone yet */
}


/*
 *  Restore everything to how it was before we came...
 */

cleanup ()
{
	move (maxy, 0);
	refresh ();
	nocbreak ();
	echo ();
	endwin ();
	did_screen = 0;
}


/*
 *  Select the new current window to use, set the window bounds,
 *  and set the new cursor position ...  Returns the old window number.
 */

selwin (win)
char    win;
{
	int  old = current;

	if (win == current && !stretching)	/* no change */
		return (win);
	if (win == ourwin)				/* map our window to window 0 */
		win = 0;

	if (windows[win].inuse == 0)
		return (old);

	if (win >= 0 && win < WINDOWS) {
		currwin->y = y;				/* save last coordinates */
		currwin->x = x;
		currwin = &windows[win];	/* set pointer to new window */
		y = currwin->y;				/* and retrieve from new win */
		x = currwin->x;
		upper_y = currwin->upper;	/* get bounds */
		lower_y = currwin->lower;
		move (y + upper_y, x);		/* restore cursor pos */
		current = win;
	}
	return (old);
}



/*
 *  Add a new window to the screen.
 *  Buf looks like "user:host:tty:realname"
 */

adduser (slot, buf)
int     slot;
char    *buf;
{
	char	*i;

	/*
	 *  We rely on the fact that the first window given to us is
	 *  our own, so we remember which one that is and remap it 
	 *  to the top window. This makes people happier.
	 */

	if (users == 0) {
		ourwin = slot;
		slot = 0;
	}

	if (windows[slot].inuse)				/* that window is already in use */
		return;

	windows[slot].inuse = 1;				/* mark as being in use */
	windows[slot].x = 0;					/* and current coordinates */
	windows[slot].y = 1;
	windows[slot].buf = malloc (BUFFER);	/* and grab a buffer for it */
	windows[slot].old = 0;					/* nothing in the buffer yet */
	windows[slot].new = 0;

	/*  get the info out of the buffer line  */
	if (i = index (buf, ':'))    /* name */
		*i++ = '\0';
	else
		return;
	windows[slot].login = strsave (buf);
	buf = i;
	i = index (buf, ':');    /* host */
	*i++ = '\0';
	windows[slot].host = strsave (buf);
	buf = i;
	i = index (buf, ':');    /* tty */
		*i++ = '\0';
	windows[slot].tty = strsave (buf);
	buf = i;
	windows[slot].realname = strsave (buf);  /* real name */

	if (users++ == 1)					/* add another user to the count */
		(void) write (1, "\07\07", 2);	/* initial connection - beep the user */
	stretch (0);						/* and recompute all the windows */
}



/*
 *  Delete a user and his window.
 */

deluser (win)
char     win;
{
	if (win < 0 || win >= WINDOWS)
		return;

	if (windows[win].inuse == 0)
		return;

	windows[win].inuse = 0;			/* this window no longer in use */
	free (windows[win].login);		/* so free up any space allocated for it */
	free (windows[win].host);
	free (windows[win].tty);
	free (windows[win].realname);
	free (windows[win].buf);

	if (--users > 1) {				/* decrement the count */
		stretch (0);				/* and redraw */
		refresh ();
	} else							/* if only one user left .. */
		quit ();					/* then we're done */
}



/*
 *  Show the given character, performing word and line erase.
 */

showch (ch)
char    ch;
{
	if (users == 0)
		return;

	if (!stretching) {
		/* save char into buffer */
		if (currwin->new >= BUFFER)
			error (1, "New >= BUFFER in showch()");
		currwin->buf[currwin->new++] = ch;
		if (currwin->new >= BUFFER)			/* wrap around */
			currwin->new = 0;
		if (currwin->new == currwin->old)	/* tail caught up */
			currwin->old++;
		if (currwin->old >= BUFFER)			/* wrap old around */
			currwin->old = 0;
	}

	switch (ch) {
		case '\0177':	/* delete */
		case ctrl(H):   /* backspace */
						if (x > 0) {
							addstr ("\b \b");
							x--;
						}
						break;

		case ctrl(I):	/* tab */
						addch (' ');
						x++;
						while (x++ % 8)
							addch (' ');
						break;

		case ctrl(J):	/* newline */
						x = 0;
						if ((++y + upper_y) > lower_y)
							y = 1;
						move (y + upper_y, 0);
						clrtoeol ();
						break;

		case ctrl(L):	/* clear to end of current line */
						clrtoeol ();
						break;

		case ctrl(M):	/* carriage return */
						x = 0;
						move (y + upper_y, 0);
						break;

		case ctrl(N):	/* clear window */
						for (y = 1; (y+upper_y) <= lower_y; y++) {
							move (y + upper_y, 0);
							clrtoeol ();
						}
						y = 1; x = 0;
						move (y + upper_y, 0);
						break;

		case ctrl(U):	/* line kill */
		case ctrl(X):	x = 0;
						move (y + upper_y, 0);
						clrtoeol ();
						break;

		case ctrl(W):	/* word erase */
						while (x >= 0) {         /* nuke the spaces */
							if (inch () != ' ')
								break;
							addstr (" \b\b");
							if (x > 0)
								x--;
							else
								break;
						}
						while (x >= 0) {        /* and the non-spaces */
							if ((ch = inch ()) == ' ') {
								addch (ch);
								x++;
								break;
							}
							addstr (" \b\b");
							if (x > 0)
								x--;
							else
								break;
						}
						break;

		case ctrl(^):	/* home cursor */
						y = 1; x = 0;
						move (y + upper_y, 0);
						break;

		case ctrl(G):	/* bell - switchable */
						if (Bells) {
							(void) write (1, "\007", 1);
							break;
						}
						/* fall through ... */

		default:		if (ch < ' ') {
							addch ('^');
							ch += '@';
							x++;
						}
						addch (ch);
						x++;
						break;
	}

	if (x >= maxx) {				/* have wrapped around to next line */
		y++;
		x %= maxx;
		if (y + upper_y > lower_y)	/* wrapped back to top of window */
			y = 1;
		move (upper_y + y, 0);
		clrtoeol ();
	}
}


/*
 *  Print a message to the bottom line.
 */

putmessage (str)
char *str;
{
	int  y, x;
	extern int Inverse;
	extern int did_screen;

	if (!did_screen) {
		printf ("%s\r\n", str);
		fflush (stdout);
		return;
	}
	touched25 = 1;				/* has been changed  */
	getyx (stdscr, y, x);		/* save current pos  */
	move (maxy, 0);				/* go to bottom line */
	clrtoeol ();				/* clear it          */
	refresh ();					/* they see it clear */
	if (Inverse)
		standout ();

	while (*str) {				/* print out the message, removing ctrl chars */
		if (*str < ' ') {
			addch ('^');
			addch (*str + '@');
		} else
			addch (*str);
		str++;
	}

	if (Inverse)
		standend ();
	move (y, x);				/* restore old pos   */
	refresh ();					/* etc ...           */
}


/*
 *  Print out a header for the given slot.
 */

header (num)
int num;
{
	register int i;
	extern   int Inverse;

	if (windows[num].inuse == 0) {
		error (0, "header(): slot not in use");
		return;
	}

	sprintf (buf, "---- %s@%s on %s (%s) ", windows[num].login, 
			windows[num].host, windows[num].tty, windows[num].realname);
	
	for (i = strlen (buf); i < maxx; i++)
		buf[i] = '-';
	buf[maxx-2] = '\0';		/* two (four?) for magic-cookie terminals */

	move (windows[num].upper, 0);
	if (Inverse)
		standout ();
	addstr (buf);
	if (Inverse)
		standend ();
}



/*
 *  Take the current screen and contort it to handle the new windows.
 *
 *  Handles both the screen changing size and just the 
 *  number of windows changing.
 *
 *  We set caught_sig to be non-zero if coming here on SIGWINCH, 0 otherwise.
 *  We also only refresh the screen if we caught the signal....
 */

stretch (caught_sig)
int caught_sig;
{
	register int   i, w;
	register int   extra;
	register int   lines;
	int      oldy, oldx;
	int      newy, newx;
	int      lines_win;
	int      oldcurr = current;

	if (users == 0)		/* nothing to recompute */
		return;

	stretching = 1;

	oldy = maxy + 1;
	oldx = maxx + 1;

	if (caught_sig) {			/* got SIGWINCH, have to redo screen */
#if defined(TIOCGWINSZ) || defined(TIOCGSIZE)
#ifdef TIOCGWINSZ
		struct winsize winsize;
#else  TIOCGSIZE	/* (older?) suns */
		struct ttysize ttysize;
#endif TIOGSIZE

		/* get new screen size */
#ifdef TIOCGWINSZ
		if (ioctl (0, TIOCGWINSZ, &winsize)) {
			error (0, "ioctl: TIOCGWINSZ");
			return;
		}
		newy = winsize.ws_row;
		newx = winsize.ws_col;
#else TIOCGSIZE
		if (ioctl (0, TIOCGSIZE, &ttysize)) {
			error (0, "ioctl: TIOCGSIZE");
			return;
		}
		newy = ttysize.ts_lines;
		newx = ttysize.ts_cols;
#endif TIOCGSIZE 
		
#else /* don't have window size ioctls */
		/* 
		 *  Can't get new window size, so presumably sigwinch 
		 *  doesn't exist and this code will never happen !!!
		 */
		newy = oldy;
		newx = oldx;
#endif TIOCGWINSZ
		maxy = newy - 1;	/* max screen coordinate - zero-based */
		maxx = newx - 1;
	} else {				/* screen size hasn't changed */
		newy = oldy;
		newx = oldx;
	}

	if (newx != oldx || newy != oldy) {			/* have to shut down curses */
		echo ();
		nocbreak ();
		endwin ();
		did_screen = 0;

		initscr ();			/* and restart it */
		did_screen = 1;
		cbreak ();
		noecho ();
		nonl ();
		signal (SIGTSTP, sigstop);
	} else {				/* just have to clear all the lines */
		for (i = 0; i < newy; i++) {
			move (i, 0);
			clrtoeol ();
		}
	}

	lines_win = (newy - 1) / users;		/* recompute window boundaries */
	extra = (newy - 1) - (users * lines_win);	/* total extra lines */
	lines = 0;

	/* 
	 *   Now calculate the new boundaries and
	 *   copy the buffer back to the new window.
	 */
	
	x = 0; y = 1;		/* go back to the beginning of the window */
	for (w = 0; w < WINDOWS; w++) {		/* now go copy boundaries */
		if (windows[w].inuse == 0)
			continue;
		windows[w].upper = lines;	/* new top boundary */
		if (extra-- > 0)			/* extra lines for this window */
			lines++;
		lines += lines_win;			/* add on lines */
		windows[w].lower = lines-1;	/* new bottom boundary */
		header (w);					/* redraw the window header */
		windows[w].x = 0;			/* reset position to top corner */
		windows[w].y = 1;
		selwin (w);					/* switch to this window */
		/* and dump the buffer back into the window */
		for (i = windows[w].old; i != windows[w].new; i = (i+1) % BUFFER)
			showch (windows[w].buf[i]);
	}

	move (y, x);
	selwin (oldcurr);				/* reselect the right window */
	if (caught_sig)
		refresh ();					/* and make it all visible */
	stretching = 0;
}
