#include "defs.h"
#include <stdio.h>
#include <signal.h>

/*
 *  Routines for dealing with the messages printed 
 *  down on the bottom line of the screen.
 */

struct mesg {
	char   *text;          /* text of this message */
	struct mesg	*next;     /* link to next message */
};

static struct mesg *mesghead = (struct mesg *) 0;  /* top of queue     */
static struct mesg *last     = (struct mesg *) 0;  /* bottom of queue  */
static struct mesg *freelist = (struct mesg *) 0;  /* stuff to free up */
int    msgpending  = 0;          		/* number of pending messages  */


/*
 *  Add a message to the pending message queue.
 */

message (str)
char    *str;
{
	struct mesg *new;
	int    sigs;

	sigs = sigblock (1 << SIGALRM);

	if (!freelist) {		/* need to malloc some up */
		new = (struct mesg *) malloc (sizeof (struct mesg));
		if (!new)
			error (1, "cannot malloc message buffer!");
	} else {				/* take off the top of the list */
		new = freelist;
		freelist = freelist->next;
		free (new->text);
	}
	new->text = strsave (str);
	new->next = (struct mesg *) 0;

	if (last)       				/* add to end of currently pending list */
		last->next = new;
	if (!mesghead)  				/* empty queue - add to top */
		mesghead = new;
	last = new;     				/* update bottom pointer */
	if (msgpending <= 0) { 			/* need to restart alarms */
		alarm (1);
		msgpending = 1;
	} else
		msgpending++;
	(void) sigsetmask (sigs);
}


/*
 *  Called by the alarm interrupt routine - print the 
 *  next pending message and remove it from the queue.
 */

showmessage ()
{
	static int ticks = 0;
	struct mesg *m;

	if ((ticks = (ticks + 1) % Interval) != 0)
		return;

	if (msgpending == 0) {     /* just clean up after last message */
		putmessage ("");
		msgpending = -1;
		return;
	}

	m = mesghead;           /* save pointer to this message */

	putmessage (m->text);
	mesghead = m->next;     /* move down top of queue */
	if (last == m)          /* was the last item, too */
		last = (struct mesg *) 0;

	m->next = freelist;		/* add this item to the free list */
	freelist = m;			/* this way we don't free in an interrupt !!! */
	msgpending--;		    /* and decrement counter  */
}



/*
 *  Flush the pending message buffer.  Not actually used yet.
 */

flush ()
{
	last->next = freelist;				/* add curr freelist to end of list */
	last = (struct mesg *) 0;
	freelist = mesghead;				/* move pending list to free list */
	mesghead = (struct mesg *) 0;		/* zap the head */
	msgpending = 0;						/* and reset the counter */
}
