;;******************************************************************************
;;                         bridge.asm      bridge.asm
;;******************************************************************************
;;
;;  Copyright (C) 1989 Northwestern University, Vance Morrison
;;
;;
;; Permission to view, compile, and modify for LOCAL (intra-organization) 
;; USE ONLY is hereby granted, provided that this copyright and permission 
;; notice appear on all copies.  Any other use by permission only.
;;
;; Northwestern University makes no representations about the suitability 
;; of this software for any purpose.  It is provided "as is" without expressed 
;; or implied warranty.  See the copywrite notice file for complete details.
;;
;;******************************************************************************
;; 
;; Bridge.asm provides ethernet bridge cababilities.  Bridge will do bridging
;; between the interfaces provided to it.  It will also act like a IF object
;; and return packets that are directed to any of its interfaces.
;; 
;; At present, this module requires the use of the WDE IF object.  At some time
;; I may lift the following assumtions I make in this module, and thus any IF
;; object could be used for the bridge.
;;
;; The functions provided by this file are
;;
;;   BDG_DECLARE name, ifs, forward_ip
;;   BDG_DEFINE name
;;   BDG_IF_R_ACCESS_out_BX_CX_ES interface, no_packet
;;   BDG_IF_R_CONT_in_BX_CX_ES_const_BX_CX_DX_BP_SI_DI_ES interface, ok
;;   BDG_IF_R_FREE_const_BX_CX_BP_SI_DI_ES interface
;;   BDG_IF_W_ACCESS_in_CX_out_DI_ES_const_BX_CX_BP interface, no_buffer
;;   BDG_IF_W_WRITE_in_CX_const_BX_BP_ES interface
;;   BDG_IF_SET_ADDRESS_in_SI_const_BX_CX_BP_DI_ES interface
;;   BDG_IF_COPY_in_CX_SI_DI_ES_out_SI_DI_const_BX_BP_ES interface
;;
;; Variables set by this module
;;
;;   bdg_&name&_declared                       ;; one if this interface exists
;;   if_&name&_address                         ;; the hardware address
;;
;;******************************************************************************


;;******************************************************************************
;;   BDG_DECLARE name, ifs, forward_ip  
;;      BDG_DECLARE declares a bridge made up of the ifs 1..ifs.  All of these
;;      ifs are assumed to be declared to be PROMISCUOUS and at present
;;      if 'forward_ip' is not blank, then IP broadcasts and ARPS will be 
;;      forwarded, otherwize they will not
;;
BDG_DECLARE MACRO name, ifs, forward_ip
    .errb <name>
    .errb <ifs>

    if name ne 1
        .err                ;; you are only alowed 1 bridge and it must be
                            ;; named '1'
    endif

    bdg_&name&_declared  = 1
    bdg_&name&_ifs       = ifs
    global bdg_&name&_table:byte 
    ifnb <forward_ip>
        bdg_&name&_forw_ip   = 1
    else
        bdg_&name&_forw_ip   = 0
    endif
ENDM


;;******************************************************************************
;;   BDG_DEFINE name
;;      sets asside memory for the bridge and initializes it.
;;      routine is a no-op if 'name' was not declared
;;
BDG_DEFINE MACRO name
    LOCAL loop1, loop2, loop3
    .errb <name>

ifdef bdg_&name&_declared
    .FarData?
    bdg_&name&_table db 65535 dup (?)           ;; set aside a whole segment
                                                ;; for the bridge table

    .CODE
    mov AX, SEG bdg_&name&_table
    mov AX, ES
    xor DI, DI
    xor CX, CX
    xor AX, AX
    rep
    stosw                       ;; null the bridge table

endif
ENDM


;;******************************************************************************
;;   IF_R_ACCESS_out_BX_ES interface, no_packet
;;       IF_R_ACCESS waits for the next packet to come from the the board
;;       associated with 'interface' and returns a pointer to the begining of 
;;       an ethernet packet in BX:ES.  CX holds the length of the packet
;;       R_ACCESS jumps to 'no_packet' if there are no packets waiting to 
;;       be read in
;;       
BDG_IF_R_ACCESS_out_BX_CX_ES MACRO interface, no_packet
    local got_packet, forward_all, forward, not_me, drop, dont_insert
    local broadcast, broadcast_drop
    .errb <no_packet>

    WDE_IF_R_ACCESS_out_BX_CX_ES interface, no_packet
;print_reg <got a packet size >, CX
;print_reg <got a packet ptr >, BX
;push SI
;push CX
;mov SI, BX
;mov CX, 6
;dump_in_SI_CX ES
;print <myaddress>
;mov SI, offset if_&interface&_address
;mov CX, 6
;dump_in_SI_CX DS
;pop CX
;pop SI

    mov AX, ES:[BX+4]
    cmp AX, word ptr if_&interface&_address+4
    jnz not_me
    mov AX, ES:[BX+2]
    cmp AX, word ptr if_&interface&_address+2
    jnz not_me
    mov AX, ES:[BX]
    cmp AX, word ptr if_&interface&_address
    jz got_packet
    not_me:
;print <not me>

    mov BP, DS                  ;; save DS
    mov SI, BX

    mov AX, ES
    mov DS, AX                  ;; DS points to the packet
    mov AX, SEG bdg_1_table
    mov ES, AX                  ;; ES points to the table

    add SI, 6                   ;; skip to source address of the packet

;push DS
;mov DS, BP
;print <source ethernet address>
;push SI
;push CX
;mov CX, 6
;dump_in_SI_CX DS
;pop CX
;pop SI
;pop DS

    mov DI, [SI+4]              ;; compute hash function (LSB * 8 mod 65k)
    shl DI, 1
    shl DI, 1
    shl DI, 1
    movsw                       ;; save the ethernet address
    movsw
    movsw
    mov AL, interface           ;; and the fact that it is on this interface
    stosb

    sub SI, 12                  ;; restore SI to the begining of the packet

;push DS
;xor AX, AX
;mov AL, byte ptr [SI]
;mov DS, BP
;print_reg <first byte>, AX
;pop DS

    test byte ptr [SI], 80H     ;; is it a broadcast
    jnz broadcast

    mov DI, [SI+4]              ;; compute hash function (LSB * 8 mod 65k)
    shl DI, 1
    shl DI, 1
    shl DI, 1

;push DS
;mov DS, BP
;print <lookup table address>
;push SI
;push CX
;mov CX, 8
;mov SI, DI
;dump_in_SI_CX ES
;pop CX
;pop SI
;pop DS

    cmpsw
    jnz forward_all         ;; didn't find in table
    cmpsw
    jnz forward_all         ;; didn't find in table
    cmpsw
    jnz forward_all         ;; didn't find in table
    mov DS, BP              ;; restore DS
    mov AL, ES:[DI]
    cmp AL, interface
    jnz forward
        drop:
        ;print <dropping>
        WDE_IF_R_FREE_const_BX_CX_BP_SI_DI_ES interface
        jmp no_packet

    broadcast_drop:
    mov DS, BP              ;; restore DS
    WDE_IF_R_FREE_const_BX_CX_BP_SI_DI_ES interface
    jmp no_packet

    broadcast:
;push DS
;mov AX, word ptr [SI+12]
;mov DS, BP
;print_reg <broadcast, type =>, AX
;pop DS
    if bdg_1_forw_ip eq 0
;push DS
;mov DS, BP
;print <checking for drop>
;pop DS
        mov AX, word ptr [SI+12]
        cmp AX, 0008H           ;; is it an IP packet
        jz broadcast_drop
        cmp AX, 0608H           ;; is it an ARP packet
        jz broadcast_drop
;print <forwarding>
    endif

    forward_all:
        mov DS, BP              ;; restore DS
        mov BP, CX
        ;print_reg <forwarding to all>, SI
        IRP idx, <1,2,3,4,5,6,7,8>
        if idx le bdg_1_ifs 
        if idx ne interface
            ;print <forwarding to if idx>
            mov CX, BP
            WDE_IF_W_ACCESS_in_CX_out_DI_ES_const_BX_CX_BP idx, drop
            mov SI, BX
            WDE_IF_COPY_in_CX_SI_DI_ES_out_SI_DI_const_BX_BP_ES interface
            mov CX, BP
            WDE_IF_W_WRITE_in_CX_const_BX_BP_ES idx
        endif
        endif
        endm
       jmp drop
    
    forward:
        mov BP, CX
        ;print_reg <forwarding to specific if>, SI
        IRP idx, <1,2,3,4,5,6,7,8>
        local next
        if idx le bdg_1_ifs 
        if idx ne interface
            cmp AL, idx
            jnz next
                ;print_reg <forwarding to if idx, LEN>, CX
                WDE_IF_W_ACCESS_in_CX_out_DI_ES_const_BX_CX_BP idx, drop
                mov SI, BX
                WDE_IF_COPY_in_CX_SI_DI_ES_out_SI_DI_const_BX_BP_ES interface
                mov CX, BP
                WDE_IF_W_WRITE_in_CX_const_BX_BP_ES idx
            next:
        endif
        endif
        endm
       jmp drop

    got_packet:
ENDM


;;******************************************************************************
;;   IF_R_FREE_const_BX_CX_BP_SI_DI_ES  interface
;;       After the client is through processing the packet returned by 
;;       IF_R_ACCESS, IF_R_FREE must be called to inform 'interface' that the 
;;       memory that the packet was in can be reused for future packets.
;;
BDG_IF_R_FREE_const_BX_CX_BP_SI_DI_ES MACRO interface
    local inside
    .errb <interface>

    WDE_IF_R_FREE_const_BX_CX_BP_SI_DI_ES interface
ENDM


;;******************************************************************************
;;   BDG_IF_R_CONT_in_BX_CX_ES interface, ok
;;       IF_R_CONT determines if the packet returned by R_READ in BX:ES
;;       of length CX is continuous.  If it is it jumps to 'ok' otherwise
;;       it just returns
;;
BDG_IF_R_CONT_in_BX_CX_ES_const_BX_CX_DX_BP_SI_DI_ES MACRO interface, ok
    .errb <ok>

    WDE_IF_R_CONT_in_BX_CX_ES_const_BX_CX_DX_BP_SI_DI_ES interface, ok
ENDM


;;******************************************************************************
;;   IF_W_ACCESS_in_CX_out_DI_ES interface, no_buffer
;;       IF_W_ACCESS returns a pointer to an output buffer for a packet.  The 
;;       pointer is returned in DI:ES.  If the ouptut buffer is busy, this 
;;       routine will jump to 'no_buffer'.  The output buffer  min(CX, 1536) 
;;       bytes long
;;
BDG_IF_W_ACCESS_in_CX_out_DI_ES_const_BX_CX_BP MACRO interface, no_buffer
    .errb <no_buffer>

    WDE_IF_W_ACCESS_in_CX_out_DI_ES_const_BX_CX_BP interface, no_buffer
ENDM


;;******************************************************************************
;;   IF_W_WRITE_in_CX interface
;;       IF_W_WRITE actually signals the ethernet board to write a packet to 
;;       the ethernet.  The packet is assumed to be in the buffer returned by 
;;       IF_W_ACCESS. CX is the length of the packet to send.  
;;
BDG_IF_W_WRITE_in_CX_const_BX_BP_ES MACRO interface
    .errb <interface>

    WDE_IF_W_WRITE_in_CX_const_BX_BP_ES interface
ENDM


;;******************************************************************************
;;   IF_SET_ADDRESS_in_SI interface
;;       IF_SET_ADDRESS_in_SI sets the hardware address to be the value
;;       pointed to by SI.  Note this function may be a no-op if the
;;       hardware address cannot be set (ETHERNET for example)
;;
BDG_IF_SET_ADDRESS_in_SI_const_BX_CX_BP_DI_ES MACRO interface
    .errb <interface>

    WDE_IF_SET_ADDRESS_in_SI_const_BX_CX_BP_DI_ES interface
ENDM


;;******************************************************************************
;;   IF_COPY_in_CX_SI_DI_ES interface
;;      IF_COPY_in_CX_SI_DI_ES copys a packet from the input buffer (pointed 
;;      to by SI and the segement register given in IF_DECLARE) to an output 
;;      buffer (pointed to by DI and dest_reg) of length CX.   It assumes the
;;      output buffer is contiguous.  (and the caller shouln't care if the 
;;      input buffer is contiguous)
;;
BDG_IF_COPY_in_CX_SI_DI_ES_out_SI_DI_const_BX_BP_ES MACRO interface
    local wrap, done
    .errb <interface>

    WDE_IF_COPY_in_CX_SI_DI_ES_out_SI_DI_const_BX_BP_ES interface
ENDM

