;;*****************************************************************************
;;                         slip.inc            slip.inc
;;*****************************************************************************
;;
;;  Copyright (C) 1989 Northwestern University, Vance Morrison
;;
;;
;; Permission to view, compile, and modify for LOCAL (intra-organization)
;; USE ONLY is hereby granted, provided that this copyright and permission
;; notice appear on all copies.  Any other use by permission only.
;;
;; Northwestern University makes no representations about the suitability
;; of this software for any purpose.  It is provided "as is" without expressed
;; or implied warranty.  See the copywrite notice file for complete details.
;;
;;*****************************************************************************
;;
;;  slip.asm contains the DL_IP interface for the serial SLIP protocols.
;;  That is this module contains the code that wraps IP packets up into
;;  SLIP packets before sending them out, and unencapsulates incomming
;;  IP packets.  As far has IP is concerned this is just a DL_IP interface
;;
;;   SLIP_DECLARE name, task, port, wqueuesize, wbuffsize, rqueuesize, rbuffsiz
;;   SLIP_DEFINE name, ip_address, ip_mask
;;   SLIP_DL_IP_R_READ name, code_label
;;   SLIP_DL_IP_R_CONT_in_BX_CX_ES_const_BX_CX_DX_BP_SI_DI_ES name, ok
;;   SLIP_DL_IP_RETURN name
;;   SLIP_DL_IP_W_ACCESS_in_CX_out_DI_ES_const_CX_BP name, fail
;;   SLIP_DL_IP_W_WRITE_in_AX_CX_const_BP name, broadcast
;;   SLIP_DL_IP_IS_BROADCAST_in_BX_ES_const_AX_BX_CX_DX_BP_DI_ES name
;;   SLIP_DL_IP_COPY_in_CX_SI_DI_ES_out_SI_DI_const_BX_BP_ES name
;;
;;  Variables Provided by this module (READ ONLY!!!!)
;;
;;      slip_&name&_declared        1 if this object has been declared
;;      dl_ip_&name&_ip             the IP address
;;      dl_ip_&name&_mask           the network mask
;;      dl_ip_&name&_net            the network (IP addr bitwize AND ip_mask)
;;      dl_ip_&name&_broad          the network broadcast address
;;      dl_ip_&name&_flags          A word exclusively for IP use
;;      dl_ip_&name&_mtu            the Maximum transmission unit (packet size)
;;
;; AUTHOR: David Johnson
;; DATE:   7/20/89            ####### BETA TEST #######
;;*****************************************************************************
 

;;*****************************************************************************
;; data storage needed by this module
 
;; 8250 serial controller base registers
 
ComPort1 = 3F8H
ComPort2 = 2F8H
ComPort3 = 3E8H
ComPort4 = 2E8H
 
;; 8259 interrupt controller registers
 
IntCtl   = 20H
IntMsk   = 21H
 
;; COM port interrupt vectors
 
ComInt1  = 4H
ComInt2  = 3H
 
;; SLIP special characters
 
FR_END   = 0C0H
FR_ESC   = 0DBH
T_FR_END = 0DCH
T_FR_ESC = 0DDH
 
 
slip_w_q_entry STRUC
    slip_q_wstart    DW
    slip_q_wend      DW
slip_w_q_entry ENDS
 
slip_r_q_entry STRUC
    slip_q_rstart    DW
    slip_q_rend      DW
slip_r_q_entry ENDS
 
slip_data    STRUC
    slip_speed       DW                 ;; baud rate divisor for 8250
 
    slip_write       DB                 ;; currently in process of write
    slip_wstart      DW                 ;; start of current write packet
    slip_wend        DW                 ;; end of current write packet
    slip_wpos        DW                 ;; current position of write
    slip_wstate      DB                 ;; current write state
 
    slip_rstart      DW
    slip_rend        DW
    slip_rpos        DW
    slip_rstate      DB
    slip_rdrop       DB                 ;; drop current packet? (overrun)
slip_data     ENDS
 

;;*****************************************************************************
;; SLIP_DECLARE name, task, port, wqueuesize, wbuffsize, rqueuesize, rbuffsize
;;    SLIP_DECLARE declares all the external defintions for the SLIP object
;;
SLIP_DECLARE MACRO name,task,port, wqueuesize, wbuffsize, rqueuesize, rbuffsize
    .errb <port>
 
    slip_&name&_declared = 1
    slip_&name&_task = task
    slip_&name&_wqueuesize = wqueuesize
    slip_&name&_wbuffsize  = wbuffsize
    slip_&name&_rqueuesize = rqueuesize
    slip_&name&_rbuffsize  = rbuffsize
    slip_&name&_wbuff      = name*100+1     ;; unique ids
    slip_&name&_wqueue     = name*100+2
    slip_&name&_rbuff      = name*100+3
    slip_&name&_rqueue     = name*100+4
 
    slip_&name&_port       = ComPort&port&  ;; actual port for int handler
    dl_ip_&name&_mtu       = 1006
 
    .DATA
    global dl_ip_&name&_ip:dword
    global dl_ip_&name&_mask:dword
    global dl_ip_&name&_net:dword
    global dl_ip_&name&_broad:dword
    global dl_ip_&name&_flags:word
    global dl_ip_&name&_metric:word
    global slip_&name&_data:slip_data
    global slip_&name&_read:word
    global slip_&name&_wbuffer:byte
    global slip_&name&_rbuffer:byte
    dl_ip_&name&_haddr = (slip_&name&_data.slip_speed)
 
    .CODE
    global slip_&name&_continue:near
    global slip_&name&_real_define:near
 
    BUFF_DECLARE %slip_&name&_wbuff, slip_&name&_wbuffer, wbuffsize
    BUFF_DECLARE %slip_&name&_rbuff, slip_&name&_rbuffer, rbuffsize
    QUEUE_DECLARE %slip_&name&_wqueue, wqueuesize, %(size slip_w_q_entry)
    QUEUE_DECLARE %slip_&name&_rqueue, rqueuesize, %(size slip_r_q_entry)
 
ENDM
 

;;*****************************************************************************
;;   SLIP_DEFINE name, ip_address, ip_mask, speed, fail
;;          speed is actual divisor for 8250
;;
SLIP_DEFINE MACRO name, ip_address, ip_mask, speed, fail
 
ifdef slip_&name&_declared
 
    mov AX, word ptr ip_address                 ;; copy over params
    mov BX, 0FFFFH                              ;; Force our assumtion
    mov word ptr dl_ip_&name&_ip, AX
    mov word ptr dl_ip_&name&_net, AX
    mov word ptr dl_ip_&name&_broad, AX
    mov word ptr dl_ip_&name&_mask, BX
 
    mov AX, word ptr ip_address+2
    mov BX, word ptr ip_mask+2
    mov word ptr dl_ip_&name&_ip+2, AX
    mov word ptr dl_ip_&name&_mask+2, BX
    and AX, BX
    mov word ptr dl_ip_&name&_net+2, AX
    not BX
    or AX, BX
    mov word ptr dl_ip_&name&_broad+2, AX
 
    mov AX, speed
    mov slip_&name&_data.slip_speed, AX
 
    call slip_&name&_real_define
    or AX, AX
    jnz fail

    SLIP_HANDLER_DEFINE name        ;; define interupt handler 
endif
ENDM
 
SLIP_REAL_DEFINE_in_SI_out_AX MACRO name
    local start, around, null_read, fail
 
ifdef slip_&name&_declared
    .DATA
    slip_&name&_data    slip_data <>            ;; create storage needed
    slip_&name&_read    DW
    slip_&name&_wbuffer DB  slip_&name&_wbuffsize dup (0)
    slip_&name&_rbuffer DB  slip_&name&_rbuffsize dup (0)
    dl_ip_&name&_ip     DD
    dl_ip_&name&_mask   DD
    dl_ip_&name&_net    DD
    dl_ip_&name&_broad  DD
    dl_ip_&name&_flags  DW
    dl_ip_&name&_metric  DW
 
    .CODE
    jmp around
            ;; declare these in the code segement in case we put this in ROM
 
        start:
        SLIP_TASK name
            ;; this does NOT fall through
 
        null_read:
            SLIP_DL_IP_R_RETURN name
                ;; this does NOT fall through
 
    around:
    slip_&name&_real_define:
 
            ;; INITIALIZE THE SERIAL PORT
 
    SLIP_CHECK_PORT name, fail      ;; check if port specified exists
 
    mov DX, IntMsk
    in AL, DX                       ;; make sure com port ints are masked
    or AL, 18H
    out DX, AL
 
    mov AL, 80H                     ;; set divisor latch access
    mov DX, slip_&name&_port        ;; set to port base register
    add DX, 3                       ;; inc to line control reg
    out DX, AL
 
    mov AX, slip_&name&_data.slip_speed ;; speed is actual divisor for 8250
 
    mov DX, slip_&name&_port
    out DX, AL                      ;; output low value of divisor
 
    inc DX
    mov AL, AH
    out DX, AL                      ;; output high value
 
    mov AL, 03H                     ;; set 8 bits & 1 stop bit
    inc DX
    inc DX                          ;; line control reg
    out DX, AL
 
    mov AL, 0BH                     ;; set DTR & RTS & OUT2
    inc DX                          ;; modem control reg
    out DX, AL
 
    mov AL, 03H                     ;; 8250 receive & xmit data ints
    mov DX, slip_&name&_port
    inc DX                          ;; interrupt enable reg
    out DX, AL
 
    dec DX
    in AL, DX                       ;; make sure receive buffer empty
 
        ;; no interrupts will occur until SLIP_HANDLER_DEFINE is processed
        ;; which unmasks the serial port interrupts
 
    TASK_DEFINE %slip_&name&_task, start          ;; start the task
    mov word ptr slip_&name&_read, offset null_read
 
    BUFF_DEFINE %slip_&name&_wbuff
    BUFF_DEFINE %slip_&name&_rbuff
    QUEUE_DEFINE %slip_&name&_wqueue
    QUEUE_DEFINE %slip_&name&_rqueue
 
    mov slip_&name&_data.slip_write, 0  ;; no packet to send
 
    SLIP_READ_NEW name                  ;; allocate buffer for read
 
    mov slip_&name&_data.slip_wstate, 1 ;; start of frame
    mov slip_&name&_data.slip_rstate, 1 ;; normal read (no transpose)
    mov slip_&name&_data.slip_rdrop,  0 ;; do not drop current read packet
 
    xor AX, AX                          ;; success return
    RET
 
    fail:                               ;; fail return
        xor AX, AX
        dec AX
        RET
endif
ENDM
 

;;******************************************************************************
;;   DL_IP_R_READ name, code_label
;;       DL_IP_R_READ declares that the code starting at 'code_label'
;;       should be called whenever a IP packet is read.  BX:ES is initilized
;;       to the begining of the IP packet before 'code_label' is called
;;       The code at 'code_label' should call SLIP_DL_IP_R_RETURN when it
;;       is done processing the packet.
;;       This procedure can only be called once per 'name'
;;
SLIP_DL_IP_R_READ MACRO name, code_label
    .errb <code_label>
 
    mov word ptr slip_&name&_read, offset code_label
ENDM
 
;;******************************************************************************
;;   DL_IP_R_CONT_in_BX_CX_ES name, ok
;;       DL_IP_R_CONT determines if the packet returned by R_READ in BX:ES
;;       of length CX is continuous.  If it is it jumps to 'ok' otherwise
;;       it just returns
;;
SLIP_DL_IP_R_CONT_in_BX_CX_ES_const_BX_CX_DX_BP_SI_DI_ES MACRO name, ok
    .errb <ok>
 
    jmp ok                      ;; it is always continuous
ENDM
 
;;******************************************************************************
;;   DL_IP_R_RETURN name
;;       DL_IP_R_RETURN should be executed by the READ routine to signal
;;       that it is done processing the packet.
;;
SLIP_DL_IP_R_RETURN MACRO name
    local done
    .errb <name>
 
    ;; remove entry from queue and free its buffer
 
    cli
    QUEUE_HEAD_out_SI_const_AX_BX_CX_DX_BP_DI_ES %slip_&name&_rqueue, done
    mov DI, [SI+slip_q_rend]
    BUFF_FREE_in_DI_const_AX_BX_CX_DX_BP_SI_DI_ES %slip_&name&_rbuff
    QUEUE_DEQUEUE_in_SI_const_AX_BX_CX_DX_BP_DI_ES %slip_&name&_rqueue
 
    done:
    sti
    jmp slip_&name&_continue
ENDM
 

;;******************************************************************************
;;   DL_IP_IS_BROADCAST_in_BX_ES name
;;      DL_IP_IS_BROADCAST_in_BX_ES determines if the packet pointed to
;;      by BX:ES is a broadcast and sets the zero flag if it is NOT a
;;      broadcast
;;
SLIP_DL_IP_IS_BROADCAST_in_BX_ES_const_AX_BX_CX_DX_BP_DI_ES MACRO name
    .errb <name>
 
	mov SI, word ptr [BX+ip_dst+2]   		;; is it on my subnet
    and SI, word ptr dl_ip_&name&_mask+2
    cmp SI, word ptr dl_ip_&name&_net+2
	jnz not_broad							
	mov SI, word ptr [BX+ip_dst]   		
    cmp SI, word ptr dl_ip_&name&_net
	jnz not_broad
		mov SI, word ptr [BX+ip_dst+2] 
	    cmp SI, word ptr dl_ip_&name&_ip+2
		jnz done
	not_broad:
		cmp AX, AX
	done:

ENDM
 

;;******************************************************************************
;;   DL_IP_W_ACCESS returns a pointer to an output buffer for a IP (network)
;;   packet.  The buffer is at least min(CX, dl_ip_mtu) bytes long.  The
;;   pointer is returned in DI.  If there is no buffer available
;;   this routine jumps to 'no_buffer'
;;
SLIP_DL_IP_W_ACCESS_in_CX_out_DI_ES_const_CX_BP MACRO name, no_buffer
    .errb <name>
    .errb <no_buffer>
 
        ;; check if there is space for the next one to go out
    cli
    BUFF_CHECK_in_CX_out_SI_DI_const_BX_CX_DX_BP_ES %slip_&name&_wbuff,no_buffer
    sti
    mov DI, SI
    mov SI, DS
    mov ES, SI
ENDM
 

;;******************************************************************************
;; DL_IP_W_WRITE_in_AX_CX name, broadcast
;;    DL_IP_W_WRITE actually signals the link layer to write a packet to the
;;    network.  The packet is assumed to be in the buffer returned by
;;    DL_IP_W_ACCESS.  CX is the length of the packet to send.   AX holds the
;;    last two bytes of the IP address to send the packet to.  (notice we are
;;    assuming a host portion of less than 16 bits).  if 'broadcast' is not
;;    blank, then the packet is written to the broadcast address.  AX is
;;    ignored in this case
;;
SLIP_DL_IP_W_WRITE_in_AX_CX_const_BP MACRO name, broadcast 
    local done, bad 
    .errb <name> 

    cli 
    BUFF_CHECK_in_CX_out_SI_DI_const_BX_CX_DX_BP_ES %slip_&name&_wbuff, done 
    mov BX, DI 
    QUEUE_ENQUEUE_out_DI_const_BX_CX_DX_BP_SI_ES %slip_&name&_wqueue, done 
    xchg BX, DI
    BUFF_GET_in_DI_const_AX_BX_CX_DX_BP_SI_DI_ES %slip_&name&_wbuff
    mov [BX+slip_q_wend], DI
    mov [BX+slip_q_wstart], SI
 
    cmp slip_&name&_data.slip_write, 0
    jg done
        ;; no packet currently being sent
        ;; must force xmit to begin
 
        mov slip_&name&_data.slip_wstart, SI
        mov slip_&name&_data.slip_wpos, SI
        mov slip_&name&_data.slip_wend, DI
        mov slip_&name&_data.slip_wstate, 2 ;; normal character
 
        mov DX, slip_&name&_port            ;; send start of frame character
        mov AL, FR_END                      ;; int handler will take care
        out DX, AL                          ;; of rest of packet
 
        mov slip_&name&_data.slip_write, 1  ;; set flag to let int handler know
                                            ;; packet needs sending
    done:
    sti
ENDM
 

;;******************************************************************************
;;   DL_IP_COPY_in_CX_SI_DI_ES_out_SI_DI name
;;      DL_IP_COPY_in_CX_SI_DI_ES copys a packet from the input buffer (pointed
;;      to by SI and the segment register given if IF_DECLARE) to an output
;;      buffer (pointed to by DI and dest_reg) of length CX.  It assumes the
;;      output buffer is contiguous.  (and the caller shouldn't care if the
;;      input buffer is contiguous)  COPY updates the pointers SI and DI
;;      to the end of the packet, and COPY could be called again if CX is not
;;      the total packet length (Note that CX MUST be even if you care about
;;      SI, and DI being updated properly)
;;
SLIP_DL_IP_COPY_in_CX_SI_DI_ES_out_SI_DI_const_BX_BP_ES MACRO name
    .errb <name>
 
    inc CX
    shr CX, 1
    rep
    movsw
ENDM
 

;;******************************************************************************
;;   SLIP_TASK is the read task that reads the next packet from the interface
;;   and dispatches it to the next higher protocol layer
;;
SLIP_TASK MACRO name
    local done
    .errb <name>
 
    QUEUE_HEAD_out_SI_const_AX_BX_CX_DX_BP_DI_ES %slip_&name&_rqueue, done
 
        ;; slip_&name&_read routine wants packet pointer in BX:ES
 
        mov BX, [SI+slip_q_rstart]
        mov AX, DS
        mov ES, AX
 
        jmp word ptr slip_&name&_read
 
    done:
    slip_&name&_continue:
    TASK_RETURN %slip_&name&_task
ENDM

;;*****************************************************************************
;; SLIP_CHECK_PORT
;;      checks to see if the serial port specified is available. A port not
;;      available is assumed to return FFH from the line status register.
;;      If not it jumps to 'fail'.
;;
SLIP_CHECK_PORT MACRO name, fail
 
    mov DX, slip_&name&_port
    add DX, 5                       ;; line status register
    in AL, DX
    cmp AL, 0FFH                    ;; is result all ones?
    je fail                         ;; yes!
 
    ;; no!! serial port not available!!
ENDM

;;*****************************************************************************
;;*****************************************************************************
;;************ROUTINES BELOW DEFINE THE SLIP INTERRUPT HANDLER*****************
;;*****************************************************************************
;;*****************************************************************************
;; SLIP_HANDLER_DEFINE
;;      this defines the SLIP interrupt handler.  All initialization for
;;      the individual ports is done in SLIP_&name&_DEFINE.  Initialize
;;      the interrupt controller and install interrupt handler.
;;      'name' is the name of the SLIP interface that it making this request
;;      for the handler.  The handler will only be installed if this is the
;;      last of the delcared SLIP interface
;;
SLIP_CHECK_DEFINE MACRO name
    ifdef slip_&name&_declared
        slip_handler_declare = name
    endif
ENDM
 
SLIP_HANDLER_DEFINE MACRO name
    local around
 
    slip_handler_declare = 0
    irp idx, <1,2,3,4,5,6,7,8>
        SLIP_CHECK_DEFINE idx
    endm
 
    if slip_handler_declare eq name
        mov DX, offset cs:slip_handler
        push DS
        mov AX, 11
        mov AH, 25H
        push CS
        pop  DS
        int 21H                     ;; install interrupt handler for com1 & 3
        mov AX, 12
        mov AH, 25H
        int 21H                     ;; install interrupt handler for com2 & 4
        pop DS
        mov word ptr CS:slip_ds, DS ;; save ds for use in int handler
        sti                         ;; allow interrupts
 
        in AL, IntMsk               ;; unmask com port interrupts
        and AL, 0E7H
        out IntMsk, AL
 
        jmp around
            slip_ds DW  0
 
            slip_handler:
                cli                 ;; interrupts off
                push AX             ;; save registers
                push BX
                push CX
                push DX
                push SI
                push DI
                push BP
                push ES
                push DS
                mov DS, word ptr CS:slip_ds
 
                ;; install interrupt handling code for each slip line
 
                irp idx, <1,2,3,4,5,6,7,8>
                    SLIP_PORT_DEFINE idx
                endm
 
                mov AL, 20H
                out IntCtl, AL      ;; tell 8259 we are done
                pop DS
                pop ES
                pop BP
                pop DI
                pop SI
                pop DX
                pop CX
                pop BX
                pop AX
                iret
        around:
    endif
ENDM
 

;;*****************************************************************************
;; SLIP_PORT_DEFINE name
;;      this installs the actual slip port handling routines.  If this DL
;;      is a slip line, install handler code.
;;
SLIP_PORT_DEFINE MACRO name
    local done, chk_xmit, chk_rcv, no_send, anymore, chk_again
    local chk_w
    .errb <name>
 
    ifdef slip_&name&_declared
        anymore:
            mov DX, slip_&name&_port
            inc DX
            inc DX      ;; IIR
            in AL, DX
            test AL, 01H
            jnz chk_w    ;; no int
            and AX, 06H
            cmp AX, 04H
            jne chk_xmit
            SLIP_READ_CHAR name
            jmp chk_again
        chk_xmit:
            cmp AX, 02H
            jne chk_again
            cmp slip_&name&_data.slip_write, 0
            je chk_again
            SLIP_SEND_CHAR name
        chk_again:
            jmp anymore

        chk_w:
            cmp slip_&name&_data.slip_write, 0
            jz done
            mov DX, slip_&name&_port+5
            in AL, DX
            test AL, 40H
            jz done
            SLIP_SEND_CHAR name
        done:

    endif
ENDM
 

;;*****************************************************************************
;; SLIP_READ_CHAR name
;;      this routine reads the next character from the port and places it
;;      into the packet buffer.  the packet pointer is updated as necessary.
;;      if done it will enqueue the packet and get another buffer
;;
SLIP_READ_CHAR MACRO name
    local done, not_end, not_esc, not_normal, next_char, good_packet, not_full
    local len_ok
    .errb <name>
 
        mov DX, slip_&name&_port
        in AL, DX                   ;; read character
        cmp AL, FR_END              ;; is char end or start of frame
        jne not_end                 ;; no!
        mov BX, slip_&name&_data.slip_rpos
        cmp BX, slip_&name&_data.slip_rstart
                                    ;; if comparison is true, nothing is in
                                    ;; the packet buffer and therefore it
        je done                     ;; must be start of packet
 
        ;; end of packet
        cmp slip_&name&_data.slip_rdrop, 1
        jne good_packet             ;; will fail if an overrun occured while
                                    ;; reading packet
 
        SLIP_READ_DROP name         ;; drop last packet
 
        jmp done
      good_packet:
        SLIP_READ_ENQUEUE name      ;; place last packet on read queue
        SLIP_READ_NEW name          ;; get new read buffer
 
        jmp done
    not_end:
        cmp slip_&name&_data.slip_rstate, 1 ;; normal character?
        jg not_normal               ;; no!
        cmp AL, FR_ESC              ;; check for ESC?
        jne next_char               ;; no!
        mov slip_&name&_data.slip_rstate, 2 ;; have ESC, get next char
        jmp SHORT done
      not_normal:
        mov slip_&name&_data.slip_rstate, 1 ;; have ESCaped char, next is norm
        cmp AL, T_FR_ESC            ;; is translated ESC?
        jne not_esc                 ;; no!
        mov AL, FR_ESC              ;; yes, save actual ESC
        jmp SHORT next_char
      not_esc:
        mov AL, FR_END              ;; save actual END
 
    next_char:
        mov BX, slip_&name&_data.slip_rpos
        mov byte ptr [BX], AL       ;; save character
        inc BX                      ;; update pointer
        cmp BX, slip_&name&_data.slip_rend
        jna len_ok
        mov slip_&name&_data.slip_rdrop, 1
        mov BX, slip_&name&_data.slip_rstart
      len_ok:
        mov slip_&name&_data.slip_rpos, BX
    done:
ENDM
 

;;*****************************************************************************
;; SLIP_READ_ENQUEUE name
;;      this routine equeues the new packet on the read queue and sets
;;      the read flag so that SLIP_TASK will know a new packet is available
;;
SLIP_READ_ENQUEUE MACRO name
    local done,bad
    .errb <name>
 
        QUEUE_ENQUEUE_out_DI_const_BX_CX_DX_BP_SI_ES %slip_&name&_rqueue, done
        mov BX, DI
 
        ;; the BUFF_GET is done when packet buffer is allocated and then done
        ;; here after entire packet is read.  This allows use to only use
        ;; the space necessary for this packet and return that not needed.
 
        mov DI, slip_&name&_data.slip_rpos
        BUFF_GET_in_DI_const_AX_BX_CX_DX_BP_SI_DI_ES %slip_&name&_rbuff
 
        mov SI, slip_&name&_data.slip_rstart
        mov [BX+slip_q_rstart], SI
        mov [BX+slip_q_rend], DI
    done:
ENDM
 
;;*****************************************************************************
;; SLIP_READ_NEW name
;;      this routine allocates a new buffer for reading and sets the read
;;      packet pointers to the new buffer.
;;
SLIP_READ_NEW MACRO name
    local done, bad
    .errb <name>
 
        ;; allocate a new buffer for next incoming packet
 
        mov CX, dl_ip_&name&_mtu    ;; MTU for SLIP lines
 
        BUFF_CHECK_in_CX_out_SI_DI_const_BX_CX_DX_BP_ES %slip_&name&_rbuff, done
        BUFF_GET_in_DI_const_AX_BX_CX_DX_BP_SI_DI_ES %slip_&name&_rbuff
 
        ;; setup incoming packet pointers
 
        mov slip_&name&_data.slip_rpos, SI
        mov slip_&name&_data.slip_rstart, SI
        mov slip_&name&_data.slip_rend, DI
        mov slip_&name&_data.slip_rstate, 1
 
    done:
ENDM
 
;;*****************************************************************************
;; SLIP_READ_DROP name
;;      this routine reset the current read pointers to the beginning of
;;      the buffer to drop the packet.  In effect, the current packet is
;;      dropped.  This routine may also be of use if error checking were
;;      done on the packets and it failed.
;;
SLIP_READ_DROP MACRO name
    .errb <name>
 
    mov slip_&name&_data.slip_rdrop, 0
    mov SI, slip_&name&_data.slip_rstart
    mov slip_&name&_data.slip_rpos, SI  ;; reset read position
    mov slip_&name&_data.slip_rstate, 1
 
ENDM
 

;;*****************************************************************************
;; SLIP_SEND_CHAR name
;;      this routine sends to next character and updates the write pointers.
;;      if this is the end of packet, the status is set as done and the next
;;      packet is dequeued to be sent.
;;
SLIP_SEND_CHAR MACRO name
    local done, start_frame, escape, escape_needed, next_packet, not_end
    .errb <name>
 
        ;; send next character
        mov DX, slip_&name&_port
        cmp slip_&name&_data.slip_wstate, 2 ;; start of frame?
    jl start_frame      ;; if state = 1
    jg escape       ;; if state = 3
        ;; if this is a normal character in the middle of the frame
        mov BX, slip_&name&_data.slip_wpos
        cmp BX, slip_&name&_data.slip_wend  ;; end of frame?
        jge next_packet

        mov AL, byte ptr [BX]
        cmp AL, FR_END              ;; is char an END?
        je escape_needed
        cmp AL, FR_ESC              ;; is char an ESC?
        je escape_needed
        out DX, AL
        inc BX
        mov slip_&name&_data.slip_wpos, BX
        jmp done
        escape_needed:
        mov AL, FR_ESC              ;; yes, send ESC
        out DX, AL
        mov slip_&name&_data.slip_wstate, 3 ;; we did an escape
        jmp done

        next_packet:
        SLIP_SEND_NEXT name         ;; yes, release packet and get next
        mov AL, FR_END
        out DX, AL                  ;; send end of frame
        jmp done

    escape:     ;; we are in the middle of an escape sequence, finish it
        mov BX, slip_&name&_data.slip_wpos
        mov AH, byte ptr [BX]

        mov AL, T_FR_ESC
        cmp AH, FR_END              ;; is char an END?
        jne not_end
        mov AL, T_FR_END
        not_end:
        out DX, AL                  ;; send second half of escape sequence
        inc BX
        mov slip_&name&_data.slip_wpos, BX
        mov slip_&name&_data.slip_wstate, 2
        jmp done

    start_frame:
        mov AL, FR_END                      ;; yes, send SOF
        out DX, AL
        mov slip_&name&_data.slip_wstate, 2
    done:
ENDM
 

;;*****************************************************************************
;; SLIP_SEND_NEXT name
;;      this routine discards the packet just sent and checks to see
;;      if another packet needs to be sent.
;;
SLIP_SEND_NEXT MACRO name
    local done
    .errb <name>
 
    mov slip_&name&_data.slip_write, 0
 
    ;; release last packet sent
 
    QUEUE_HEAD_out_SI_const_AX_BX_CX_DX_BP_DI_ES %slip_&name&_wqueue, done
    mov DI, [SI+slip_q_wend]
    BUFF_FREE_in_DI_const_AX_BX_CX_DX_BP_SI_DI_ES %slip_&name&_wbuff
    QUEUE_DEQUEUE_in_SI_const_AX_BX_CX_DX_BP_DI_ES %slip_&name&_wqueue
 
    ;; check for another packet which needs sending
 
    QUEUE_HEAD_out_SI_const_AX_BX_CX_DX_BP_DI_ES %slip_&name&_wqueue, done
    mov BX, SI
    mov SI, [BX+slip_q_wstart]
    mov DI, [BX+slip_q_wend]
    mov slip_&name&_data.slip_wpos, SI
    mov slip_&name&_data.slip_wstart, SI
    mov slip_&name&_data.slip_wend, DI
    mov slip_&name&_data.slip_write, 1
    mov slip_&name&_data.slip_wstate, 1
 
    done:                           ;; no more packets to send
ENDM
