;;************************************************************************* 
;;                     log.inc       log.inc
;;*************************************************************************
;;
;;  Copyright (C) 1989 Northwestern University, Vance Morrison
;;
;;
;; Permission to view, compile, and modify for LOCAL (intra-organization) 
;; USE ONLY is hereby granted, provided that this copyright and permission 
;; notice appear on all copies.  Any other use by permission only.
;;
;; Northwestern University makes no representations about the suitability 
;; of this software for any purpose.  It is provided "as is" without expressed 
;; or implied warranty.  See the copywrite notice file for complete details.
;;
;;*****************************************************************************
;;
;; log.asm a module that allows significant events to be logged.  It does
;; this by using the BSD UNIX syslog facility. 
;;
;;      This module provides the following functions
;;
;; Routines provided by this module
;;
;;   LOG_DECLARE name, facility, udp
;;   LOG_DEFINE_in_AX_BX name, log_host
;;   LOG_PRINT name, type, severity, string
;;   LOG_PRINT_REG_HEX  name, type, severity, string, reg
;;   LOG_PRINT_REG_DEC  name, type, severity, string, reg
;;   LOG_PRINT_INET_in_AX_BX name, type, severity, string
;;
;;  
;;
;;*****************************************************************************

;;*****************************************************************************
;; definition of LOG packet structure

SYSLOG_PORT       = 514

log_data STRUC
    log_host    DD
    log_ip_addr DD
    log_mask    DW
    log_pri     DW
log_data ENDS

;; posible severity levels (see /usr/include/syslogd.h on a unix host)
L_EMERG = 0
L_ALERT = 1
L_CRIT  = 2
L_ERR   = 3
L_WARNING = 4
L_NOTICE  = 5
L_INFO    = 6
L_DEBUG   = 7

;; possible facility types
L_SYS     = 0       ;; system messages
L_ROUTE   = 1       ;; routing messages
L_MON     = 2       ;; monitoring messages
L_ATALK   = 3       ;; localtalk messages

;; posible facilities levels (see /usr/include/syslogd.h on a unix host)
LOG_USER    = 1
LOG_LOCAL0  = 16    ;; this is the recommended one
LOG_LOCAL1  = 17
LOG_LOCAL2  = 18
LOG_LOCAL3  = 19
LOG_LOCAL4  = 20
LOG_LOCAL5  = 21
LOG_LOCAL6  = 22
LOG_LOCAL7  = 23


;;******************************************************************************
;;  LOG_DECLARE name, facility, udp
;;       LOG_DECLARE delcares a new loging object called 'name'. 
;;       'facility' is the numeric facility number that is used to
;;       identify the log messages to syslogd.  16 is local0.  For
;;       others look in your /usr/include/syslog.h file.
;;       'udp' is the udp socket listener object.
;;
LOG_DECLARE   MACRO  name, facility, udp, timer
    .errb <timer>

    .DATA
    global log_&name&_data:log_data
    log_&name&_facility = facility
    log_&name&_udp_sock = (name*100+1)
    log_&name&_timer    = timer

    .CODE
    global log_&name&_define_in_AX_BX_SI:near
    global log_&name&_print_in_DX_SI_const_ALL:near
    global log_&name&_print_hex_in_AX_DX_SI_const_ALL:near
    global log_&name&_print_dec_in_AX_DX_SI_const_ALL:near
    global log_&name&_print_inet_in_AX_BX_DX_SI_const_ALL:near

    global log_&name&_send_syslog_buff_in_CX:near
    global log_&name&_get_syslog_buff_in_DL_out_AX_DI_ES:near
    global log_dec_ascii_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES:near
    global log_hex_ascii_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES:near
    global log_print_str_in_CX_SI_DI_ES_out_DI_CX_const_BX_DX_BP_ES:near

    UDP_SOCK_DECLARE %log_&name&_udp_sock, udp, SYSLOG_PORT
ENDM


;;******************************************************************************
;;   LOG_DEFINE_in_AX_BX name
;;       LOG_DEFINE sets aside the memory and acutally does the 
;;       initialization for the routeing objects 'name'.  AX holds the
;;       priority level (only messages with a priority less than this
;;       value will be logged).  BX holds the logging bitmask  Only those 
;;       loging types that are  '0' in this  mask will actually be logged.  
;;       Thus the bitmap DISABLES loging.
;;
LOG_DEFINE_in_AX_BX   MACRO   name, log_host
    mov SI, offset log_host
    call log_&name&_define_in_AX_BX_SI
ENDM

LOG_REAL_DEFINE_in_AX_BX_SI MACRO name
    local around, read_code, timer_code, startup, no_monitor, time_ok
    .errb <name>

    .DATA
    log_&name&_data log_data

    .CODE
    jmp around
                ;; 'external' routines
        log_&name&_print_in_DX_SI_const_ALL:
           LOG_REAL_PRINT_in_DX_SI_const_ALL name
            RET
        log_&name&_print_hex_in_AX_DX_SI_const_ALL:
           LOG_REAL_PRINT_HEX_in_AX_DX_SI_const_ALL name
            RET
        log_&name&_print_dec_in_AX_DX_SI_const_ALL:
           LOG_REAL_PRINT_DEC_in_AX_DX_SI_const_ALL name
            RET
        log_&name&_print_inet_in_AX_BX_DX_SI_const_ALL:
           LOG_REAL_PRINT_INET_in_AX_BX_DX_SI_const_ALL name
            RET

            ;; 'internal' routines
        log_print_str_in_CX_SI_DI_ES_out_DI_CX_const_BX_DX_BP_ES:
           LOG_REAL_PRINT_STR_in_CX_SI_DI_ES_out_DI_CX_const_BX_DX_BP_ES name
            RET
        log_dec_ascii_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES:
            LOG_REAL_DEC_ASCII_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES name
            RET
        log_hex_ascii_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES:
            LOG_REAL_HEX_ASCII_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES name
            RET
        log_&name&_send_syslog_buff_in_CX:
            LOG_REAL_SEND_SYSLOG_BUFF_in_CX name
            RET
        log_&name&_get_syslog_buff_in_DL_out_AX_DI_ES:
            LOG_REAL_GET_SYSLOG_BUFF_in_DL_out_AX_DI_ES name
            RET

        read_code:          ;; do nothing on syslog packet in
            RET

        startup:            ;; issue a startup message
            LOG_PRINT %mylog, L_SYS, L_INFO, <PCroute starting up>
            TIMER_RETURN %log_&name&_timer 

        timer_code:
            ;; note that the timer is limited to 32K ticks
        LOG_PRINT name, L_MON, L_NOTICE, <Router is up>
        mov AX, 15*60*18            ;; go off every 15 min
        cmp byte ptr log_&name&_data.log_pri, L_DEBUG+1
        jbe time_ok
            mov AX, 1*60*18         ;; go off every 1 min
        time_ok:
        TIMER_MARK_in_AX_const_CX_BP_ES %log_&name&_timer, timer_code
        TIMER_RETURN %log_&name&_timer 
    around:
    log_&name&_define_in_AX_BX_SI:
    mov log_&name&_data.log_pri , AX
    mov log_&name&_data.log_mask, BX
    lodsw                                   ;; save the admin host
    mov word ptr log_&name&_data.log_host, AX
    lodsw
    mov word ptr log_&name&_data.log_host+2, AX

    UDP_SOCK_DEFINE %log_&name&_udp_sock, read_code

    test word ptr log_&name&_data.log_mask, (1 SHL L_MON)
    jnz no_monitor
        mov AX, 18*5
        TIMER_MARK_in_AX_const_CX_BP_ES %log_&name&_timer, timer_code
    no_monitor:

    mov AX, 0                       ;; send startup message
    TIMER_MARK_in_AX_const_CX_BP_ES %log_&name&_timer, startup
                                    ;; since the first one will be discarded
                                    ;; since the arp cache is empty we send
                                    ;; it again after it has been filled

    mov AX, 10                      ;; give it time to get a rip packet
    TIMER_MARK_in_AX_const_CX_BP_ES %log_&name&_timer, startup
    RET
ENDM


;;**************************************************************************
;;   LOG_PRINT  name, type, severity, string
;;
;; LOG_PRINT logs the message 'string' to the syslogd with a serverity of
;;    'severity'.  a severity should be on of those defined above
;;    Note that this module violates convention and saves ALL registers
;;
LOG_PRINT  MACRO name, type, severity, string
   local mystring
   .errb <string>

    .DATA
mystring    db '&string', 0

   .CODE
   push SI
   push DX

   mov SI, offset mystring
   mov DL, severity
   mov DH, type
   call log_&name&_print_in_DX_SI_const_ALL

   pop DX
   pop SI
ENDM


;;**************************************************************************
;;   LOG_PRINT_REG_HEX  name, type, severity, string, reg
;;
;; LOG_PRINT_REG_HEX logs the message 'string' to the syslogd with a serverity
;;    of 'severity'.  a severity should be on of those defined above
;;    Note that this module violates convention and saves ALL registers
;;    reg is then printed after the string in Hexidecimal notation
;;
LOG_PRINT_REG_HEX  MACRO name, type, severity, string, reg
   local mystring
   .errb <string>

    .DATA
mystring    db '&string', 0

   .CODE
   push SI
   push AX
   push DX

   mov AX, reg
   mov SI, offset mystring
   mov DL, severity
   mov DH, type
   call log_&name&_print_hex_in_AX_DX_SI_const_ALL

   pop DX
   pop AX
   pop SI
ENDM


;;**************************************************************************
;;   LOG_PRINT_REG_DEC  name, type, severity, string, reg
;;
;; LOG_PRINT_REG_DEC logs the message 'string' to the syslogd with a serverity
;;    of 'severity'.  a severity should be on of those defined above
;;    Note that this module violates convention and saves ALL registers
;;    reg is then printed after the string in Hexidecimal notation
;;
LOG_PRINT_REG_DEC  MACRO name, type, severity, string, reg
   local mystring
   .errb <reg>

    .DATA
mystring    db '&string', 0

   .CODE
   push SI
   push AX
   push DX

   mov AX, reg
   mov SI, offset mystring
   mov DL, severity
   mov DH, type
   call log_&name&_print_dec_in_AX_DX_SI_const_ALL

   pop DX
   pop AX
   pop SI
ENDM


;;**************************************************************************
;;   LOG_PRINT_INET_in_AX_BX  name, type, severity, string
;;
;; LOG_PRINT_INET_in_AX_BX logs the message 'string' to syslogd with a severity
;;    of 'severity'.  a severity should be on of those defined above
;;    Note that this module violates convention and saves ALL registers
;;    the register pair AX:BX is then printed in internet dot notation
;;
LOG_PRINT_INET_in_AX_BX  MACRO name, type, severity, string
   local mystring
   .errb <string>

    .DATA
mystring    db '&string', 0

   .CODE
   push SI
   push DX

   mov SI, offset mystring
   mov DL, severity
   mov DH, type
   call log_&name&_print_inet_in_AX_BX_DX_SI_const_ALL

   pop DX
   pop SI
ENDM


;;**************************************************************************
;; Does the real work in doing printing
;;
LOG_REAL_PRINT_in_DX_SI_const_ALL MACRO name
    local done
    .errb <name>

    push AX
    push BX
    push CX
    push DX
    push BP
    push SI
    push DI
    push ES

    mov CL, DH          ;; should we log this
    mov DI, 1
    shl DI, CL
    test DI, log_&name&_data.log_mask
    jnz done
    cmp DL, byte ptr log_&name&_data.log_pri
    jae done

    push SI
    call log_&name&_get_syslog_buff_in_DL_out_AX_DI_ES
    pop SI
    or AX, AX
    jnz done

    xor CX, CX
    call log_print_str_in_CX_SI_DI_ES_out_DI_CX_const_BX_DX_BP_ES
    inc CX              ;; count the null terminator

    call log_&name&_send_syslog_buff_in_CX

    done:
    pop ES
    pop DI
    pop SI
    pop BP
    pop DX
    pop CX
    pop BX
    pop AX
ENDM


;;**************************************************************************
;; Does the real work in doing printing a register in hex
;;
LOG_REAL_PRINT_HEX_in_AX_DX_SI_const_ALL MACRO name
    local done
    .errb <name>

    push AX
    push BX
    push CX
    push DX
    push BP
    push SI
    push DI
    push ES

    mov CL, DH          ;; should we log this
    mov DI, 1
    shl DI, CL
    test DI, log_&name&_data.log_mask
    jnz done
    cmp DL, byte ptr log_&name&_data.log_pri
    jae done

    push AX
    push SI
    call log_&name&_get_syslog_buff_in_DL_out_AX_DI_ES
    pop SI
    pop BX
    or AX, AX
    jnz done

    xor CX, CX
    call log_print_str_in_CX_SI_DI_ES_out_DI_CX_const_BX_DX_BP_ES

    mov AX, BX
    mov BP, DI 
    mov DX, CX              ;; save CX
    call log_hex_ascii_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES
    mov CX, DX              ;; restore CX
    sub BP, DI
    sub CX, BP              ;; add in the length (negative of negative)

    mov AL, 0
    stosb
    inc CX              ;; count the null terminator

    call log_&name&_send_syslog_buff_in_CX
    done:
    pop ES
    pop DI
    pop SI
    pop BP
    pop DX
    pop CX
    pop BX
    pop AX
ENDM


;;**************************************************************************
;; Does the real work in doing printing a register in decimal
;;
LOG_REAL_PRINT_DEC_in_AX_DX_SI_const_ALL MACRO name
    local done
    .errb <name>

    push AX
    push BX
    push CX
    push DX
    push BP
    push SI
    push DI
    push ES

    mov CL, DH          ;; should we log this
    mov DI, 1
    shl DI, CL
    test DI, log_&name&_data.log_mask
    jnz done
    cmp DL, byte ptr log_&name&_data.log_pri
    jae done

    push AX
    push SI
    call log_&name&_get_syslog_buff_in_DL_out_AX_DI_ES
    pop SI
    pop BX

    or AX, AX
    jnz done

    xor CX, CX
    call log_print_str_in_CX_SI_DI_ES_out_DI_CX_const_BX_DX_BP_ES

    mov AX, BX
    mov BP, DI 
    mov DX, CX              ;; save CX
    call log_dec_ascii_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES
    mov CX, DX              ;; restore CX
    sub BP, DI
    sub CX, BP              ;; add in the length (negative of negative)

    mov AL, 0
    stosb
    inc CX              ;; count the null terminator

    call log_&name&_send_syslog_buff_in_CX
    done:
    pop ES
    pop DI
    pop SI
    pop BP
    pop DX
    pop CX
    pop BX
    pop AX
ENDM


;;**************************************************************************
;; Does the real work in doing printing an internet dot notation
;;
LOG_REAL_PRINT_INET_in_AX_BX_DX_SI_const_ALL MACRO name
    local done
    .errb <name>

    push AX
    push BX
    push CX
    push DX
    push BP
    push SI
    push DI
    push ES

    mov word ptr log_&name&_data.log_ip_addr, AX
    mov word ptr log_&name&_data.log_ip_addr+2, BX

    mov CL, DH          ;; should we log this
    mov BX, 1
    shl BX, CL
    test BX, log_&name&_data.log_mask
    jnz done
    cmp DL, byte ptr log_&name&_data.log_pri
    jae done

    push SI
    call log_&name&_get_syslog_buff_in_DL_out_AX_DI_ES
    pop SI
    or AX, AX
    jnz done

    xor CX, CX
    call log_print_str_in_CX_SI_DI_ES_out_DI_CX_const_BX_DX_BP_ES

    push CX             ;; save CX
    push DI

    mov CX, 15          ;; maximum size of IP addr
    mov AL, ' '
    rep
    stosb

    mov SI, offset log_&name&_data.log_ip_addr
    IP_ASCII_in_SI_DI_ES_out_DI_const_BX_BP_ES 

    pop DI
    pop CX              ;; restore CX

    add DI, 15          ;; size of IP address 
    add CX, 15          ;; size of IP address 

    mov AL, 0
    stosb
    inc CX              ;; count the null terminator

    call log_&name&_send_syslog_buff_in_CX
    done:
    pop ES
    pop DI
    pop SI
    pop BP
    pop DX
    pop CX
    pop BX
    pop AX
ENDM


;;*************************************************************************
;; LOG_REAL_PRINT_STR_in_CX_SI_DI_ES_out_DI_CX_const_AX_BX_DX_BP_ES name
;;      copies the null terminated string in SI:DS to DI:ES, updates DI
;;      to point to the end of the new string.  CX is the present length
;;      of the output string and it is updated to the new length.
;;
LOG_REAL_PRINT_STR_in_CX_SI_DI_ES_out_DI_CX_const_BX_DX_BP_ES MACRO name
    local copy_loop, copy_loop_done, len_ok
    .errb <name>

    copy_loop:
        lodsb
        stosb
        inc CX

        cmp CX, 1000
        jb len_ok
            xor AL, AL
            stosb
            inc CX
            jmp copy_loop_done
        len_ok:
        or AL, AL
    jnz copy_loop
    copy_loop_done:
    dec CX                  ;; point to end of string
    dec DI
ENDM


;;*************************************************************************
;; LOG_REAL_SEND_SYSLOG_BUFF_in_CX sends the filled syslog buffer to 
;; the loging host. CX is the length of the messages.  CX MUST be less
;; than 256 (since syslog prints it all on one line, 80 is better)
;;
LOG_REAL_SEND_SYSLOG_BUFF_in_CX MACRO name
    .errb <name>

    add CX, 5
    UDP_SOCK_W_WRITE_in_CX %log_&name&_udp_sock
ENDM


;;*************************************************************************
;; LOG_REAL_GET_SYSLOG_BUFF_in_DL_out_AX_DI_ES gets a buffer for writing a 
;; Log message to the syslog host of priority 'DL' and returns it in DI:ES.  
;; If successful AX is zero
;;
LOG_REAL_GET_SYSLOG_BUFF_in_DL_out_AX_DI_ES MACRO name
    local done, get_buff
    .errb <name>

    push DX                     ;; save the severity
    mov CX, 256
    mov DX, SYSLOG_PORT     
    mov AX, word ptr log_&name&_data.log_host
    mov BX, word ptr log_&name&_data.log_host+2
    or AX, AX
    jnz get_buff
    or BX, BX
    jnz get_buff
        dec AX                 ;; make non-zero
        jmp done
    get_buff:
    UDP_SOCK_W_ACCESS_in_AX_BX_CX_DX_out_AX_DI_ES %log_&name&_udp_sock
    pop DX
    or AX, AX
    jnz done

    mov AL, '<'
    stosb

    mov AX, DX                  ;; AX = severity
    xor AH, AH
    add AX, log_&name&_facility*8
    mov BL, 10
    div BL                      ;; AL = AX / 10     AH = AX mod 10
    mov BH, AH
    xor AH, AH
    div BL                      ;; AL = AX / 10     AH = AX mod 10
    add AL, '0'
    add AH, '0'
    stosw
    mov AL, BH
    add AL, '0'
    stosb
    mov AL, '>'
    stosb
    xor AX, AX                  ;; success AX = 0
    done:
ENDM


;;*************************************************************************
;; LOG_REAL_HEX_ASCII_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES name
;;      This routine converts AX to Hex and puts the represenation in
;;      the buffer DI:ES.  DI points to the end of the buffer
;; 
LOG_REAL_HEX_ASCII_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES MACRO name
    local print_it, hex_loop
    .errb <name>

    add DI, 3
    std             ;; count down

    mov CX, 4
    hex_loop:
        mov SI, AX
        shr SI, 1
        shr SI, 1
        shr SI, 1
        shr SI, 1
        and AL, 0FH

        add AL, '0'
        cmp AL, '9'
        jbe print_it
            add AL, ('A' - '0' - 10)
        print_it:
        stosb
        mov AX, SI
    loop hex_loop
    cld                 ;; count up again
    add DI, 5           ;; make point to the end of the string

    mov AL, 'H'         ;; indicate that it is HEX
    stosb
ENDM


;;**************************************************************************
;; LOG_REAL_DEC_ASCII_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES name
;;      This routine converts AX to decimal and puts the represenation in
;;      the buffer DI:ES.  DI points to the end of the buffer
;; 
LOG_REAL_DEC_ASCII_in_AX_DI_ES_out_DI_const_BX_DX_BP_ES MACRO name
    local print_it, dec_loop
    .errb <name>

    push DX
    add DI, 4
    std             ;; count down

    mov SI, 10
    mov CX, 5
    dec_loop:
        xor DX, DX
        idiv SI         ;; AX = AX / 10     DX = DX mod 10
        xchg AX, DX
        add AL, '0'
        cmp AL, '0'
        jnz print_it
            cmp CX, 5
            jz print_it
            mov AL, ' '
        print_it:
        stosb
        mov AX, DX
    loop dec_loop
    cld                 ;; count up again
    add DI, 6           ;; make point to the end of the string
    pop DX
ENDM

