#ifndef lint
static char *RCSid = "$Header: /tmp_mnt/net/sparky.a/davy/system/nfswatch/RCS/rpcfilter.c,v 1.2 90/08/17 15:47:44 davy Exp $";
#endif

/*
 * rpcfilter.c - filter RPC packets.
 *
 * David A. Curry
 * SRI International
 * 333 Ravenswood Avenue
 * Menlo Park, CA 94025
 * davy@itstd.sri.com
 *
 * $Log:	rpcfilter.c,v $
 * Revision 1.2  90/08/17  15:47:44  davy
 * NFSWATCH Version 2.0.
 * 
 * Revision 1.1  88/11/29  11:20:51  davy
 * NFSWATCH Release 1.0
 * 
 */
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <rpc/auth.h>
#include <rpc/clnt.h>
#include <rpc/rpc_msg.h>
#include <rpc/pmap_clnt.h>
#include <rpc/svc.h>
#include <netdb.h>
#include <errno.h>
#include <stdio.h>

#define NFSSERVER	1

#ifdef sun
#include <sys/vfs.h>
#endif /* sun */
#ifdef ultrix
#include <sys/types.h>
#include <sys/time.h>
#endif /* ultrix */
#include <nfs/nfs.h>

#include "nfswatch.h"
#include "externs.h"
#include "rpcdefs.h"

/*
 * NFS procedure types and XDR argument decoding routines.
 */
static struct nfs_proc nfs_procs[] = {
/* RFS_NULL (0)		*/
	NFS_READ,	xdr_void,	0,
/* RFS_GETATTR (1)	*/
	NFS_READ,	xdr_fhandle,	sizeof(fhandle_t),
/* RFS_SETATTR (2)	*/
	NFS_WRITE,	xdr_saargs,	sizeof(struct nfssaargs),
/* RFS_ROOT (3)		*/
	NFS_READ,	xdr_void,	0,
/* RFS_LOOKUP (4)	*/
	NFS_READ,	xdr_diropargs,	sizeof(struct nfsdiropargs),
/* RFS_READLINK (5)	*/
	NFS_READ,	xdr_fhandle,	sizeof(fhandle_t),
/* RFS_READ (6)		*/
	NFS_READ,	xdr_readargs,	sizeof(struct nfsreadargs),
/* RFS_WRITECACHE (7)	*/
	NFS_WRITE,	xdr_void,	0,
/* RFS_WRITE (8)	*/
	NFS_WRITE,	xdr_writeargs,	sizeof(struct nfswriteargs),
/* RFS_CREATE (9)	*/
	NFS_WRITE,	xdr_creatargs,	sizeof(struct nfscreatargs),
/* RFS_REMOVE (10)	*/
	NFS_WRITE,	xdr_diropargs,	sizeof(struct nfsdiropargs),
/* RFS_RENAME (11)	*/
	NFS_WRITE,	xdr_rnmargs,	sizeof(struct nfsrnmargs),
/* RFS_LINK (12)	*/
	NFS_WRITE,	xdr_linkargs,	sizeof(struct nfslinkargs),
/* RFS_SYMLINK (13)	*/
	NFS_WRITE,	xdr_slargs,	sizeof(struct nfsslargs),
/* RFS_MKDIR (14)	*/
	NFS_WRITE,	xdr_creatargs,	sizeof(struct nfscreatargs),
/* RFS_RMDIR (15)	*/
	NFS_WRITE,	xdr_diropargs,	sizeof(struct nfsdiropargs),
/* RFS_READDIR (16)	*/
	NFS_READ,	xdr_rddirargs,	sizeof(struct nfsrddirargs),
/* RFS_STATFS (17)	*/
	NFS_READ,	xdr_fhandle,	sizeof(fhandle_t)
};

/*
 * rpc_filter - pass off RPC packets to other filters.
 */
void
rpc_filter(data, length, src, dst)
register u_long src, dst;
register u_int length;
register char *data;
{
	register struct rpc_msg *msg;

	msg = (struct rpc_msg *) data;

	/*
	 * See which "direction" the packet is going.  We
	 * can classify RPC CALLs, but we cannot classify
	 * REPLYs, since they no longer have the RPC
	 * program number in them (sigh).
	 */
	switch (ntohl(msg->rm_direction)) {
	case CALL:			/* RPC call			*/
		rpc_callfilter(data, length, src, dst);
		break;
	case REPLY:			/* RPC reply			*/
		rpc_replyfilter(data, length, src, dst);
		break;
	default:			/* probably not an RPC packet	*/
		break;
	}
}

/*
 * rpc_callfilter - filter RPC call packets.
 */
void
rpc_callfilter(data, length, src, dst)
register u_long src, dst;
register u_int length;
register char *data;
{
	register struct rpc_msg *msg;

	msg = (struct rpc_msg *) data;

	/*
	 * Decide what to do based on the program.
	 */
	switch (ntohl(msg->rm_call.cb_prog)) {
	case RPC_NFSPROG:
		nfs_filter(data, length, src, dst);
		break;
	case RPC_YPPROG:
	case RPC_YPBINDPROG:
	case RPC_YPPASSWDPROG:
		pkt_counters[PKT_YELLOWPAGES].pc_interval++;
		pkt_counters[PKT_YELLOWPAGES].pc_total++;
		break;
	case RPC_MOUNTPROG:
		pkt_counters[PKT_NFSMOUNT].pc_interval++;
		pkt_counters[PKT_NFSMOUNT].pc_total++;
		break;
#ifdef notdef
	case RPC_PMAPPROG:
	case RPC_RSTATPROG:
	case RPC_RUSERSPROG:
	case RPC_DBXPROG:
	case RPC_WALLPROG:
	case RPC_ETHERSTATPROG:
	case RPC_RQUOTAPROG:
	case RPC_SPRAYPROG:
	case RPC_IBM3270PROG:
	case RPC_IBMRJEPROG:
	case RPC_SELNSVCPROG:
	case RPC_RDATABASEPROG:
	case RPC_REXECPROG:
	case RPC_ALICEPROG:
	case RPC_SCHEDPROG:
	case RPC_LOCKPROG:
	case RPC_NETLOCKPROG:
	case RPC_X25PROG:
	case RPC_STATMON1PROG:
	case RPC_STATMON2PROG:
	case RPC_SELNLIBPROG:
	case RPC_BOOTPARAMPROG:
	case RPC_MAZEPROG:
	case RPC_YPUPDATEPROG:
	case RPC_KEYSERVEPROG:
	case RPC_SECURECMDPROG:
	case RPC_NETFWDIPROG:
	case RPC_NETFWDTPROG:
	case RPC_SUNLINKMAP_PROG:
	case RPC_NETMONPROG:
	case RPC_DBASEPROG:
	case RPC_PWDAUTHPROG:
	case RPC_TFSPROG:
	case RPC_NSEPROG:
	case RPC_NSE_ACTIVATE_PROG:
	case RPC_PCNFSDPROG:
	case RPC_PYRAMIDLOCKINGPROG:
	case RPC_PYRAMIDSYS5:
	case RPC_CADDS_IMAGE:
	case RPC_ADT_RFLOCKPROG:
#endif /* notdef */
	default:
		pkt_counters[PKT_OTHERRPC].pc_interval++;
		pkt_counters[PKT_OTHERRPC].pc_total++;
		break;
	}
}

/*
 * rpc_replyfilter - count RPC reply packets.
 */
void
rpc_replyfilter(data, length, src, dst)
register u_long src, dst;
register u_int length;
register char *data;
{
	register struct rpc_msg *msg;

	msg = (struct rpc_msg *) data;

	pkt_counters[PKT_RPCAUTH].pc_interval++;
	pkt_counters[PKT_RPCAUTH].pc_total++;
}

/*
 * nfs_filter - filter NFS packets.
 */
void
nfs_filter(data, length, src, dst)
register u_long src, dst;
register u_int length;
register char *data;
{
	u_int proc;
	caddr_t args;
	SVCXPRT *xprt;
	struct rpc_msg msg;
	union nfs_rfsargs nfs_rfsargs;
	char cred_area[2*MAX_AUTH_BYTES];

	msg.rm_call.cb_cred.oa_base = cred_area;
	msg.rm_call.cb_verf.oa_base = &(cred_area[MAX_AUTH_BYTES]);

	/*
	 * Act as if we received this packet through RPC.
	 */
	if (!udprpc_recv(data, length, &msg, &xprt))
		return;

	/*
	 * Get the NFS procedure number.
	 */
	proc = msg.rm_call.cb_proc;

	if (proc >= RFS_NPROC)
		return;

	/*
	 * Now decode the arguments to the procedure from
	 * XDR format.
	 */
	args = (caddr_t) &nfs_rfsargs;
	(void) bzero(args, nfs_procs[proc].nfs_argsz);

	if (!SVC_GETARGS(xprt, nfs_procs[proc].nfs_xdrargs, args))
		return;

	/*
	 * Now count the packet in the appropriate file system's
	 * counters.
	 */
	switch (proc) {
	case RFS_NULL:
		break;
	case RFS_GETATTR:
		nfs_count(&nfs_rfsargs.fhandle, proc);
		break;
	case RFS_SETATTR:
		nfs_count(&nfs_rfsargs.nfssaargs.saa_fh, proc);
		break;
	case RFS_ROOT:
		break;
	case RFS_LOOKUP:
		nfs_count(&nfs_rfsargs.nfsdiropargs.da_fhandle, proc);
		break;
	case RFS_READLINK:
		nfs_count(&nfs_rfsargs.fhandle, proc);
		break;
	case RFS_READ:
		nfs_count(&nfs_rfsargs.nfsreadargs.ra_fhandle, proc);
		break;
	case RFS_WRITECACHE:
		break;
	case RFS_WRITE:
		nfs_count(&nfs_rfsargs.nfswriteargs.wa_fhandle, proc);
		break;
	case RFS_CREATE:
		nfs_count(&nfs_rfsargs.nfscreatargs.ca_da.da_fhandle, proc);
		break;
	case RFS_REMOVE:
		nfs_count(&nfs_rfsargs.nfsdiropargs.da_fhandle, proc);
		break;
	case RFS_RENAME:
		nfs_count(&nfs_rfsargs.nfsrnmargs.rna_from.da_fhandle, proc);
		break;
	case RFS_LINK:
		nfs_count(&nfs_rfsargs.nfslinkargs.la_from, proc);
		break;
	case RFS_SYMLINK:
		nfs_count(&nfs_rfsargs.nfsslargs.sla_from.da_fhandle, proc);
		break;
	case RFS_MKDIR:
		nfs_count(&nfs_rfsargs.nfscreatargs.ca_da.da_fhandle, proc);
		break;
	case RFS_RMDIR:
		nfs_count(&nfs_rfsargs.nfsdiropargs.da_fhandle, proc);
		break;
	case RFS_READDIR:
		nfs_count(&nfs_rfsargs.nfsrddirargs.rda_fh, proc);
		break;
	case RFS_STATFS:
		nfs_count(&nfs_rfsargs.fhandle, proc);
		break;
	}

	/*
	 * Decide whether it's a read or write process.
	 */
	switch (nfs_procs[proc].nfs_proctype) {
	case NFS_READ:
		pkt_counters[PKT_NFSREAD].pc_interval++;
		pkt_counters[PKT_NFSREAD].pc_total++;
		break;
	case NFS_WRITE:
		pkt_counters[PKT_NFSWRITE].pc_interval++;
		pkt_counters[PKT_NFSWRITE].pc_total++;
		break;
	}
}

/*
 * nfs_count - count an NFS reference to a specific file system.
 */
void
nfs_count(fh, proc)
register fhandle_t *fh;
int proc;
{
	long fsid;
	register int i, match1, match2;

	/*
	 * Run through the NFS counters looking for the matching
	 * file system.
	 */
	match1 = 0;

	for (i = 0; i < nnfscounters; i++) {
		if (learnfs)
			fsid = nfs_counters[i].nc_fsid;
		else
			fsid = (long) nfs_counters[i].nc_dev;

		/*
		 * Compare the device numbers.  Sun uses an
		 * fsid_t for the device number, which is an
		 * array of 2 longs.  The first long contains
		 * the device number.
		 */
		match1 = !bcmp((char *) &(fh->fh_fsid), (char *) &fsid,
			sizeof(long));

		/*
		 * Check server address.
		 */
		if (allflag && match1)
			match1 = (thisdst == nfs_counters[i].nc_ipaddr);

		if (match1) {
			nfs_counters[i].nc_proc[proc]++;
			nfs_counters[i].nc_interval++;
			nfs_counters[i].nc_total++;
			break;
		}
	}

	/*
	 * We don't know about this file system, but we can
	 * learn.
	 */
	if (!match1 && learnfs && (nnfscounters < MAXEXPORT)) {
		static char fsname[64], prefix[64];
		long fsid;

		i = nnfscounters++;

		bcopy((char *) &(fh->fh_fsid), (char *) &fsid, sizeof(long));

		nfs_counters[i].nc_fsid = fsid;
		nfs_counters[i].nc_proc[proc]++;
		nfs_counters[i].nc_interval++;
		nfs_counters[i].nc_total++;

		/*
		 * See if server uses opposite byte order.
		 */
		if ((fsid & 0xffff0000) && ((fsid & 0xffff) == 0))
			fsid = ntohl(fsid);

		/*
		 * Some hosts use 32-bit values.
		 */
		if (fsid & 0xffff0000) {
			nfs_counters[i].nc_dev = makedev((fsid >> 16) & 0xff,
							 minor(fsid));
		}
		else {
			nfs_counters[i].nc_dev = makedev(major(fsid),
							 minor(fsid));
		}

		*prefix = 0;

		if (allflag) {
			struct hostent *hp;

			nfs_counters[i].nc_ipaddr = thisdst;
			hp = gethostbyaddr(&thisdst, sizeof(thisdst), AF_INET);

			if (hp) {
				char *index();
				char *dotp;

				sprintf(prefix, "%s", hp->h_name);

				if ((dotp = index(prefix, '.')) != NULL)
					*dotp = 0;
			}
			else {
				sprintf(prefix, "%08x", thisdst);
			}
		}

		sprintf(fsname, "%.12s(%d,%d)", prefix,
			major(nfs_counters[i].nc_dev),
			minor(nfs_counters[i].nc_dev));

		nfs_counters[i].nc_name = savestr(fsname);
		sort_nfs_counters();
		label_screen();
	}

	if (filelist == NULL)
		return;

	/*
	 * Run through the file counters looking for the matching
	 * file.
	 */
	for (i = 0; i < nfilecounters; i++) {
		fsid = (long) fil_counters[i].fc_dev;

		/*
		 * Compare device numbers and file numbers.  Sun
		 * uses an fsid_t for the device, which is an
		 * array of two longs.  They use an fid for the
		 * inode.  The inode number is the first part
		 * of this.
		 */
		match1 = !bcmp((char *) &(fh->fh_fsid), (char *) &fsid,
			 sizeof(long));

		if (!match1)
			continue;

#ifdef sun
		/*
		 * NOTE: this is dependent on the contents of the fh_data
		 *       part of the file handle.  This is correct for
		 *       SunOS 4.1 on SPARCs.
		 */
		match2 = !bcmp((char *) &(fh->fh_data[2]),
			 (char *) &(fil_counters[i].fc_ino), sizeof(ino_t));
#endif /* sun */

#ifdef ultrix
		match2 = !bcmp((char *) fh->fh_fno,
			 (char *) &(fil_counters[i].fc_ino), sizeof(ino_t));
#endif /* ultrix */

		if (match2) {
			fil_counters[i].fc_proc[proc]++;
			fil_counters[i].fc_interval++;
			fil_counters[i].fc_total++;
			break;
		}
	}
}
