#ifndef lint
static char *RCSid = "$Header: /tmp_mnt/net/sparky.a/davy/system/nfswatch/RCS/nfswatch.c,v 1.2 90/08/17 15:47:29 davy Exp $";
#endif

/*
 * nfswatch - NFS server packet monitoring program.
 *
 * David A. Curry
 * SRI International
 * 333 Ravenswood Avenue
 * Menlo Park, CA 94025
 * davy@itstd.sri.com
 *
 * $Log:	nfswatch.c,v $
 * Revision 1.2  90/08/17  15:47:29  davy
 * NFSWATCH Version 2.0.
 * 
 * Revision 1.1  88/11/29  11:20:40  davy
 * NFSWATCH Release 1.0
 * 
 */
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <net/if.h>
#include <signal.h>
#include <errno.h>
#include <stdio.h>

#ifdef sun
#include <net/nit_if.h>
#include <net/nit_buf.h>

#define USE_NIT
#endif /* sun */

#ifdef ultrix
#include <net/pfilt.h>

#define USE_PFILT
#endif /* ultrix */

#include "nfswatch.h"

char		*pname;				/* program name		*/

FILE		*logfp;				/* log file pointer	*/

Counter		pkt_total = 0;			/* total packets seen	*/
Counter		pkt_drops = 0;			/* total packets dropped*/
Counter		int_pkt_total = 0;		/* packets this interval*/
Counter		int_pkt_drops = 0;		/* dropped this interval*/
Counter		dst_pkt_total = 0;		/* total pkts to host	*/
Counter		int_dst_pkt_total = 0;		/* pkts to host this int*/

int		if_fd = -1;			/* nit device file desc	*/
int		srcflag = 0;			/* "-src" specified	*/
int		dstflag = 0;			/* "-dst" specified	*/
int		allflag = 0;			/* "-all" specified	*/
int		logging = 0;			/* 1 when logging on	*/
int		learnfs = 0;			/* learn other servers	*/
int		do_update = 0;			/* time to update screen*/
int		showwhich = 0;			/* show filesys or files*/
int		cycletime = CYCLETIME;		/* update cycle time	*/
int		truncation = 200;		/* pkt trunc len - magic*/
int		sortbyusage = 0;		/* sort by usage counts	*/
int		nnfscounters = 0;		/* # of NFS counters	*/
int		nfilecounters = 0;		/* # of file counters	*/
int		screen_inited = 0;		/* 1 when in curses	*/

struct timeval	starttime;			/* time we started	*/

u_long		thisdst = 0;			/* cached IP dst of pkt	*/
u_long		srcaddrs[MAXHOSTADDR];		/* src host net addrs	*/
u_long		dstaddrs[MAXHOSTADDR];		/* dst host net addrs	*/

char		myhost[MAXHOSTNAMELEN];		/* local host name	*/
char		srchost[MAXHOSTNAMELEN];	/* source host name	*/
char		dsthost[MAXHOSTNAMELEN];	/* destination host name*/

char		*prompt = PROMPT;		/* prompt string	*/
char		*filelist = NULL;		/* list of files	*/
char		*logfile = LOGFILE;		/* log file name	*/
char		*snapshotfile = SNAPSHOTFILE;	/* snapshot file name	*/

NFSCounter	nfs_counters[MAXEXPORT];	/* NFS request counters	*/
FileCounter	fil_counters[MAXEXPORT];	/* file request counters*/
PacketCounter	pkt_counters[PKT_NCOUNTERS];	/* packet counters	*/

main(argc, argv)
int argc;
char **argv;
{
	struct sigvec sv;
	char *device = NULL;
	extern void finish(), nfswatch();

	pname = *argv;

	/*
	 * Get our host name.  The default destination
	 * host is the one we're running on.
	 */
	if (gethostname(myhost, sizeof(myhost)) < 0) {
		error("gethostname");
		finish(-1);
	}

	(void) strcpy(dsthost, myhost);

	/*
	 * Process arguments.
	 */
	while (--argc) {
		if (**++argv != '-')
			usage();

		/*
		 * Set destination host.
		 */
		if (!strcmp(*argv, "-dst")) {
			if (--argc <= 0)
				usage();

			(void) strcpy(dsthost, *++argv);
			dstflag++;
			continue;
		}

		/*
		 * Set source host.
		 */
		if (!strcmp(*argv, "-src")) {
			if (--argc <= 0)
				usage();

			(void) strcpy(srchost, *++argv);
			srcflag++;
			continue;
		}

		/*
		 * Device to use.
		 */
		if (!strcmp(*argv, "-dev")) {
			if (--argc <= 0)
				usage();

			device = *++argv;
			continue;
		}

		/*
		 * Log file name.
		 */
		if (!strcmp(*argv, "-lf")) {
			if (--argc <= 0)
				usage();

			logfile = *++argv;
			continue;
		}

		/*
		 * Snapshot file name.
		 */
		if (!strcmp(*argv, "-sf")) {
			if (--argc <= 0)
				usage();

			snapshotfile = *++argv;
			continue;
		}

		/*
		 * List of files.
		 */
		if (!strcmp(*argv, "-f")) {
			if (--argc <= 0)
				usage();

			if (showwhich == 0)
				showwhich = SHOWINDVFILES;

			filelist = *++argv;
			continue;
		}

		/*
		 * Change cycle time.
		 */
		if (!strcmp(*argv, "-t")) {
			if (--argc <= 0)
				usage();

			cycletime = atoi(*++argv);
			continue;
		}

		/*
		 * Show file systems.
		 */
		if (!strcmp(*argv, "-fs")) {
			showwhich = SHOWFILESYSTEM;
			continue;
		}

		/*
		 * Show individual files.
		 */
		if (!strcmp(*argv, "-if")) {
			showwhich = SHOWINDVFILES;
			continue;
		}

		/*
		 * Turn on logging.
		 */
		if (!strcmp(*argv, "-l")) {
			logging++;
			continue;
		}

		/*
		 * Watch all traffic.
		 */
		if (!strcmp(*argv, "-all")) {
			allflag++;
			continue;
		}

		/*
		 * Sort file systems by usage, not name.
		 */
		if (!strcmp(*argv, "-usage")) {
			sortbyusage++;
			continue;
		}

		usage();
	}

	/*
	 * Check what we're showing.
	 */
	switch (showwhich) {
	case 0:			/* default */
		showwhich = SHOWFILESYSTEM;
		break;
	case SHOWINDVFILES:
		if (filelist == NULL) {
			(void) fprintf(stderr, "%s: must specify file list with -fi.\n", pname);
			finish(-1);
		}

		break;
	}

	/*
	 * Trap signals so we can clean up.
	 */
	sv.sv_handler = finish;
	sv.sv_mask = sv.sv_flags = 0;

	(void) sigvec(SIGINT, &sv, (struct sigvec *) 0);
	(void) sigvec(SIGQUIT, &sv, (struct sigvec *) 0);

#ifdef USE_NIT
	/*
	 * Set up the network interface tap right away,
	 * since we probably need super-user permission.
	 */
	setup_nit_dev(device);
#endif /* USE_NIT */

#ifdef USE_PFILT
	/*
	 * Set up the packet filter interface now,
	 * although we don't need super-user permission.
	 */
	setup_pfilt_dev(device);
#endif /* USE_PFILT */

	/*
	 * Now lose super-user permission, since we
	 * don't need it for anything else.
	 */
	(void) setreuid(getuid(), getuid());

	/*
	 * Look up the network addresses of the source and
	 * destination hosts.
	 */
	get_net_addrs();

	/*
	 * Tell the user what's going on.
	 */
	(void) printf("NFSWATCH Version 2.0; August, 1990\n");
	(void) printf("Watch packets from %s to %s;\n",
		(srcflag ? srchost : "all hosts"), dsthost);
	(void) printf("log to \"%s\" (logging %s);\n", logfile,
		(logging ? "on" : "off"));
	(void) printf("snapshots to \"%s\";\n", snapshotfile);
	(void) printf("cycle time %d seconds...", cycletime);
	(void) fflush(stdout);

	/*
	 * Set up a pseudo RPC server.
	 */
	setup_rpcxdr();

	/*
	 * Set up the screen.
	 */
	setup_screen();

	/*
	 * Set up the packet counters.  This must be done after
	 * setup_screen because they use the LINES variable.
	 */
	setup_pkt_counters();
	setup_nfs_counters();

	if (filelist)
		setup_fil_counters();

	/*
	 * Now label the screen.
	 */
	label_screen();

	/*
	 * Open log file if logging is on.
	 */
	if (logging) {
		if ((logfp = fopen(logfile, "a")) == NULL) {
			error(logfile);
			finish(-1);
		}

		(void) fprintf(logfp, "#\n# startlog\n#\n");
		(void) fprintf(logfp, "# NFSwatch log file\n");
		(void) fprintf(logfp, "#    Packets from: %s\n",
			(srcflag ? srchost : "all hosts"));
		(void) fprintf(logfp, "#    Packets to:   %s\n#\n",
			dsthost);
	}

	/*
	 * Go watch packets.  Never returns.
	 */
	nfswatch();
}

/*
 * nfswatch - main packet reading loop.
 */
void
nfswatch()
{
	int cc;
	char *buf;
	char *malloc();
	fd_set readfds;
	struct sigvec sv;
	struct timeval tv;
	extern void wakeup();
	struct itimerval itv;
	register char *bp, *cp, *bufstop;
#ifdef USE_NIT
	struct nit_bufhdr *hdrp;
	struct nit_ifdrops *ndp;
#endif /* USE_NIT */

#ifdef USE_PFILT
	struct enstamp stamp;
	int datalen;
#endif /* USE_PFILT */

	/*
	 * Allocate a buffer so it's properly aligned for
	 * casting to structure types.
	 */
	if ((buf = malloc(NIT_CHUNKSIZE)) == NULL) {
		(void) fprintf(stderr, "%s: out of memory.\n", pname);
		finish(-1);
	}

	/*
	 * Set up the alarm handler.
	 */
	sv.sv_handler = wakeup;
	sv.sv_mask = sv.sv_flags = 0;

	(void) sigvec(SIGALRM, &sv, (struct sigvec *) 0);

	/*
	 * Set up the alarm clock.
	 */
	(void) bzero((char *) &itv, sizeof(struct itimerval));

	itv.it_interval.tv_sec = cycletime;
	itv.it_interval.tv_usec = 0;

	itv.it_value = itv.it_interval;

	(void) setitimer(ITIMER_REAL, &itv, (struct itimerval *) 0);

	/*
	 * Set the start time.
	 */
	(void) gettimeofday(&starttime, (struct timezone *) 0);

#ifdef USE_NIT
	/*
	 * Flush the read queue of any packets that accumulated
	 * during setup time.
	 */
	flush_nit();

	/*
	 * Now read packets from the nit device.
	 */
	while ((cc = read(if_fd, buf, NIT_CHUNKSIZE)) >= 0) {
		bufstop = buf + cc;
		bp = buf;

		/*
		 * Loop through the chunk, extracting packets.
		 */
		while (bp < bufstop) {
			cp = bp;

			/*
			 * Get the nit header.
			 */
			hdrp = (struct nit_bufhdr *) cp;
			cp += sizeof(struct nit_bufhdr);

			/*
			 * Get the number of dropped packets.
			 */
			ndp = (struct nit_ifdrops *) cp;
			cp += sizeof(struct nit_ifdrops);

			int_pkt_drops += ndp->nh_drops - pkt_drops;
			pkt_drops = ndp->nh_drops;

			/*
			 * Filter the packet.
			 */
			pkt_filter(cp, hdrp->nhb_msglen);

			/*
			 * Skip over this packet.
			 */
			bp += hdrp->nhb_totlen;
		}
#endif /* USE_NIT */

#ifdef USE_PFILT
	/*
	 * Flush the read queue of any packets that accumulated
	 * during setup time.
	 */
	flush_pfilt();

	/*
	 * Now read packets from the packet filter device.
	 */
	for (;;) {
		if ((cc = read(if_fd, buf, NIT_CHUNKSIZE)) < 0) {
			lseek(if_fd, 0L, 0);

			/*
			* Might have read MAXINT bytes.  Try again.
			*/
			if ((cc = read(if_fd, buf, NIT_CHUNKSIZE)) < 0) {
				error("pfilt read");
				finish(-1);
			}
		}
		
		bp = buf;

		/*
		 * Loop through buffer, extracting packets.
		 */
		while (cc > 0) {
			/*
			 * Avoid alignment issues.
			 */
			(void) bcopy(bp, &stamp, sizeof(stamp));

			/*
			 * Treat entire buffer as garbage.
			 */
			if (stamp.ens_stamplen != sizeof(stamp))
				break;

			/*
			 * Get the number of dropped packets.
			 */
			int_pkt_drops += stamp.ens_dropped;
			pkt_drops += stamp.ens_dropped;

			/*
			 * Filter the packet.
			 */
			datalen = stamp.ens_count;

			if (datalen > truncation)
				datalen = truncation;

			pkt_filter(&(bp[sizeof(stamp)]), datalen);

			/*
			 * Skip over this packet.
			 */
			if (cc == (datalen + sizeof(stamp)))
				break;

			datalen = ENALIGN(datalen);
			datalen += sizeof(stamp);
			cc -= datalen;
			bp += datalen;
		}
#endif /* USE_PFILT */

		/*
		 * Do a screen update if necessary.
		 * Clear the interval counters and
		 * reset the timer.
		 */
		if (do_update) {
			/*
			 * Re-sort and re-do the labels.
			 */
			if (sortbyusage) {
				sort_nfs_counters();
				label_screen();
			}

			update_screen();

			if (logging)
				update_logfile();

			clear_vars();

			do_update = 0;
		}

		/*
		 * Set up to see if there's input.
		 */
		tv.tv_sec = 0;
		tv.tv_usec = 0;
		FD_ZERO(&readfds);
		FD_SET(0, &readfds);

		/*
		 * See if the user typed a command,
		 * and process it if he did.
		 */
		cc = select(NFDBITS, &readfds, (fd_set *) 0, (fd_set *) 0, &tv);

		if ((cc > 0) && FD_ISSET(0, &readfds))
			command();
	}

#ifdef USE_NIT
	/*
	 * Should never get here.
	 */
	error("nit read");
	finish(-1);
#endif /* USE_NIT */
}
