#include <stdio.h>
#include <netdb.h>
#include <errno.h>
#include <fstab.h>
#include <signal.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <xdr.h>
#include <rpc.h>
#include <nfs.h>

#include <htable.h>
#include <malloc.h>

extern int errno;

/* extern */ char **argvec;

typedef struct {
	  int i1;
	  int i2;
	  int i3;
	  int i4; } FOURINTS;

dev_t devlist[MAX_EXPORTS];
dev_t *devptrs[MAX_EXPORTS];
int ndevs;
char pmap_service[] = "sunrpc";
char packet[10240];
int packetlen;
char reply[10240];
int sock;
RPC_MSG req;
struct sockaddr_in from;
RPC_MSG rep;
struct auth_unix au;
int shoulddie;
int lastkillsig;
int nobody = -2; /* I consider it broken, but people seem to want it */

int rpc_errno;

#if defined(SYSLOG_SLEEPS)
syslog_(level,fmt,a1,a2,a3,a4,a5,a6,a7,a8,a9,a10)
int level;
char *fmt;
int a1;
int a2;
int a3;
int a4;
int a5;
int a6;
int a7;
int a8;
int a9;
int a10;
{
 syslog(level,fmt,a1,a2,a3,a4,a5,a6,a7,a8,a9,a10);
 sleep(1);
}
#else
#define syslog_ syslog
#endif

bool_t xdr_4ints(x,fi)
XDR *x;
FOURINTS *fi;
{
 return( xdr_int(x,&fi->i1) &&
	 xdr_int(x,&fi->i2) &&
	 xdr_int(x,&fi->i3) &&
	 xdr_int(x,&fi->i4) );
}

char *rpc_errorstr()
{
 static char *rpc_errs[] = { "Error 0",
			     "RPC version mismatch",
			     "Bad credentials",
			     "Rejected credentials",
			     "Bad verifier",
			     "Rejected verifier",
			     "Insufficient privilege",
			     "Unsupported program version",
			     "Program unavailable",
			     "Unsupported procedure number",
			     "Bad argument format",
			     "Bad reply format",
			     "Timed out" };
 static int rpc_nerr = sizeof(rpc_errs) / sizeof(rpc_errs[0]);
 static char unknbuf[64];

 if ((rpc_errno < 0) || (rpc_errno >= rpc_nerr))
  { sprintf(unknbuf,"Unknown RPC error %d",rpc_errno);
    return(unknbuf);
  }
 else
  { return(rpc_errs[rpc_errno]);
  }
}

bool_t rpc_call(s,port,addr,prog,vers,proc,argx,arga,retx,reta)
int s;
int port;
struct in_addr addr;
int prog;
int vers;
int proc;
bool_t (*argx)();
char *arga;
bool_t (*retx)();
char *reta;
{
 XDR x;
 int rv;
 struct sockaddr_in sin;
 int tries;
 fd_set fds;
 struct timeval delay;

 req.xid = get_rpcxid();
 req.type = CALL;
 req.u.call.rpcvers = 2;
 req.u.call.prog = prog;
 req.u.call.vers = vers;
 req.u.call.proc = proc;
 req.u.call.cred.type = AUTH_NULL;
 req.u.call.cred.len = 0;
 req.u.call.verf.type = AUTH_NULL;
 req.u.call.verf.len = 0;
 xdrmem_create(&x,packet,sizeof(packet),XDR_ENCODE);
 if ( !xdr_rpc_msg(&x,&req) ||
      !(*argx)(&x,arga) )
  { xdr_destroy(&x);
    return(FALSE);
  }
 packetlen = xdr_getpos(&x);
 xdr_destroy(&x);
 sin.sin_family = AF_INET;
 sin.sin_addr = addr;
 sin.sin_port = ntohs(port);
 for (tries=0;tries<5;tries++)
  { rv = sendto(s,packet,packetlen,0,(char *)&sin,sizeof(sin));
    if (rv < 0)
     { syslog_(LOG_ERR,"can't send rpc request to %s/%d",inet_ntoa(addr),port);
       return(FALSE);
     }
    FD_ZERO(&fds);
    FD_SET(s,&fds);
    delay.tv_sec = 0;
    delay.tv_usec = 500000;
    rv = select(FD_SETSIZE,&fds,(fd_set *)0,(fd_set *)0,&delay);
    if (rv == 0)
     { continue;
     }
    if (! FD_ISSET(s,&fds))
     { continue;
     }
    rv = recvfrom(s,reply,sizeof(reply),0,(char *)0,(int *)0);
    xdrmem_create(&x,reply,rv,XDR_DECODE);
    if (! xdr_rpc_msg(&x,&rep))
     {
badreply:
       syslog_(LOG_WARNING,"bad rpc reply from %s/%d",inet_ntoa(addr),port);
       xdr_destroy(&x);
       continue;
     }
    if (rep.xid != req.xid)
     { syslog_(LOG_INFO,"xid mismatch from %s/%d",inet_ntoa(addr),port);
       xdr_destroy(&x);
       continue;
     }
    if (rep.type != REPLY)
     { syslog_(LOG_WARNING,"non-reply rpc reply from %s/%d",inet_ntoa(addr),port);
       xdr_destroy(&x);
       continue;
     }
    switch (rep.u.reply.type)
     { case MSG_DENIED:
	  switch (rep.u.reply.u.denied.type)
	   { case RPC_MISMATCH:
		rpc_errno = RPCERR_RPCMISMATCH;
		goto lose;
		break;
	     case AUTH_ERROR:
		switch (rep.u.reply.u.denied.u.error)
		 { case AUTH_BADCRED:
		      rpc_errno = RPCERR_BADCRED;
		      goto lose;
		      break;
		   case AUTH_REJECTEDCRED:
		      rpc_errno = RPCERR_REJECTEDCRED;
		      goto lose;
		      break;
		   case AUTH_BADVERF:
		      rpc_errno = RPCERR_BADVERF;
		      goto lose;
		      break;
		   case AUTH_REJECTEDVERF:
		      rpc_errno = RPCERR_REJECTEDVERF;
		      goto lose;
		      break;
		   case AUTH_TOOWEAK:
		      rpc_errno = RPCERR_TOOWEAK;
		      goto lose;
		      break;
		   default:
		      goto badreply;
		      break;
		 }
		break;
	     default:
		goto badreply;
		break;
	   }
	  break;
       case MSG_ACCEPTED:
	  switch (rep.u.reply.u.accepted.type)
	   { case SUCCESS:
		if (! (*retx)(&x,reta))
		 { rpc_errno = RPCERR_GARBAGEREPLY;
		   goto lose;
		 }
		else
		 { goto win;
		 }
		break;
	     case PROG_MISMATCH:
		rpc_errno = RPCERR_PROGMISMATCH;
		goto lose;
		break;
	     case PROG_UNAVAIL:
		rpc_errno = RPCERR_PROGUNAVAIL;
		goto lose;
		break;
	     case PROC_UNAVAIL:
		rpc_errno = RPCERR_PROCUNAVAIL;
		goto lose;
		break;
	     case GARBAGE_ARGS:
		rpc_errno = RPCERR_GARBAGEARGS;
		goto lose;
		break;
	     default:
		goto badreply;
		break;
	   }
	  break;
       default:
	  goto badreply;
	  break;
     }
  }
 rpc_errno = RPCERR_TIMEDOUT;
 return(FALSE);
lose:
 xdr_destroy(&x);
 return(FALSE);
win:
 xdr_destroy(&x);
 rpc_errno = RPCERR_NONE;
 return(TRUE);
}

int cmpdev(d1,d2)
char *d1;
char *d2;
{
 if ((int)*(dev_t *)d1 > (int)*(dev_t *)d2)
  { return(1);
  }
 else if ((int)*(dev_t *)d1 < (int)*(dev_t *)d2)
  { return(-1);
  }
 else
  { return(0);
  }
}

readfstab()
{
 struct fstab *fs;
 struct stat stb;
 int i;

 ndevs = 0;
 setfsent();
 while (fs=getfsent())
  { if (stat(fs->fs_spec,&stb) < 0)
     { syslog(LOG_WARNING,"ignoring %s, cannot stat (%m)",fs->fs_spec);
       continue;
     }
    if (ndevs >= MAX_EXPORTS)
     { syslog(LOG_WARNING,"ignoring %s, too many devices already",fs->fs_spec);
       continue;
     }
    devlist[ndevs++] = stb.st_rdev;
  }
 endfsent();
 if (ndevs == 0)
  { syslog(LOG_ERR,"can't find any devices!");
    exit(1);
  }
 for (i=0;i<ndevs;i++)
  { devptrs[i] = &devlist[i];
  }
 heapsort((char **)devptrs,ndevs,cmpdev);
 for (i=0;i<ndevs;i++)
  { syslog_(LOG_DEBUG,"fsid %d -> dev %x",i,(int)*devptrs[i]);
  }
}

dev_t fsno_to_dev(fsno)
int fsno;
{
 return(*devptrs[fsno]);
}

int dev_to_fsno(dev)
dev_t dev;
{
 return(binsearch((char **)devptrs,ndevs,cmpdev,(char *)&dev));
}

killsig(sig)
int sig;
{
 lastkillsig = sig;
 shoulddie ++;
}

main(ac,av)
int ac;
char **av;
{
 struct sockaddr_in sin;
 int sinlen;
 struct servent *sp;
 int i;
 FOURINTS fi;
 char *getenv();
 int errs;
 int rest;
 int skip;
 int argno;
 int debugmode;

 argvec = av;
 debugmode = 0;
 errs = 0;
 skip = 0;
 argno = 0;
 for (ac--,av++;ac;ac--,av++)
  { if (skip > 0)
     { skip --;
       continue;
     }
    if (**av == '-')
     { rest = 0;
       for (++*av;**av&&!rest;++*av)
	{ switch (**av)
	   { default:
		fprintf(stderr,"%s: bad flag -%c\n",argvec[0],**av);
		errs ++;
		break;
	     case 'd':
		debugmode = 1;
		break;
	     case 'r':
		rest = 1;
		if (!*++*av)
		 { nobody = 0;
		 }
		else
		 { nobody = atoi(*av);
		 }
		break;
	   }
	}
     }
    else
     { switch (argno++)
	{ default:
	     fprintf(stderr,"%s: extra argument `%s'\n",argvec[0],*av);
	     errs ++;
	     break;
	}
     }
  }
 if (errs)
  { exit(1);
  }
 openlog("nfsd",LOG_PID|LOG_CONS,LOG_DAEMON);
 if ((getenv("NFSD_DEBUG") == 0) && !debugmode)
  { setlogmask(LOG_UPTO(LOG_INFO));
  }
 i = fork();
 if (i < 0)
  { perror("fork");
  }
 if (i)
  { exit(0);
  }
 i = open("/dev/tty",O_RDWR,0);
 if (i >= 0)
  { ioctl(i,TIOCNOTTY,0);
    close(i);
  }
 readfstab();
 sock = socket(AF_INET,SOCK_DGRAM,0);
 if (sock < 0)
  { syslog_(LOG_ERR,"can't create socket: %m");
    exit(1);
  }
 sp = getservbyname(pmap_service,"udp");
 if (sp == 0)
  { syslog_(LOG_ERR,"%s/udp service unknown",pmap_service);
    exit(1);
  }
 sin.sin_family = AF_INET;
 sin.sin_addr.s_addr = INADDR_ANY;
 sin.sin_port = htons(NFS_RPC_PORT); /* bug in protocol: should be 0 */
 if (bind(sock,(char *)&sin,sizeof(sin)) < 0)
  { syslog_(LOG_ERR,"can't bind socket: %m");
    exit(1);
  }
 sinlen = sizeof(sin);
 if (getsockname(sock,(char *)&sin,&sinlen) < 0)
  { syslog_(LOG_ERR,"can't get socket address: %m");
    exit(1);
  }
 fi.i1 = NFS_PROGRAM;
 fi.i2 = NFS_VERSION;
 fi.i3 = 0;
 fi.i4 = 0;
 if (! rpc_call( sock,
		 111, sin.sin_addr,
		 100000, 2, 2,
		 xdr_4ints, (char *)&fi,
		 xdr_bool,(bool_t *)&i ) )
  { syslog_(LOG_ERR,"can't delete mapping: %s",rpc_errorstr());
    exit(1);
  }
 fi.i1 = NFS_PROGRAM;
 fi.i2 = NFS_VERSION;
 fi.i3 = IPPROTO_UDP;
 fi.i4 = ntohs(sin.sin_port);
 if (! rpc_call( sock,
		 111, sin.sin_addr,
		 100000, 2, 1,
		 xdr_4ints, (char *)&fi,
		 xdr_bool,(bool_t *)&i ) )
  { syslog_(LOG_ERR,"can't install mapping: %s",rpc_errorstr());
    exit(1);
  }
 if (! i)
  { syslog_(LOG_ERR,"portmapper refused to install mapping");
    exit(1);
  }
 shoulddie = 0;
 signal(SIGHUP,killsig);
 signal(SIGINT,killsig);
 signal(SIGTERM,killsig);
 signal(SIGTSTP,killsig);
 signal(SIGTTIN,killsig);
 signal(SIGTTOU,killsig);
 signal(SIGXCPU,killsig);
 signal(SIGXFSZ,killsig);
  { long int now;
    time(&now);
    syslog(LOG_INFO,"startup %.24s",ctime(&now));
  }
 umask(0);
 i = 0;
 while (1)
  { if (shoulddie)
     { goto die;
     }
    sinlen = sizeof(sin);
    packetlen = recvfrom(sock,packet,sizeof(packet),0,(char *)&sin,&sinlen);
    if (packetlen < 0)
     { if (errno == EINTR)
	{ continue;
	}
       syslog_(LOG_ERR,"error in recvfrom(): %m");
badpacket:
       i ++;
       if (i > 10)
	{ goto die;
	}
       sleep(2);
       continue;
     }
    if (packetlen == 0)
     { syslog_(LOG_WARNING,"zero-size input packet");
       goto badpacket;
     }
    if (sinlen != sizeof(sin))
     { syslog_(LOG_WARNING,"bad from-address size (%d bytes)",sinlen);
       goto badpacket;
     }
    i -= 4;
    if (i < 0)
     { i = 0;
     }
    from = sin;
    inputpacket(packet,packetlen);
  }
die:
 syslog_(LOG_INFO,"dying: received signal %d",lastkillsig);
 sinlen = sizeof(sin);
 if (getsockname(sock,(char *)&sin,&sinlen) < 0)
  { syslog_(LOG_ERR,"can't get socket address: %m");
    exit(1);
  }
 fi.i1 = NFS_PROGRAM;
 fi.i2 = NFS_VERSION;
 fi.i3 = 0;
 fi.i4 = 0;
 if (! rpc_call( sock,
		 111, sin.sin_addr,
		 100000, 2, 2,
		 xdr_4ints, (char *)&fi,
		 xdr_bool,(bool_t *)&i ) )
  { syslog_(LOG_ERR,"can't delete mapping: %s",rpc_errorstr());
    exit(1);
  }
 if (! i)
  { syslog_(LOG_ERR,"portmapper refused to delete mapping");
    exit(1);
  }
 exit(0);
}

inputpacket(pkt,len)
char *pkt;
int len;
{
 XDR x;

 xdrmem_create(&x,pkt,len,XDR_DECODE);
 if (! xdr_rpc_msg(&x,&req))
  { syslog_(LOG_WARNING,"XDR failed on input packet from %s/%d",
			inet_ntoa(from.sin_addr),ntohs(from.sin_port));
    goto out;
  }
 syslog_(LOG_DEBUG,"input packet from %s/%d",
			inet_ntoa(from.sin_addr),ntohs(from.sin_port));
 rep.xid = req.xid;
 rep.type = REPLY;
 if (req.type != CALL)
  { syslog_(LOG_INFO,"non-call packet from %s/%d",
			inet_ntoa(from.sin_addr),ntohs(from.sin_port));
    goto out;
  }
 if (req.u.call.rpcvers != 2)
  { rpc_mismatch();
    goto out;
  }
 if (req.u.call.prog != NFS_PROGRAM)
  { syslog_(LOG_WARNING,"non-nfsd packet (program %d) from %s/%d",
	req.u.call.prog,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
    goto out;
  }
 if (req.u.call.vers != NFS_VERSION)
  { vers_mismatch();
    goto out;
  }
 switch (req.u.call.proc)
  { case 0:
       msg_null(&x);
       break;
    case 1:
       msg_getattr(&x);
       break;
    case 2:
       msg_setattr(&x);
       break;
    case 3:
       badproc(); /* old _root procedure */
       break;
    case 4:
       msg_lookup(&x);
       break;
    case 5:
       msg_readlink(&x);
       break;
    case 6:
       msg_read(&x);
       break;
    case 7:
       badproc(); /* old _writecache procedure */
       break;
    case 8:
       msg_write(&x);
       break;
    case 9:
       msg_create(&x);
       break;
    case 10:
       msg_remove(&x);
       break;
    case 11:
       msg_rename(&x);
       break;
    case 12:
       msg_link(&x);
       break;
    case 13:
       msg_symlink(&x);
       break;
    case 14:
       msg_mkdir(&x);
       break;
    case 15:
       msg_rmdir(&x);
       break;
    case 16:
       msg_readdir(&x);
       break;
    case 17:
       msg_statfs(&x);
       break;
    default:
       badproc();
       break;
  }
out:
 xdr_destroy(&x);
 return;
}

sendreply(xextra,extraarg)
bool_t (*xextra)();
char *extraarg;
{
 XDR x;
 int len;
 int sent;

 xdrmem_create(&x,reply,sizeof(reply),XDR_ENCODE);
 if (xdr_rpc_msg(&x,&rep) && (*xextra)(&x,extraarg))
  { len = xdr_getpos(&x);
    sent = sendto(sock,reply,len,0,&from,sizeof(struct sockaddr_in));
    if (sent < 0)
     { syslog_(LOG_ERR,"can't return message (length %d) to %s/%d: %m",
			len,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
     }
    else if (sent != len)
     { syslog_(LOG_WARNING,"sendto sent %d not %d",sent,len);
     }
  }
 else
  { syslog_(LOG_ERR,"XDR can't encode return message");
  }
 xdr_destroy(&x);
}

rpc_mismatch()
{
 rep.u.reply.type = MSG_DENIED;
 rep.u.reply.u.denied.type = RPC_MISMATCH;
 rep.u.reply.u.denied.u.mismatch.low = 2;
 rep.u.reply.u.denied.u.mismatch.high = 2;
 syslog_(LOG_WARNING,"bad RPC version (%d) from %s/%d",
	req.u.call.rpcvers,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
 sendreply(xdr_void,(char *)0);
}

vers_mismatch()
{
 rep.u.reply.type = MSG_ACCEPTED;
 rep.u.reply.u.accepted.verf.type = AUTH_NULL;
 rep.u.reply.u.accepted.verf.len = 0;
 rep.u.reply.u.accepted.type = PROG_MISMATCH;
 rep.u.reply.u.accepted.u.mismatch.low = MOUNT_VERSION;
 rep.u.reply.u.accepted.u.mismatch.high = MOUNT_VERSION;
 syslog_(LOG_WARNING,"bad nfs version (%d) from %s/%d",
	req.u.call.vers,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
 sendreply(xdr_void,(char *)0);
}

badproc()
{
 rep.u.reply.type = MSG_ACCEPTED;
 rep.u.reply.u.accepted.verf.type = AUTH_NULL;
 rep.u.reply.u.accepted.verf.len = 0;
 rep.u.reply.u.accepted.type = PROC_UNAVAIL;
 syslog_(LOG_WARNING,"bad procedure number (%d) from %s/%d",
	req.u.call.proc,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
 sendreply(xdr_void,(char *)0);
}

garbage_args()
{
 rep.u.reply.type = MSG_ACCEPTED;
 rep.u.reply.u.accepted.verf.type = AUTH_NULL;
 rep.u.reply.u.accepted.verf.len = 0;
 rep.u.reply.u.accepted.type = GARBAGE_ARGS;
 syslog_(LOG_WARNING,"can't decode args (procedure %d) from %s/%d",
	req.u.call.proc,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
 sendreply(xdr_void,(char *)0);
}

tooweak()
{
 rep.u.reply.type = MSG_DENIED;
 rep.u.reply.u.denied.type = AUTH_ERROR;
 rep.u.reply.u.denied.u.error = AUTH_TOOWEAK;
 syslog_(LOG_DEBUG,"authorization too weak (procedure %d) from %s/%d",
	req.u.call.proc,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
 sendreply(xdr_void,(char *)0);
}

success(xextra,extraarg)
bool_t (*xextra)();
char *extraarg;
{
 rep.u.reply.type = MSG_ACCEPTED;
 rep.u.reply.u.accepted.verf.type = AUTH_NULL;
 rep.u.reply.u.accepted.verf.len = 0;
 rep.u.reply.u.accepted.type = SUCCESS;
 sendreply(xextra,extraarg);
}

int authunix()
{
 XDR x;

 if (req.u.call.cred.type != AUTH_UNIX)
  { tooweak();
    return(0);
  }
 xdrmem_create(&x,req.u.call.cred.body,req.u.call.cred.len,XDR_DECODE);
 if (! xdr_auth_unix(&x,&au))
  { xdr_destroy(&x);
    tooweak();
    return(0);
  }
 /* I consider this a misfeature.  However, people seem to want it. */
 if (au.uid == 0)
  { au.uid = nobody;
  }
 return(1);
}

stb_to_fh(stb,fh)
struct stat *stb;
FHANDLE *fh;
{
 bzero(fh->bytes,FHSIZE);
 fh->s.dev = stb->st_dev;
 fh->s.ino = stb->st_ino;
 fh->s.usecount = stb->st_usecount;
}

stb_to_fattr(stb,fa)
struct stat *stb;
FATTR *fa;
{
 fa->type = maptype(stb->st_mode);
 fa->mode = stb->st_mode;
 fa->nlink = stb->st_nlink;
 fa->uid = stb->st_uid;
 fa->gid = stb->st_gid;
 fa->size = stb->st_size;
 fa->blocksize = DEV_BSIZE;
 fa->rdev = stb->st_rdev;
 fa->blocks = stb->st_blocks;
 fa->fsid = dev_to_fsno(stb->st_dev);
 fa->fileid = stb->st_ino +
	      (fa->fsid << FSID_SHIFT);
 fa->atime.tv_sec = stb->st_atime;
 fa->atime.tv_usec = 0;
 fa->mtime.tv_sec = stb->st_mtime;
 fa->mtime.tv_usec = 0;
 fa->ctime.tv_sec = stb->st_ctime;
 fa->ctime.tv_usec = 0;
}

msg_null(x)
XDR *x;
{
 syslog_(LOG_DEBUG,"null()");
 success(xdr_void,(char *)0);
}

msg_getattr(x)
XDR *x;
{
 FHANDLE fh;
 ATTRSTAT rv;

 if (! xdr_fhandle(x,&fh))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"getattr(%x/%d/%d)",
			(int)fh.s.dev,(int)fh.s.ino,(int)fh.s.usecount);
 nfs_getattr(&au,&fh,&rv);
 success(xdr_attrstat,(char *)&rv);
}

msg_setattr(x)
XDR *x;
{
 FHANDLE fh;
 SATTR sa;
 ATTRSTAT rv;

 if (!xdr_fhandle(x,&fh) || !xdr_sattr(x,&sa))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"setattr(%x/%d/%d)",
			(int)fh.s.dev,(int)fh.s.ino,(int)fh.s.usecount);
 nfs_setattr(&au,&fh,&sa,&rv);
 success(xdr_attrstat,(char *)&rv);
}

msg_lookup(x)
XDR *x;
{
 DIROPARGS doa;
 DIROPRES rv;

 if (! xdr_diropargs(x,&doa))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"lookup(%x/%d/%d,\"%.*s\")",
	(int)doa.dir.s.dev,(int)doa.dir.s.ino,(int)doa.dir.s.usecount,
	doa.name.len,doa.name.name);
 nfs_lookup(&au,&doa,&rv);
 success(xdr_diropres,(char *)&rv);
}

static bool_t xdr__readlink_reply(x,rrp)
XDR *x;
struct _readlink_reply *rrp;
{
 static XDR_DISCRIM rrxd[] = { { NFS_OK, xdr_path },
			       { __dontcare__, 0 } };

 return(xdr_union(x,&rrp->status,(char *)&rrp->u,rrxd,xdr_void));
}

msg_readlink(x)
XDR *x;
{
 FHANDLE fh;
 struct _readlink_reply rr;

 if (! xdr_fhandle(x,&fh))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"readlink(%x/%d/%d)",
		(int)fh.s.dev,(int)fh.s.ino,(int)fh.s.usecount);
 nfs_readlink(&au,&fh,&rr);
 success(xdr__readlink_reply,(char *)&rr);
}

static bool_t xdr__read_reply_ok(x,rrop)
XDR *x;
struct _read_reply_ok *rrop;
{
 char *cp;

 if (x->x_op == XDR_FREE)
  { return(xdr_fattr(x,&rrop->attributes));
  }
 cp = &rrop->data[0];
 return( xdr_fattr(x,&rrop->attributes) &&
	 xdr_bytes(x,&cp,&rrop->datalen,MAXDATA) );
}

static bool_t xdr__read_reply(x,rrp)
XDR *x;
struct _read_reply *rrp;
{
 static XDR_DISCRIM rrxd[] = { { NFS_OK, xdr__read_reply_ok },
			       { __dontcare__, 0 } };

 return(xdr_union(x,&rrp->status,(char *)&rrp->u,rrxd,xdr_void));
}

msg_read(x)
XDR *x;
{
 FHANDLE fh;
 unsigned int offset;
 unsigned int count;
 unsigned int totalcount; /* ignored */
 struct _read_reply rr;

 if ( !xdr_fhandle(x,&fh) ||
      !xdr_u_int(x,&offset) ||
      !xdr_u_int(x,&count) ||
      !xdr_u_int(x,&totalcount) )
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"read(%x/%d/%d,%d,%d)",
		(int)fh.s.dev,(int)fh.s.ino,(int)fh.s.usecount,
		(int)offset,(int)count);
 nfs_read(&au,&fh,offset,count,&rr);
 success(xdr__read_reply,(char *)&rr);
}

msg_write(x)
XDR *x;
{
 FHANDLE fh;
 unsigned int beginoffset; /* ignored */
 unsigned int offset;
 unsigned int count;
 unsigned int totalcount; /* ignored */
 int datalen;
 char data[MAXDATA];
 char *cp;
 ATTRSTAT rv;

 cp = &data[0];
 if ( !xdr_fhandle(x,&fh) ||
      !xdr_u_int(x,&beginoffset) ||
      !xdr_u_int(x,&offset) ||
      !xdr_u_int(x,&totalcount) ||
      !xdr_bytes(x,&cp,&datalen,MAXDATA) )
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"write(%x/%d/%d,%d,%d)",
		(int)fh.s.dev,(int)fh.s.ino,(int)fh.s.usecount,
		(int)offset,datalen);
 nfs_write(&au,&fh,offset,datalen,data,&rv);
 success(xdr_attrstat,(char *)&rv);
}

msg_create(x)
XDR *x;
{
 DIROPARGS doa;
 SATTR sa;
 DIROPRES rv;

 if (!xdr_diropargs(x,&doa) || !xdr_sattr(x,&sa))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"create(%x/%d/%d,\"%.*s\")",
	(int)doa.dir.s.dev,(int)doa.dir.s.ino,(int)doa.dir.s.usecount,
	doa.name.len,doa.name.name);
 nfs_create(&au,&doa,&sa,&rv);
 success(xdr_diropres,(char *)&rv);
}

msg_remove(x)
XDR *x;
{
 DIROPARGS doa;
 enum_t status;

 if (! xdr_diropargs(x,&doa))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"remove(%x/%d/%d,\"%.*s\")",
	(int)doa.dir.s.dev,(int)doa.dir.s.ino,(int)doa.dir.s.usecount,
	doa.name.len,doa.name.name);
 nfs_remove(&au,&doa,&status);
 success(xdr_enum,(char *)&status);
}

msg_rename(x)
XDR *x;
{
 DIROPARGS from;
 DIROPARGS to;
 enum_t status;

 if (!xdr_diropargs(x,&from) || !xdr_diropargs(x,&to))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"rename(%x/%d/%d,\"%.*s\",%x/%d/%d,\"%.*s\")",
	(int)from.dir.s.dev,(int)from.dir.s.ino,(int)from.dir.s.usecount,
	from.name.len,from.name.name,
	(int)to.dir.s.dev,(int)to.dir.s.ino,(int)to.dir.s.usecount,
	to.name.len,to.name.name );
 nfs_rename(&au,&from,&to,&status);
 success(xdr_enum,(char *)&status);
}

msg_link(x)
XDR *x;
{
 FHANDLE from;
 DIROPARGS to;
 enum_t status;

 if (!xdr_fhandle(x,&from) || !xdr_diropargs(x,&to))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"link(%x/%d/%d,%x/%d/%d,\"%.*s\")",
	(int)from.s.dev,(int)from.s.ino,(int)from.s.usecount,
	(int)to.dir.s.dev,(int)to.dir.s.ino,(int)to.dir.s.usecount,
	to.name.len,to.name.name);
 nfs_link(&au,&from,&to,&status);
 success(xdr_enum,(char *)&status);
}

msg_symlink(x)
XDR *x;
{
 DIROPARGS from;
 PATH to;
 SATTR sa; /* ignored, on a UNIX server */
 enum_t status;

 if (!xdr_diropargs(x,&from) || !xdr_path(x,&to) || !xdr_sattr(x,&sa))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"symlink(%x/%d/%d,\"%.*s\",\"%.*s\")",
	(int)from.dir.s.dev,(int)from.dir.s.ino,(int)from.dir.s.usecount,
	from.name.len,from.name.name,
	(int)to.len,to.path);
 nfs_symlink(&au,&from,&to,&status);
 success(xdr_enum,(char *)&status);
}

msg_mkdir(x)
XDR *x;
{
 DIROPARGS where;
 SATTR sa;
 DIROPRES rv;

 if (!xdr_diropargs(x,&where) || !xdr_sattr(x,&sa))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"mkdir(%x/%d/%d,\"%.*s\")",
	(int)where.dir.s.dev,(int)where.dir.s.ino,(int)where.dir.s.usecount,
	where.name.len,where.name.name);
 nfs_mkdir(&au,&where,&sa,&rv);
 success(xdr_diropres,(char *)&rv);
}

msg_rmdir(x)
XDR *x;
{
 DIROPARGS which;
 enum_t status;

 if (! xdr_diropargs(x,&which))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"rmdir(%x/%d/%d,\"%.*s\")",
	(int)which.dir.s.dev,(int)which.dir.s.ino,(int)which.dir.s.usecount,
	which.name.len,which.name.name);
 nfs_rmdir(&au,&which,&status);
 success(xdr_enum,(char *)&status);
}

static bool_t xdr__readdir_reply(x,rrp)
XDR *x;
struct _readdir_reply *rrp;
{
 return(xdr_opaque(x,rrp->buf,rrp->len));
}

static bool_t xdr__readdir_entry(x,rep)
XDR *x;
struct _readdir_entry *rep;
{
 return( xdr_u_int(x,&rep->fileid) &&
	 xdr_filename(x,&rep->name) &&
	 xdr_cookie(x,&rep->cookie) );
}

msg_readdir(x)
XDR *x;
{
 FHANDLE dir;
 COOKIE cookie;
 unsigned int count;
 struct _readdir_reply rr;
 struct _readdir_entry re;
 char obuf[1024];
 char dbuf[1024];
 int dblen;
 int entrypos;
 int dpos;
 int iseof;
 XDR rx;
 struct direct *d;
 int didany;
 int i;
 bool_t v;
 int blkoff;
 struct stat stb;
/* SLOP is space for two booleans, at least. */
#define SLOP 16

 if (!xdr_fhandle(x,&dir) || !xdr_cookie(x,&cookie) || !xdr_u_int(x,&count))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"readdir(%x/%d/%d,%d,%d)",
	(int)dir.s.dev,(int)dir.s.ino,(int)dir.s.usecount,
	(int)cookie.offset,(int)count);
 didany = 0;
 if (count > sizeof(obuf))
  { count = sizeof(obuf);
  }
 xdrmem_create(&rx,obuf,count-SLOP,XDR_ENCODE);
  { enum_t status;
    status = NFS_OK;
    xdr_enum(&rx,&status);
  }
 dblen = 0;
 dpos = 0;
 iseof = 0;
 while (1)
  { entrypos = xdr_getpos(&rx);
    syslog_(LOG_DEBUG,"top, entrypos %d, dblen %d, dpos %d",entrypos,dblen,dpos);
    if (dblen == 0)
     { dblen = nfs_readdir(&au,&dir,cookie.offset,dbuf,sizeof(dbuf));
       syslog_(LOG_DEBUG,"nfs_readdir at %d returned %d",cookie.offset,dblen);
       if (dblen == 0)
	{ iseof = 1;
	  break;
	}
       dpos = 0;
       blkoff = cookie.offset;
     }
    if (dblen < 0)
     { enum_t status;
       status = errno;
       xdr_destroy(&rx);
       xdrmem_create(&rx,obuf,count-SLOP,XDR_ENCODE);
       xdr_enum(&rx,&status);
       rr.len = xdr_getpos(&rx);
       xdr_destroy(&rx);
       goto out;
     }
    if (dpos == dblen)
     { syslog_(LOG_DEBUG,"out of directory data");
       dblen = 0;
       continue;
     }
    /* is the rhs legal? teehee... */
    if (dblen-dpos < (int)&(((struct direct *)0)->d_name[0]))
     { syslog_(LOG_DEBUG,"not enough space for an entry");
       errno = EIO;
       continue;
     }
    d = (struct direct *) &dbuf[dpos];
    syslog_(LOG_DEBUG,"entry ino %d reclen %d namlen %d name %.*s",(int)d->d_ino,(int)d->d_reclen,(int)d->d_namlen,(int)d->d_namlen,d->d_name);
    if (dpos+d->d_reclen > dblen)
     { dblen = (dpos == 0) ? -1 : 0;
       errno = EIO;
       syslog_(LOG_DEBUG,"record overflows data");
       continue;
     }
    dpos += d->d_reclen;
    cookie.offset = blkoff + dpos;
    if (d->d_ino == 0)
     { syslog_(LOG_DEBUG,"zero inumber");
       continue;
     }
    v = TRUE;
    if (! xdr_bool(&rx,&v))
     { syslog_(LOG_DEBUG,"can't xdr first boolean");
       break;
     }
    /* Slow, but we prefer slow-&-correct over fast-&-wrong */
    /*  should be optimized by reading /etc/mtab */
    if (nfssvc_lookup(&dir,&au,&d->d_name[0],&stb) == 0)
     { re.fileid = stb.st_ino + (dev_to_fsno(stb.st_dev) << FSID_SHIFT);
     }
    else
     { re.fileid = d->d_ino + (dev_to_fsno(dir.s.dev) << FSID_SHIFT);
     }
    re.name.len = d->d_namlen;
    bcopy(&d->d_name[0],re.name.name,re.name.len);
    re.cookie.offset = cookie.offset;
    if (! xdr__readdir_entry(&rx,&re))
     { syslog_(LOG_DEBUG,"can't xdr entry");
       break;
     }
    didany ++;
  }
 syslog_(LOG_DEBUG,"loop exit, didany %d iseof %d",didany,iseof);
 xdr_destroy(&rx);
 xdrmem_create(&rx,obuf+entrypos,SLOP,XDR_ENCODE);
 v = FALSE;
 xdr_bool(&rx,&v);
 v = (iseof || !didany) ? TRUE : FALSE;
 xdr_bool(&rx,&v);
 rr.len = entrypos + xdr_getpos(&rx);
 xdr_destroy(&rx);
out:
 rr.buf = obuf;
 success(xdr__readdir_reply,(char *)&rr);
}

static bool_t xdr__statfs_reply_ok(x,srop)
XDR *x;
struct _statfs_reply_ok *srop;
{
 return( xdr_u_int(x,&srop->tsize) &&
	 xdr_u_int(x,&srop->bsize) &&
	 xdr_u_int(x,&srop->blocks) &&
	 xdr_u_int(x,&srop->bfree) &&
	 xdr_u_int(x,&srop->bavail) );
}

static bool_t xdr__statfs_reply(x,srp)
XDR *x;
struct _statfs_reply *srp;
{
 static XDR_DISCRIM srxd[] = { { NFS_OK, xdr__statfs_reply_ok },
			       { __dontcare__, 0 } };

 return(xdr_union(x,&srp->status,(char *)&srp->u,srxd,xdr_void));
}

msg_statfs(x)
XDR *x;
{
 FHANDLE fh;
 struct _statfs_reply fr;

 if (! xdr_fhandle(x,&fh))
  { garbage_args();
    return;
  }
 if (! authunix())
  { return;
  }
 syslog_(LOG_DEBUG,"statfs(%x/%d/%d)",
	(int)fh.s.dev,(int)fh.s.ino,(int)fh.s.usecount);
 nfs_statfs(&au,&fh,&fr);
 success(xdr__statfs_reply,(char *)&fr);
}

enum_t maperrno()
{
 switch (errno)
  { default:
       syslog_(LOG_WARNING,"maperrno(%d): unknown error (%m)",errno);
       return(NFSERR_ACCES);
       break;
    case EPERM:
    case ENOENT:
    case EIO:
    case ENXIO:
    case EACCES:
    case EEXIST:
    case ENODEV:
    case ENOTDIR:
    case EISDIR:
    case EFBIG:
    case ENOSPC:
    case EROFS:
    case ENAMETOOLONG:
    case ENOTEMPTY:
    case EDQUOT:
       return(errno);
       break;
  }
}

enum_t maptype(stmode)
unsigned short int stmode;
{
 switch (stmode & S_IFMT)
  { default:
       syslog_(LOG_ERR,"unknown file mode %o",(int)stmode);
       return(NFNON);
       break;
    case S_IFDIR:
       return(NFDIR);
       break;
    case S_IFCHR:
       return(NFCHR);
       break;
    case S_IFBLK:
       return(NFBLK);
       break;
    case S_IFREG:
       return(NFREG);
       break;
    case S_IFLNK:
       return(NFLNK);
       break;
    case S_IFSOCK:
       return(NFNON);
       break;
  }
}

nfs_getattr(au,fh,rv)
struct auth_unix *au;
FHANDLE *fh;
ATTRSTAT *rv;
{
 struct stat stb;

 if (nfssvc_stat(fh,au,&stb) < 0)
  { rv->status = maperrno();
    return;
  }
 rv->status = NFS_OK;
 stb_to_fattr(&stb,&rv->u.attributes);
}

nfs_setattr(au,fh,sa,rv)
struct auth_unix *au;
FHANDLE *fh;
SATTR *sa;
ATTRSTAT *rv;
{
 struct stat stb;

 if (nfssvc_setattr(fh,au,sa,&stb) < 0)
  { rv->status = maperrno();
    return;
  }
 rv->status = NFS_OK;
 stb_to_fattr(&stb,&rv->u.attributes);
}

nfs_lookup(au,doa,rv)
struct auth_unix *au;
DIROPARGS *doa;
DIROPRES *rv;
{
 struct stat stb;

 doa->name.name[doa->name.len] = '\0';
 if (nfssvc_lookup(&doa->dir,au,doa->name.name,&stb) < 0)
  { rv->status = maperrno();
  }
 else
  { rv->status = NFS_OK;
    stb_to_fh(&stb,&rv->u.ok.file);
    syslog_(LOG_DEBUG,"lookup: returning %x/%d/%d",
	(int)rv->u.ok.file.s.dev,
	(int)rv->u.ok.file.s.ino,
	(int)rv->u.ok.file.s.usecount );
    stb_to_fattr(&stb,&rv->u.ok.attributes);
  }
}

nfs_readlink(au,fh,rr)
struct auth_unix *au;
FHANDLE *fh;
struct _readlink_reply *rr;
{
 char linkbuf[MAXPATHLEN];
 int rv;

 rv = nfssvc_readlink(fh,au,linkbuf,sizeof(linkbuf));
 if (rv < 0)
  { rr->status = maperrno();
    return;
  }
 rr->status = NFS_OK;
 rr->u.path.len = rv;
 bcopy(linkbuf,rr->u.path.path,rv);
}

nfs_read(au,fh,offset,count,rr)
struct auth_unix *au;
FHANDLE *fh;
unsigned int offset;
unsigned int count;
struct _read_reply *rr;
{
 int rv;
 struct stat stb;

 if (count > MAXDATA)
  { count = MAXDATA;
  }
 rv = nfssvc_read(fh,au,offset,rr->u.ok.data,count,&stb);
 if (rv < 0)
  { rr->status = maperrno();
  }
 else
  { rr->status = NFS_OK;
    rr->u.ok.datalen = rv;
    stb_to_fattr(&stb,&rr->u.ok.attributes);
  }
}

nfs_write(au,fh,offset,datalen,data,rv)
struct auth_unix *au;
FHANDLE *fh;
unsigned int offset;
int datalen;
char *data;
ATTRSTAT *rv;
{
 struct stat stb;

 if (nfssvc_write(fh,au,offset,data,datalen,&stb) < 0)
  { rv->status = maperrno();
  }
 else
  { rv->status = NFS_OK;
    stb_to_fattr(&stb,&rv->u.attributes);
  }
}

nfs_create(au,doa,sa,rv)
struct auth_unix *au;
DIROPARGS *doa;
SATTR *sa;
DIROPRES *rv;
{
 struct stat stb;

 doa->name.name[doa->name.len] = '\0';
 if (nfssvc_create(&doa->dir,au,doa->name.name,sa,&stb) < 0)
  { rv->status = maperrno();
  }
 else
  { rv->status = NFS_OK;
    stb_to_fh(&stb,&rv->u.ok.file);
    stb_to_fattr(&stb,&rv->u.ok.attributes);
  }
}

nfs_remove(au,doa,status)
struct auth_unix *au;
DIROPARGS *doa;
enum_t *status;
{
 doa->name.name[doa->name.len] = '\0';
 if (nfssvc_remove(&doa->dir,au,doa->name.name) < 0)
  { *status = maperrno();
  }
 else
  { *status = NFS_OK;
  }
}

nfs_rename(au,from,to,status)
struct auth_unix *au;
DIROPARGS *from;
DIROPARGS *to;
enum_t *status;
{
 from->name.name[from->name.len] = '\0';
 to->name.name[to->name.len] = '\0';
 if (nfssvc_rename(&from->dir,from->name.name,au,&to->dir,to->name.name) < 0)
  { *status = maperrno();
  }
 else
  { *status = NFS_OK;
  }
}

nfs_link(au,from,to,status)
struct auth_unix *au;
FHANDLE *from;
DIROPARGS *to;
enum_t *status;
{
 to->name.name[to->name.len] = '\0';
 if (nfssvc_link(from,au,&to->dir,to->name.name) < 0)
  { *status = maperrno();
  }
 else
  { *status = NFS_OK;
  }
}

nfs_symlink(au,from,to,status)
struct auth_unix *au;
DIROPARGS *from;
PATH *to;
enum_t *status;
{
 from->name.name[from->name.len] = '\0';
 to->path[to->len] = '\0';
 if (nfssvc_symlink(&from->dir,from->name.name,au,to->path) < 0)
  { *status = maperrno();
  }
 else
  { *status = NFS_OK;
  }
}

nfs_mkdir(au,where,sa,rv)
struct auth_unix *au;
DIROPARGS *where;
SATTR *sa;
DIROPRES *rv;
{
 struct stat stb;

 where->name.name[where->name.len] = '\0';
 if (nfssvc_mkdir(&where->dir,au,where->name.name,sa,&stb) < 0)
  { rv->status = maperrno();
  }
 else
  { rv->status = NFS_OK;
    stb_to_fh(&stb,&rv->u.ok.file);
    stb_to_fattr(&stb,&rv->u.ok.attributes);
  }
}

nfs_rmdir(au,which,status)
struct auth_unix *au;
DIROPARGS *which;
enum_t *status;
{
 which->name.name[which->name.len] = '\0';
 if (nfssvc_rmdir(&which->dir,au,which->name.name) < 0)
  { *status = maperrno();
  }
 else
  { *status = NFS_OK;
  }
}

int nfs_readdir(au,dir,offset,dbuf,dblen)
struct auth_unix *au;
FHANDLE *dir;
unsigned int offset;
char *dbuf;
int dblen;
{
 return(nfssvc_read(dir,au,offset,dbuf,dblen,(struct stat *)0));
}

nfs_statfs(au,fh,fr)
struct auth_unix *au;
FHANDLE *fh;
struct _statfs_reply *fr;
{
 if (nfssvc_statfs(fh,au,&fr->u.ok) < 0)
  { fr->status = maperrno();
  }
 else
  { fr->status = NFS_OK;
    fr->u.ok.tsize = MAXDATA;
  }
}
