#include <stdio.h>
#include <ctype.h>
#include <netdb.h>
#include <errno.h>
#include <signal.h>
#include <syslog.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <xdr.h>
#include <rpc.h>
#include <nfs.h>

#include <htable.h>
#include <malloc.h>

char exportlist[] = "/local/etc/exports";

extern int errno;

typedef struct {
	  int i1;
	  int i2;
	  int i3;
	  int i4; } FOURINTS;

char pmap_service[] = "sunrpc";
char packet[10240];
int packetlen;
char reply[10240];
int sock;
RPC_MSG req;
struct sockaddr_in from;
RPC_MSG rep;
int shoulddie;
int lastkillsig;

int rpc_errno;

bool_t xdr_4ints(x,fi)
XDR *x;
FOURINTS *fi;
{
 return( xdr_int(x,&fi->i1) &&
	 xdr_int(x,&fi->i2) &&
	 xdr_int(x,&fi->i3) &&
	 xdr_int(x,&fi->i4) );
}

char *rpc_errorstr()
{
 static char *rpc_errs[] = { "Error 0",
			     "RPC version mismatch",
			     "Bad credentials",
			     "Rejected credentials",
			     "Bad verifier",
			     "Rejected verifier",
			     "Insufficient privilege",
			     "Unsupported program version",
			     "Program unavailable",
			     "Unsupported procedure number",
			     "Bad argument format",
			     "Bad reply format",
			     "Timed out" };
 static int rpc_nerr = sizeof(rpc_errs) / sizeof(rpc_errs[0]);
 static char unknbuf[64];

 if ((rpc_errno < 0) || (rpc_errno >= rpc_nerr))
  { sprintf(unknbuf,"Unknown RPC error %d",rpc_errno);
    return(unknbuf);
  }
 else
  { return(rpc_errs[rpc_errno]);
  }
}

bool_t rpc_call(s,port,addr,prog,vers,proc,argx,arga,retx,reta)
int s;
int port;
struct in_addr addr;
int prog;
int vers;
int proc;
bool_t (*argx)();
char *arga;
bool_t (*retx)();
char *reta;
{
 XDR x;
 int rv;
 struct sockaddr_in sin;
 int tries;
 fd_set fds;
 struct timeval delay;

 req.xid = get_rpcxid();
 req.type = CALL;
 req.u.call.rpcvers = 2;
 req.u.call.prog = prog;
 req.u.call.vers = vers;
 req.u.call.proc = proc;
 req.u.call.cred.type = AUTH_NULL;
 req.u.call.cred.len = 0;
 req.u.call.verf.type = AUTH_NULL;
 req.u.call.verf.len = 0;
 xdrmem_create(&x,packet,sizeof(packet),XDR_ENCODE);
 if ( !xdr_rpc_msg(&x,&req) ||
      !(*argx)(&x,arga) )
  { xdr_destroy(&x);
    return(FALSE);
  }
 packetlen = xdr_getpos(&x);
 xdr_destroy(&x);
 sin.sin_family = AF_INET;
 sin.sin_addr = addr;
 sin.sin_port = ntohs(port);
 for (tries=0;tries<5;tries++)
  { rv = sendto(s,packet,packetlen,0,(char *)&sin,sizeof(sin));
    if (rv < 0)
     { syslog(LOG_ERR,"can't send rpc request to %s/%d",inet_ntoa(addr),port);
       return(FALSE);
     }
    FD_ZERO(&fds);
    FD_SET(s,&fds);
    delay.tv_sec = 0;
    delay.tv_usec = 500000;
    rv = select(FD_SETSIZE,&fds,(fd_set *)0,(fd_set *)0,&delay);
    if (rv == 0)
     { continue;
     }
    if (rv < 0)
     { if (errno != EINTR)
	{ syslog(LOG_ERR,"can't select() for rpc reply from %s/%d: %m",
							inet_ntoa(addr),port);
	}
     }
    if (! FD_ISSET(s,&fds))
     { continue;
     }
    rv = recvfrom(s,reply,sizeof(reply),0,(char *)0,(int *)0);
    xdrmem_create(&x,reply,rv,XDR_DECODE);
    if (! xdr_rpc_msg(&x,&rep))
     {
badreply:
       syslog(LOG_WARNING,"bad rpc reply from %s/%d",inet_ntoa(addr),port);
       xdr_destroy(&x);
       continue;
     }
    if (rep.xid != req.xid)
     { syslog(LOG_INFO,"xid mismatch from %s/%d",inet_ntoa(addr),port);
       xdr_destroy(&x);
       continue;
     }
    if (rep.type != REPLY)
     { syslog(LOG_WARNING,"non-reply rpc reply from %s/%d",inet_ntoa(addr),port);
       xdr_destroy(&x);
       continue;
     }
    switch (rep.u.reply.type)
     { case MSG_DENIED:
	  switch (rep.u.reply.u.denied.type)
	   { case RPC_MISMATCH:
		rpc_errno = RPCERR_RPCMISMATCH;
		goto lose;
		break;
	     case AUTH_ERROR:
		switch (rep.u.reply.u.denied.u.error)
		 { case AUTH_BADCRED:
		      rpc_errno = RPCERR_BADCRED;
		      goto lose;
		      break;
		   case AUTH_REJECTEDCRED:
		      rpc_errno = RPCERR_REJECTEDCRED;
		      goto lose;
		      break;
		   case AUTH_BADVERF:
		      rpc_errno = RPCERR_BADVERF;
		      goto lose;
		      break;
		   case AUTH_REJECTEDVERF:
		      rpc_errno = RPCERR_REJECTEDVERF;
		      goto lose;
		      break;
		   case AUTH_TOOWEAK:
		      rpc_errno = RPCERR_TOOWEAK;
		      goto lose;
		      break;
		   default:
		      goto badreply;
		      break;
		 }
		break;
	     default:
		goto badreply;
		break;
	   }
	  break;
       case MSG_ACCEPTED:
	  switch (rep.u.reply.u.accepted.type)
	   { case SUCCESS:
		if (! (*retx)(&x,reta))
		 { rpc_errno = RPCERR_GARBAGEREPLY;
		   goto lose;
		 }
		else
		 { goto win;
		 }
		break;
	     case PROG_MISMATCH:
		rpc_errno = RPCERR_PROGMISMATCH;
		goto lose;
		break;
	     case PROG_UNAVAIL:
		rpc_errno = RPCERR_PROGUNAVAIL;
		goto lose;
		break;
	     case PROC_UNAVAIL:
		rpc_errno = RPCERR_PROCUNAVAIL;
		goto lose;
		break;
	     case GARBAGE_ARGS:
		rpc_errno = RPCERR_GARBAGEARGS;
		goto lose;
		break;
	     default:
		goto badreply;
		break;
	   }
	  break;
       default:
	  goto badreply;
	  break;
     }
  }
 rpc_errno = RPCERR_TIMEDOUT;
 return(FALSE);
lose:
 xdr_destroy(&x);
 return(FALSE);
win:
 xdr_destroy(&x);
 rpc_errno = RPCERR_NONE;
 return(TRUE);
}

killsig(sig)
int sig;
{
 lastkillsig = sig;
 shoulddie ++;
}

main()
{
 struct sockaddr_in sin;
 int sinlen;
 struct servent *sp;
 int i;
 FOURINTS fi;

 openlog("mountd",LOG_PID|LOG_CONS,LOG_DAEMON);
 i = fork();
 if (i < 0)
  { perror("fork");
  }
 if (i)
  { exit(0);
  }
 i = open("/dev/tty",O_RDWR,0);
 if (i >= 0)
  { ioctl(i,TIOCNOTTY,0);
    close(i);
  }
 sock = socket(AF_INET,SOCK_DGRAM,0);
 if (sock < 0)
  { syslog(LOG_ERR,"can't create socket: %m");
    exit(1);
  }
 sp = getservbyname(pmap_service,"udp");
 if (sp == 0)
  { syslog(LOG_ERR,"%s/udp service unknown",pmap_service);
    exit(1);
  }
 sin.sin_family = AF_INET;
 sin.sin_addr.s_addr = INADDR_ANY;
 sin.sin_port = 0;
 if (bind(sock,(char *)&sin,sizeof(sin)) < 0)
  { syslog(LOG_ERR,"can't bind socket: %m");
    exit(1);
  }
 sinlen = sizeof(sin);
 if (getsockname(sock,(char *)&sin,&sinlen) < 0)
  { syslog(LOG_ERR,"can't get socket address: %m");
    exit(1);
  }
 fi.i1 = MOUNT_PROGRAM;
 fi.i2 = MOUNT_VERSION;
 fi.i3 = 0;
 fi.i4 = 0;
 if (! rpc_call( sock,
		 111, sin.sin_addr,
		 100000, 2, 2,
		 xdr_4ints, (char *)&fi,
		 xdr_bool,(bool_t *)&i ) )
  { syslog(LOG_ERR,"can't delete mapping: %s",rpc_errorstr());
    exit(1);
  }
 /* don't check i here, probably 0 'cause probably not registered */
 fi.i1 = MOUNT_PROGRAM;
 fi.i2 = MOUNT_VERSION;
 fi.i3 = IPPROTO_UDP;
 fi.i4 = ntohs(sin.sin_port);
 if (! rpc_call( sock,
		 111, sin.sin_addr,
		 100000, 2, 1,
		 xdr_4ints, (char *)&fi,
		 xdr_bool,(bool_t *)&i ) )
  { syslog(LOG_ERR,"can't install mapping: %s",rpc_errorstr());
    exit(1);
  }
 if (! i)
  { syslog(LOG_ERR,"portmapper refused to install mapping");
    exit(1);
  }
 shoulddie = 0;
 signal(SIGHUP,killsig);
 signal(SIGINT,killsig);
 signal(SIGTERM,killsig);
 signal(SIGTSTP,killsig);
 signal(SIGTTIN,killsig);
 signal(SIGTTOU,killsig);
 signal(SIGXCPU,killsig);
 signal(SIGXFSZ,killsig);
  { long int now;
    time(&now);
    syslog(LOG_INFO,"startup %.24s",ctime(&now));
  }
 i = 0;
 while (1)
  { if (shoulddie)
     { goto die;
     }
    sinlen = sizeof(sin);
    packetlen = recvfrom(sock,packet,sizeof(packet),0,(char *)&sin,&sinlen);
    if (packetlen < 0)
     { if (errno == EINTR)
	{ continue;
	}
       syslog(LOG_ERR,"error in recvfrom(): %m");
badpacket:
       i ++;
       if (i > 10)
	{ goto die;
	}
       sleep(2);
       continue;
     }
    if (packetlen == 0)
     { syslog(LOG_WARNING,"zero-size input packet");
       goto badpacket;
     }
    if (sinlen != sizeof(sin))
     { syslog(LOG_WARNING,"bad from-address size (%d bytes)",sinlen);
       goto badpacket;
     }
    i -= 4;
    if (i < 0)
     { i = 0;
     }
    from = sin;
    inputpacket(packet,packetlen);
  }
die:
 syslog(LOG_INFO,"dying: received signal %d",lastkillsig);
 sinlen = sizeof(sin);
 if (getsockname(sock,(char *)&sin,&sinlen) < 0)
  { syslog(LOG_ERR,"can't get socket address: %m");
    exit(1);
  }
 fi.i1 = MOUNT_PROGRAM;
 fi.i2 = MOUNT_VERSION;
 fi.i3 = 0;
 fi.i4 = 0;
 if (! rpc_call( sock,
		 111, sin.sin_addr,
		 100000, 2, 2,
		 xdr_4ints, (char *)&fi,
		 xdr_bool,(bool_t *)&i ) )
  { syslog(LOG_ERR,"can't delete mapping: %s",rpc_errorstr());
    exit(1);
  }
 if (! i)
  { syslog(LOG_ERR,"portmapper refused to delete mapping");
    exit(1);
  }
 exit(0);
}

inputpacket(pkt,len)
char *pkt;
int len;
{
 XDR x;

 xdrmem_create(&x,pkt,len,XDR_DECODE);
 if (! xdr_rpc_msg(&x,&req))
  { syslog(LOG_WARNING,"XDR failed on input packet from %s/%d",
			inet_ntoa(from.sin_addr),ntohs(from.sin_port));
    goto out;
  }
 syslog(LOG_DEBUG,"input packet from %s/%d",
			inet_ntoa(from.sin_addr),ntohs(from.sin_port));
 rep.xid = req.xid;
 rep.type = REPLY;
 if (req.type != CALL)
  { syslog(LOG_INFO,"non-call packet from %s/%d",
			inet_ntoa(from.sin_addr),ntohs(from.sin_port));
    goto out;
  }
 if (req.u.call.rpcvers != 2)
  { syslog(LOG_WARNING,"bad RPC version (%d) from %s/%d",
	req.u.call.rpcvers,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
    rpc_mismatch();
    goto out;
  }
 if (req.u.call.prog != MOUNT_PROGRAM)
  { syslog(LOG_WARNING,"non-mountd packet (program %d) from %s/%d",
	req.u.call.prog,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
    goto out;
  }
 if (req.u.call.vers != MOUNT_VERSION)
  { syslog(LOG_WARNING,"bad mountd version (%d) from %s/%d",
	req.u.call.vers,inet_ntoa(from.sin_addr),ntohs(from.sin_port));
    vers_mismatch();
    goto out;
  }
 switch (req.u.call.proc)
  { case 0:
       msg_null(&x);
       break;
    case 1:
       msg_mnt(&x);
       break;
    case 2:
       msg_dump(&x);
       break;
    case 3:
       msg_umnt(&x);
       break;
    case 4:
       msg_umntall(&x);
       break;
    case 5:
       msg_export(&x);
       break;
    default:
       badproc();
       break;
  }
out:
 xdr_destroy(&x);
 return;
}

sendreply(xextra,extraarg)
bool_t (*xextra)();
char *extraarg;
{
 XDR x;
 int len;
 int sent;

 xdrmem_create(&x,reply,sizeof(reply),XDR_ENCODE);
 if (xdr_rpc_msg(&x,&rep) && (*xextra)(&x,extraarg))
  { len = xdr_getpos(&x);
    sent = sendto(sock,reply,len,0,&from,sizeof(struct sockaddr_in));
    if (sent < 0)
     { syslog(LOG_ERR,"can't return message to %s/%d: %m",
			inet_ntoa(from.sin_addr),ntohs(from.sin_port));
     }
    else if (sent != len)
     { syslog(LOG_WARNING,"sendto sent %d not %d",sent,len);
     }
  }
 else
  { syslog(LOG_ERR,"XDR can't encode return message");
  }
 xdr_destroy(&x);
}

rpc_mismatch()
{
 rep.u.reply.type = MSG_DENIED;
 rep.u.reply.u.denied.type = RPC_MISMATCH;
 rep.u.reply.u.denied.u.mismatch.low = 2;
 rep.u.reply.u.denied.u.mismatch.high = 2;
 sendreply(xdr_void,(char *)0);
}

vers_mismatch()
{
 rep.u.reply.type = MSG_ACCEPTED;
 rep.u.reply.u.accepted.verf.type = AUTH_NULL;
 rep.u.reply.u.accepted.verf.len = 0;
 rep.u.reply.u.accepted.type = PROG_MISMATCH;
 rep.u.reply.u.accepted.u.mismatch.low = MOUNT_VERSION;
 rep.u.reply.u.accepted.u.mismatch.high = MOUNT_VERSION;
 sendreply(xdr_void,(char *)0);
}

badproc()
{
 rep.u.reply.type = MSG_ACCEPTED;
 rep.u.reply.u.accepted.verf.type = AUTH_NULL;
 rep.u.reply.u.accepted.verf.len = 0;
 rep.u.reply.u.accepted.type = PROC_UNAVAIL;
 sendreply(xdr_void,(char *)0);
}

garbage_args()
{
 rep.u.reply.type = MSG_ACCEPTED;
 rep.u.reply.u.accepted.verf.type = AUTH_NULL;
 rep.u.reply.u.accepted.verf.len = 0;
 rep.u.reply.u.accepted.type = GARBAGE_ARGS;
 sendreply(xdr_void,(char *)0);
}

tooweak()
{
 rep.u.reply.type = MSG_DENIED;
 rep.u.reply.u.denied.type = AUTH_ERROR;
 rep.u.reply.u.denied.u.error = AUTH_TOOWEAK;
 sendreply(xdr_void,(char *)0);
}

success(xextra,extraarg)
bool_t (*xextra)();
char *extraarg;
{
 rep.u.reply.type = MSG_ACCEPTED;
 rep.u.reply.u.accepted.verf.type = AUTH_NULL;
 rep.u.reply.u.accepted.verf.len = 0;
 rep.u.reply.u.accepted.type = SUCCESS;
 sendreply(xextra,extraarg);
}

int authunix(aup)
struct auth_unix *aup;
{
 XDR x;

 if (req.u.call.cred.type != AUTH_UNIX)
  { tooweak();
    return(0);
  }
 xdrmem_create(&x,req.u.call.cred.body,req.u.call.cred.len,XDR_DECODE);
 if (! xdr_auth_unix(&x,aup))
  { xdr_destroy(&x);
    tooweak();
    return(0);
  }
 return(1);
}

msg_null(x)
XDR *x;
{
 syslog(LOG_DEBUG,"null()");
 success(xdr_void,(char *)0);
}

msg_mnt(x)
XDR *x;
{
 DIRPATH dn;
 FHSTATUS rv;
 struct auth_unix au;

 if (! xdr_dirpath(x,&dn))
  { garbage_args();
    return;
  }
 if (! authunix(&au))
  { return;
  }
 if (au.uid)
  { tooweak();
    return;
  }
 syslog(LOG_DEBUG,"mnt(\"%.*s\")",dn.len,dn.path);
 mount_mnt(&dn,&rv);
 success(xdr_fhstatus,(char *)&rv);
}

msg_dump(x)
XDR *x;
{
 bool_t rv;
 struct auth_unix au;

 syslog(LOG_DEBUG,"dump()");
 rv = FALSE;
 success(xdr_bool,(char *)&rv);
}

msg_umnt(x)
XDR *x;
{
 DIRPATH dn;

 if (! xdr_dirpath(x,&dn))
  { garbage_args();
    return;
  }
 syslog(LOG_DEBUG,"umnt(\"%.*s\")",dn.len,dn.path);
 success(xdr_void,(char *)0);
}

msg_umntall(x)
XDR *x;
{
 syslog(LOG_DEBUG,"umntall()");
 success(xdr_void,(char *)0);
}

msg_export(x)
XDR *x;
{
 badproc();
}

mount_mnt(dn,rv)
DIRPATH *dn;
FHSTATUS *rv;
{
 struct stat stb;
 struct hostent *hp;
 FILE *f;
 char buf[1024];
 register char *cp;
 char *hn;
 char *path;
 int ok;

 hp = gethostbyaddr(&from.sin_addr,sizeof(struct in_addr),AF_INET);
 if (hp == 0)
  { syslog(LOG_WARNING,"mount request from unknown machine: %s/%d",
				inet_ntoa(from.sin_addr),ntohs(from.sin_port));
    rv->status = EACCES;
    return;
  }
 f = fopen(exportlist,"r");
 if (f == 0)
  { syslog(LOG_ERR,"can't open export list \"%s\"",exportlist);
    rv->status = ENETDOWN;
    return;
  }
 dn->path[dn->len] = '\0';
 ok = 0;
 while (fgets(buf,sizeof(buf),f) == buf)
  { cp = index(buf,'#');
    if (cp)
     { *cp = '\0';
     }
    for (cp=buf;*cp&&isspace(*cp);cp++) ;
    if (! *cp)
     { continue;
     }
    path = cp;
    for (;*cp&&!isspace(*cp);cp++) ;
    if (*cp)
     { *cp++ = '\0';
     }
    if (!strcmp(path,dn->path) || !strcmp(path,"*"))
     { for (;*cp&&isspace(*cp);cp++) ;
       if (! *cp)
	{ ok = 1;
	  break;
	}
       while (*cp)
	{ hn = cp;
	  for (;*cp&&!isspace(*cp);cp++) ;
	  if (*cp)
	   { *cp++ = '\0';
	   }
	  if (!strcmp(hn,hp->h_name))
	   { ok = 1;
	     break;
	   }
	  for (;*cp&&isspace(*cp);cp++) ;
	}
       break;
     }
  }
 fclose(f);
 if (! ok)
  { rv->status = EACCES;
    return;
  }
 if (stat(dn->path,&stb) < 0)
  { rv->status = errno;
    return;
  }
 rv->status = 0;
 stb_to_fh(&stb,&rv->u.directory);
 syslog(LOG_DEBUG,"mounted, fh = %x/%d/%d",
	(int)rv->u.directory.s.dev,
	(int)rv->u.directory.s.ino,
	(int)rv->u.directory.s.usecount );
}

stb_to_fh(stb,fh)
struct stat *stb;
FHANDLE *fh;
{
 bzero(fh->bytes,FHSIZE);
 fh->s.dev = stb->st_dev;
 fh->s.ino = stb->st_ino;
 fh->s.usecount = stb->st_usecount;
}
