/* $Id: ctp.c,v 1.3 89/10/24 17:54:23 dupuy Exp $ */

/*
 * ctp - ethernet configuration test protocol program
 */

/*
 * derived from ethertools/ctp.c
 *
 * Copyright (c) 1988 Philip L. Budne and The Trustees of Boston University All
 * Rights Reserved
 *
 * Permission is granted to any individual or institution to use, copy, or
 * redistribute this software so long as it is not sold for profit, provided
 * that this notice and the original copyright notices are retained.  Boston
 * University makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 */

#include <stdio.h>
#include <signal.h>
#include <errno.h>

extern int errno;

#include "ctp.h"

#ifdef SIGABRT
#define sigtype void
#else
#define sigtype int
#endif

struct timeval *ether_timestamp;	/* timestamp from 3.x NIT, maybe */

/*
 * Ethernet related variables
 */

int nomulticast = 0;

#ifdef __STDC__
ether_addr ctpmulticast = {{0xcf, 0x0, 0x0, 0x0, 0x0, 0x0}};

#else
ether_addr ctpmulticast;
unsigned char bytes[6] = {0xcf, 0x0, 0x0, 0x0, 0x0, 0x0};

#endif

ether_addr localaddr;			/* our ethernet address */

/*
 * CTP statistics
 */

#define NBUCKETS 733			/* prime # of buckets */
#define hash(addr) \
 (((addr)->shorts[1] + ((addr)->shorts[1] ^ (addr)->shorts[2])) % NBUCKETS)

int nsent = 0;

struct
{
    int nrecvd;				/* total count */
    int highseq;			/* highest seq seen */
    int outofseq;			/* number recvd out of sequence */
    int min;
    int max;				/* min and max times */
    int sum;				/* sum of times */

    int valid;				/* is this a used entry */
    ether_addr eaddr;			/* ethernet address for this count */
}
       stats[NBUCKETS + 1];

int lastseq = 0;
int seq = 0;

/*
 * Program options and arguments
 */

char *interface = NULL;			/* interface to use */
int count = -1;				/* number of packets to send */
int noreturn = 0;			/* don't force final forward back */
int packlen = ETHER_MIN;		/* default total packet length */
int slptim = 1000;			/* sleep time in ms */

/*
 * Other functions and variables
 */

int pid;				/* our pid */
int tosend = 0;

short swaps ();
sigtype resend ();
sigtype die ();
void printstats ();
ctp_reply *output_packet ();
void input_packet ();


main (argc, argv)
int argc;
char *argv[];
{
    int efd;
    int optind;
    ether_addr path[MAXPATH];
    int pathcount = 0;
    ether_packet outpkt;
    ctp_packet outpktbuf;
    ctp_reply *reply;			/* pointer to ctp_reply in outpkt */
    struct itimerval interval;

    /*
     * Get program options
     */

    if ((optind = getoptions (argc, argv)) < 0 || optind >= argc)
    {
	(void) fprintf (stderr,
	  "Usage: %s [-c count] [-n] [-i intf] [-l len] [-s sleep] addr ...\n",
			argv[0]);
	exit (1);
    }

    /*
     * Open ethernet interface
     */

    if (interface == NULL)
    {
	char **intfs;

	if ((intfs = ether_interfaces ()) == NULL)
	{
	    perror ("ether_interfaces");
	    exit (1);
	}

	interface = *intfs;
    }

#ifndef __STDC__
    (void) bcopy ((char *) bytes, (char *) ctpmulticast.bytes, sizeof (bytes));
#endif

    if ((efd = ether_open (interface, CTPTYPE, &ctpmulticast)) < 0)
    {
	/* try again w/o multicast, for enetfilter systems */

	nomulticast++;

	if ((efd = ether_open (interface, CTPTYPE, (ether_addr *) 0)) < 0)
	{
	    perror (interface);
	    exit (1);
	}
    }

    /*
     * Set non-blocking, so that we can select and loop
     */

    if (ether_blocking (efd, 0) != 0)
    {
	perror ("ether_block");
	exit (1);
    }

    /*
     * Get local interface address and process id
     */

    if (ether_address (efd, &localaddr) == NULL)
    {
	perror ("ether_address");
	exit (1);
    }

    (void) printf ("%s address %s\n", interface, ether_ntoa (&localaddr));

    pid = getpid ();			/* so we can identify our packets */

    /*
     * Set up forwarding path for CTP packet
     */

    pathcount = 0;
    while (optind < argc && pathcount < MAXPATH && argv[optind] != NULL)
    {
	if (strcmp (argv[optind], "MULTICAST") == 0)
	    path[pathcount] = ctpmulticast;

	else if (strcmp (argv[optind], "BROADCAST") == 0)
	    path[pathcount] = ether_bcast_addr;

	else if (!ether_host2e (argv[optind], &path[pathcount]))
	{
	    extern char *sys_errlist[];
	    int saved_errno = errno;

	    (void) puts ("");
	    (void) fflush (stdout);
	    if (errno == ENOENT)
		(void) fprintf (stderr, "%s: unknown host %s\n",
				argv[0], argv[optind]);
	    else
		(void) fprintf (stderr, "%s: can't get address for %s: %s\n",
				argv[0], argv[optind],
				sys_errlist[saved_errno]);

	    exit (1);
	}

	(void) printf ("%s%s (%s)", ((pathcount > 0) ? "\n    " : "CTP "),
		       argv[optind], ether_ntoa (&path[pathcount]));

	pathcount++;
	optind++;
    }

    if (pathcount == MAXPATH)
    {
	fputs (": ", stdout);
	(void) fflush (stdout);
	(void) fprintf (stderr, "Too many hosts in path (%d)\n", MAXPATH);
	exit (1);
    }

    /*
     * Build output packet
     */

    reply = output_packet (efd, &outpkt, &outpktbuf, path, pathcount);

    /*
     * Set up signal handlers and timers
     */

    (void) signal (SIGALRM, resend);
    (void) signal (SIGINT, die);
    (void) signal (SIGHUP, die);
    (void) signal (SIGTERM, die);

    interval.it_interval.tv_sec = slptim / 1000;
    interval.it_interval.tv_usec = slptim % 1000;
    interval.it_value.tv_sec = slptim / 1000;
    interval.it_value.tv_usec = slptim % 1000;

    if (setitimer (ITIMER_REAL, &interval, (struct itimerval *) 0) < 0)
    {
	perror ("setitimer");
	exit (1);
    }

    /*
     * Loop checking for packets
     */

    while (count != 0)
    {
	fd_set fdset;

	if (tosend > 0)
	{
	    tosend--;

	    reply->seq = seq++;

	    /*
	     * It's safe to cast reply->sendt, since that will be long aligned
	     */

	    (void) gettimeofday ((struct timeval *) reply->sendt,
				 (struct timezone *) 0);

	    if (ether_write (efd, &outpkt) < 0)
	    {
		perror ("ether_read");
		exit (1);
	    }

	    nsent++;
	    count--;
	}

	FD_ZERO (&fdset);
	FD_SET (efd, &fdset);

	while (tosend == 0)
	{
	    ether_packet pkt;
	    ctp_packet pktbuf;
	    struct timeval tv;

	    pkt.pktbuf = (char *) &pktbuf;
	    pkt.pktlen = sizeof (pktbuf);

	    if (select (efd + 1, param (&fdset), param (0), param (0),
			(struct timeval *) 0) < 0)
	    {
		if (errno == EINTR)
		    break;		/* from while loop */

		perror ("select");
		exit (1);
	    }

	    for (;;)			/* while packets can be read */
	    {
		if (ether_read (efd, &pkt) < 0)
		{
		    if (errno == EAGAIN)
			break;		/* from for loop */

		    perror ("ether_read");
		    exit (1);
		}

		/*
		 * Use 3.x NIT timestamp if present, since latency is terrible
		 */

		if (ether_timestamp)
		    tv = *ether_timestamp;
		else
		    (void) gettimeofday (&tv, (struct timezone *) 0);

		input_packet (&pkt, &pktbuf, &tv);
	    }
	}
    }

    printstats ();
    exit (0);
}

ctp_reply *
output_packet (efd, pkt, ctp, path, pathcount)
int efd;
ether_packet *pkt;
ctp_packet *ctp;
ether_addr path[];
int pathcount;
{
    ctp_forward *forward;
    ctp_reply *reply;
    int i;

    ctp->skip = 0;

    /*
     * Since ctp is long aligned, and the data offset is 2, we are safe
     */

    forward = (ctp_forward *) ctp->data;
    for (i = 1; i < pathcount; i++)
    {
	forward->function = swaps (CTP_FWD);
	forward->addr = path[i];
	forward++;
    }

    /* ensure packet gets back -- append forward to ourselves if needed */

    if (!noreturn && ether_cmp (&path[pathcount - 1], &localaddr)
	&& (ether_cmp (&path[pathcount - 1], &ether_bcast_addr)
	    || !ether_bcast_self (efd) || pathcount == 1
	    || ether_cmp (&path[pathcount - 2], &localaddr))
	&& (nomulticast || ether_cmp (&path[pathcount - 1], &ctpmulticast)
	    || !ether_mcast_self (efd) || pathcount == 1
	    || ether_cmp (&path[pathcount - 2], &localaddr)))
    {
	forward->function = swaps (CTP_FWD);
	forward->addr = localaddr;
	forward++;
    }

    reply = (ctp_reply *) forward;

    i = ((char *) (reply + 1)) - (char *) ctp;

    if (i > packlen)
	if (i > ETHER_MAX)
	{
	    (void) fputs (": ", stdout);
	    (void) fflush (stdout);
	    (void) fprintf (stderr, "Packet too large without data (%d)\n", i);
	    exit (1);
	}
	else
	    packlen = i;

    (void) printf (": %d data bytes (%d total)\n", packlen - i, packlen + 14);
    (void) fflush (stdout);

    reply->function = swaps (CTP_REP);	/* create reply data */
    reply->pid = pid;

    pkt->dest = path[0];		/* fill in ether packet header */
    pkt->pktbuf = (char *) ctp;
    pkt->pktlen = packlen;

    return (reply);			/* return pointer to reply struct */
}


sigtype
resend ()
{
    tosend++;
}

sigtype
die ()
{
    puts ("");
    printstats ();
    exit (0);
}

void
printstats ()
{
    int bucket = 0;
    int lastbucket = -1;
    int min = ((unsigned) -1) >> 1;
    int max = -1;
    int nrecvd = 0;
    int sum = 0;

    (void) printf ("----CTP Statistics----\n");
    (void) printf ("%d packet%s transmitted", nsent,
		   (nsent == 1 ? "" : "s"));

    for (;;)
    {
	while (stats[bucket].valid == 0 && bucket < NBUCKETS)
	    bucket++;

	if (lastbucket != -1
	    && (bucket != NBUCKETS || stats[lastbucket].nrecvd != nrecvd))
	{
	    char hostaddr[ETHERSTRLEN];
	    char host[MAXHOSTNAMELEN];

	    (void) ether_e2a (&stats[lastbucket].eaddr, hostaddr);

	    (void) printf ("\n  %d received from ", stats[lastbucket].nrecvd);

	    if (ether_e2host (&stats[lastbucket].eaddr, host) == NULL)
		(void) fputs (hostaddr, stdout);
	    else
		(void) printf ("%s (%s)", host, hostaddr);

	    if (nsent - stats[lastbucket].nrecvd < 0)
		(void) puts ("");
	    else
		(void) printf (", %d%% packet loss\n", 100 *
			       (nsent - stats[lastbucket].nrecvd) / nsent);
	    if (stats[lastbucket].nrecvd > 0)
		(void)
		    printf ("  round-trip (ms)  min/avg/max = %d/%d/%d",
			    stats[lastbucket].min,
			    stats[lastbucket].sum / stats[lastbucket].nrecvd,
			    stats[lastbucket].max);
	}

	if (bucket == NBUCKETS)
	    break;			/* done with loop */

	lastbucket = bucket;

	if (min > stats[bucket].min)
	    min = stats[bucket].min;
	if (max < stats[bucket].max)
	    max = stats[bucket].max;

	nrecvd += stats[bucket].nrecvd;
	sum += stats[bucket].sum;

	bucket++;
    }

    if (lastbucket == -1 || stats[lastbucket].nrecvd == nrecvd)
    {
	(void) printf (", %d received", nrecvd);

	if (nsent - nrecvd < 0)
	    (void) puts ("");
	else
	    (void) printf (", %d%% packet loss\n",
			   100 * (nsent - nrecvd) / nsent);
    }
    else
	fputs ("\noverall ", stdout);

    if (nrecvd > 0)
	(void) printf ("round-trip (ms)	 min/avg/max = %d/%d/%d\n",
		       min, sum / nrecvd, max);
}

void
input_packet (pkt, ctp, tv)
ether_packet *pkt;
ctp_packet *ctp;
struct timeval *tv;
{
    short skip;
    ctp_reply *reply;
    int ms;
    int bucket;

    skip = swaps (ctp->skip);		/* fetch skip */

    if (skip & 1)			/* if skip is odd, we are unaligned */
	return;

    /*
     * Since ctp is long aligned, and skip is not odd, this is safe
     */

    reply = (ctp_reply *) &ctp->data[skip];

    if (pkt->pktlen - skip - 2 < 2)	/* no room for function */
	return;

    switch (swaps (reply->function))
    {
      case CTP_REP:			/* reply */
	break;

      case CTP_FWD:			/* XXX - should really process this */
	return;

      default:				/* probably a byteswapped function */
	return;
    }

    if (pkt->pktlen - skip - 2 < sizeof (ctp_reply))
	return;

    if (reply->pid != pid)		/* not our packet */
	return;

    (void) printf ("%d bytes from %s: seq=%d time=", pkt->pktlen + ETHER_PKT,
		   ether_ntoa (&pkt->src), reply->seq);

    /*
     * It's safe to cast reply->sendt, since that will be long aligned
     */

    if ((ms = delta (tv, (struct timeval *) reply->sendt)) < 0)
    {
	(void) printf ("(%d ms)!!!\n", ms);
	ms = 0;				/* time must have jumped back, yuck */
    }
    else
	(void) printf ("%d ms\n", ms);

    (void) fflush (stdout);

    bucket = hash (&pkt->src);

    while (stats[bucket].valid && ether_cmp (&stats[bucket].eaddr, &pkt->src))
	if (++bucket >= NBUCKETS)
	    bucket = 0;

    if (!stats[bucket].valid)		/* initialize new bucket */
    {
	stats[bucket].nrecvd = 0;
	stats[bucket].highseq = 0;
	stats[bucket].outofseq = 0;
	stats[bucket].min = ((unsigned) -1) >> 1;
	stats[bucket].max = -1;
	stats[bucket].sum = 0;
	stats[bucket].valid = 1;
	stats[bucket].eaddr = pkt->src;
    }

    stats[bucket].nrecvd++;

    if (reply->seq != lastseq && reply->seq != lastseq + 1)
	stats[bucket].outofseq++;
    lastseq = reply->seq;
    if (reply->seq > stats[bucket].highseq)
	stats[bucket].highseq = reply->seq;

    if (ms > stats[bucket].max)
	stats[bucket].max = ms;
    if (ms < stats[bucket].min)
	stats[bucket].min = ms;
    stats[bucket].sum += ms;
}

delta (a, b)
struct timeval *a;
struct timeval *b;
{
    long usec;
    long sec;

    usec = a->tv_usec - b->tv_usec;
    sec = a->tv_sec - b->tv_sec;

    if (usec < 0)
    {
	sec--;
	usec += 1000000;
    }

    usec += 500;			/* round to ms */
    sec = sec * 1000 + usec / 1000;	/* get ms */

    return (sec);
}					/* delta */


getoptions (argc, argv)
int argc;
char *argv[];
{
    extern char *optarg;
    extern int optind;
    int opt;
    int errs;

    errs = 0;
    while ((opt = getopt (argc, argv, "c:i:l:ns:")) != EOF)
    {
	switch (opt)
	{
	  case 'c':
	    if ((count = atoi (optarg)) <= 0)
	    {
		errs++;
		(void) fprintf (stderr, "invalid packet count: %s\n", optarg);
	    }
	    break;

	  case 'i':
	    interface = optarg;
	    break;

	  case 'l':
	    if ((packlen = atoi (optarg)) <= 0)
	    {
		errs++;
		(void) fprintf (stderr, "invalid packet length: %s\n", optarg);
	    }
	    else if (packlen < ETHER_MIN)
	    {
		(void) fprintf (stderr,
				"packet length %s increased to minimum %d\n",
				optarg, ETHER_MIN);
		packlen = ETHER_MIN;
	    }
	    else if (packlen > ETHER_MAX)
	    {
		(void) fprintf (stderr,
				"packet length %s decreased to maximum %d\n",
				optarg, ETHER_MAX);
		packlen = ETHER_MAX;
	    }
	    break;

	  case 'n':
	    noreturn ^= 1;
	    break;

	  case 's':
	    slptim = atoi (optarg);
	    break;

	  default:
	    errs++;
	}
    }

    if (errs)
	return (-1);
    else
	return (optind);
}


short
swaps (word)
unsigned short word;
{
    static int swap = -1;

    if (swap < 0)
    {
	short test = 0x0100;
	swap = *(char *) &test;
    }

    if (swap)
	return (word >> 8 | (word & 0xff) << 8);
    else
	return (word);
}
