/* $Id: nit4open.c,v 2.1 89/10/23 15:42:59 dupuy Exp $ */

#include <sys/file.h>			/* O_RDWR */
#include <sys/stropts.h>		/* RMSGD */
#include <sys/types.h>			/* NIOCBIND (u_long) */
#include <sys/time.h>			/* NIOCBIND (timeval) */
#include <sys/socket.h>			/* ifreq (sockaddr) */

#include <net/if.h>			/* ifreq */
#include <net/nit_if.h>			/* NIOCBIND */
#include <net/nit_pf.h>			/* NIOCSETF */
#include <net/packetfilt.h>		/* packetfilt */

#include <netinet/in.h>			/* htons */

#include <strings.h>			/* strncpy */
#include <errno.h>			/* EMFILE/EINVAL */

#include "libether.h"

#ifndef NIT_DEV
#define NIT_DEV "/dev/nit"
#endif

static ether_addr promiscuous;

unsigned _ether_types[FD_SETSIZE];

#define ether_type (_ether_types[fd])

static int
nioctl (fd, cmd, ptr)
int fd;
int cmd;
char *ptr;
{
    int saved_errno;

    if (ioctl (fd, cmd, ptr) < 0)
    {
	saved_errno = errno;
#ifdef DEBUG
	switch (cmd)
	{
	  case I_SRDOPT:
	    perror ("ether_open: ioctl (I_SRDOPT)");
	    break;
	  case I_PUSH:
	    perror ("ether_open: ioctl (I_PUSH)");
	    break;
	  case NIOCSETF:
	    perror ("ether_open: ioctl (NIOCSETF)");
	    break;
	  case NIOCBIND:
	    perror ("ether_open: ioctl (NIOCBIND)");
	    break;
	  case NIOCSFLAGS:
	    perror ("ether_open: ioctl (NIOCSFLAGS)");
	    break;
	  default:
	    perror ("ether_open: ioctl (????)");
	    break;
	}
#endif
	(void) close (fd);
	errno = saved_errno;
	return (-1);
    }

    return (0);
}


/*
 * Returns file descriptor for ethernet device by name ("ie0", "le0", etc.).
 * If name is NULL, uses primary ethernet interface.  Will only receive
 * packets of type specified.  Will receive packets for the ethernet address
 * specified, or local ethernet address if NULL.  If there is an error,
 * returns (-1) and the appropriate value is left in errno.  Normal return
 * status zero. Requires superuser privilege.
 */

int
ether_open (name, type, address)
char *name;
unsigned type;
ether_addr *address;
{
    int fd;
    struct ifreq ifr;
    struct packetfilt filter;
    unsigned short *fptr = 0;
    char **interfaces;
    register int i;

    if (name == 0)			/* get default ethernet interface */
    {
	interfaces = ether_interfaces ();
	if (interfaces == 0 || *interfaces == 0)
	    return (-1);

	name = *interfaces;		/* just use the first name in list */
    }

    if ((fd = open (NIT_DEV, O_RDWR)) < 0)
    {
#ifdef DEBUG
	perror (NIT_DEV);
#endif
	return (-1);
    }

    if (fd >= FD_SETSIZE)		/* not worth making it work here */
    {
	(void) close (fd);
	errno = EMFILE;
	return (-1);
    }

    /* arrange to get discrete messages from the stream */

    if (nioctl (fd, I_SRDOPT, (char *) RMSGD) < 0)
	return (-1);

    /* set up filter */

    if (type != ETHER_ALLTYPES)		/* hack for NIT features */
    {
	if (type > ETHER_MAXTYPE)
	{
	    (void) close (fd);
	    errno = EINVAL;
	    return (-1);
	}

	if (nioctl (fd, I_PUSH, "pf") < 0)
	    return (-1);

	fptr = &filter.Pf_Filter[0];

	*fptr++ = ENF_PUSHWORD + ETHER_TYPE / sizeof (short);
	*fptr++ = ENF_PUSHLIT | ENF_EQ;
	*fptr++ = htons ((u_short) type);

	filter.Pf_FilterLen = fptr - &filter.Pf_Filter[0];
	filter.Pf_Priority = 1;		/* unimportant, so long as < 2 */

	if (nioctl (fd, NIOCSETF, (char *) &filter) < 0)
	    return (-1);
    }

    /*
     * We defer the bind until after we've pushed the filter to prevent our
     * being flooded with extraneous packets
     */

    (void) strncpy (ifr.ifr_name, name, sizeof (ifr.ifr_name));

    if (nioctl (fd, NIOCBIND, (char *) &ifr) < 0)
	return (-1);

    if (address != 0)
    {
	if (ether_cmp (address, &promiscuous))
	{
#ifndef MULTICAST
	    ether_addr local_addr;
#endif
	    if (!ETHER_MCAST (address))
	    {
#ifdef DEBUG
		(void) printf ("rejecting non-multicast address argument\n");
#endif
		(void) close (fd);
		errno = EINVAL;
		return (-1);
	    }

#ifdef MULTICAST
	    /* #error not clear how to set multicast addresses */
	    exit (-1);			/* die die die */
#else

	    /*
	     * Enable address matching filter before we go into promiscuous
	     * mode.  We have to do this after the bind, since we can't get
	     * the interface address until then.
	     */

	    (void) ether_address (fd, &local_addr);

	    /* may not have pushed packet filtering module yet */

	    if (fptr == 0)
		if (nioctl (fd, I_PUSH, "pf") < 0)
		    return (-1);

	    fptr = &filter.Pf_Filter[0];

	    if (type != ETHER_ALLTYPES)	/* hack for NIT features */
	    {
		*fptr++ = ENF_PUSHWORD + ETHER_TYPE / sizeof (short);
		*fptr++ = ENF_PUSHLIT;
		*fptr++ = htons ((u_short) type);
		*fptr++ = ENF_CAND;
	    }

	    for (i = 0; i < 3; i++)	/* compare addresses in 3 shorts */
	    {
		*fptr++ = ENF_PUSHWORD + ETHER_DST / sizeof (short) + i;
		*fptr++ = ENF_PUSHLIT | ENF_EQ;
		*fptr++ = address->shorts[i];
		*fptr++ = ENF_PUSHWORD + ETHER_DST / sizeof (short) + i;
		*fptr++ = ENF_PUSHLIT | ENF_EQ;
		*fptr++ = ether_bcast_addr.shorts[i];
		*fptr++ = ENF_OR;
		*fptr++ = ENF_PUSHWORD + ETHER_DST / sizeof (short) + i;
		*fptr++ = ENF_PUSHLIT | ENF_EQ;
		*fptr++ = local_addr.shorts[i];
		*fptr++ = ENF_CNOR;
	    }

	    filter.Pf_FilterLen = fptr - &filter.Pf_Filter[0];
	    filter.Pf_Priority = 1;	/* unimportant, so long as < 2 */

	    if (nioctl (fd, NIOCSETF, (char *) &filter) < 0)
		return (-1);

#endif					/* MULTICAST */

	}

#ifdef MULTICAST
	else				/* only promiscuous if requested */
#endif

	{				/* go into promiscuous mode */
	    long flag;

	    flag = NI_PROMISC;

	    if (nioctl (fd, NIOCSFLAGS, (char *) &flag) < 0)
		return (-1);
	}
    }

    ether_type = type;
    return (fd);
}
