/* $Id: etherip2e.c,v 2.1 89/10/24 17:53:25 dupuy Exp $ */

#include <sys/types.h>			/* in_addr (u_short) */
#include <sys/ioctl.h>			/* ioctl */
#include <sys/errno.h>			/* ENXIO */
#include <sys/time.h>			/* timeval */
#include <sys/socket.h>			/* sockaddr */
#ifdef _SOCKET_				/* 4.0 system? */
#include <sys/sockio.h>			/* SIOCGCONF */
#include <net/if_arp.h>			/* arpreq */
#endif
#include <net/if.h>			/* ifreq */
#include <netinet/in.h>			/* in_addr */

#ifndef FD_SETSIZE
#define fd_set int
#else
#if defined (ultrix) && defined (lint)
#define fd_set int
#endif
#endif

#ifndef IPPORT_DISCARD
#define IPPORT_DISCARD 9		/* not worth looking up in services */
#endif

#include "libether.h"

/*
 * derived from ethertools/iptoea.c
 *
 * Copyright (c) 1988 Philip L. Budne and The Trustees of Boston University
 * All Rights Reserved
 *
 * Permission is granted to any individual or institution to use, copy, or
 * redistribute this software so long as it is not sold for profit, provided
 * that this notice and the original copyright notices are retained.  Boston
 * University makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 */

extern int errno;

ether_addr *
ether_ip2e (ipaddr, addr)
struct in_addr *ipaddr;
ether_addr *addr;
{
    int sockfd;
    char cbuf[MAXNUMIF * sizeof (struct ifreq)];
    struct sockaddr_in *sin;
    struct arpreq arp;
    struct timeval tv;
    struct ifconf ifc;
    int allocated = 0;
    int numinter;
    int saved_errno;
    int i;

    if (addr == 0)
    {
	addr = (ether_addr *) malloc (sizeof (ether_addr));
	allocated++;
    }

    if (addr == 0)
	return (0);

    if ((sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
    {
	saved_errno = errno;
#ifdef DEBUG
	perror ("ether_ip2e: socket");
#endif
	goto failed2open;
    }

    arp.arp_ha.sa_family = AF_UNSPEC;
    arp.arp_pa.sa_family = AF_INET;
    sin = (struct sockaddr_in *) &arp.arp_pa;
    bcopy ((char *) ipaddr, (char *) &sin->sin_addr, sizeof (*ipaddr));
    sin->sin_port = htons (IPPORT_DISCARD);

    /*
     * First, check the arp table once.
     */

    if (!ioctl (sockfd, SIOCGARP, (char *) &arp) && arp.arp_flags & ATF_COM)
    {
	bcopy (arp.arp_ha.sa_data, (char *) addr, sizeof (*addr));
	(void) close (sockfd);
	return (addr);
    }

    /*
     * Then force a kernel ARP resolution by sending a datagram to the IP
     * host. The MSG_DONTROUTE flag prevents the kernel from using a route to
     * another IP network (which would not be reachable at the ethernet
     * level).	We use the discard service to avoid complications.
     */

    if (sendto (sockfd, (char *) &sockfd, 1, MSG_DONTROUTE,
		(struct sockaddr *) sin, sizeof (*sin)) < 0)
    {
	saved_errno = errno;

#ifdef DEBUG
	perror ("ether_ip2e: sendto");
#endif
	goto failed;
    }

    /*
     * Now loop for a short while (.5 - 1.0 sec) looking for the ARP entry.
     */

    tv.tv_sec = 0;
    tv.tv_usec = 100000;		/* .1 second */

    for (i = 0; i < 10; i++)
    {
	if (ioctl (sockfd, SIOCGARP, (char *) &arp) < 0)
	{
	    if (errno != ENXIO)		/* unexpected error */
	    {
		saved_errno = errno;
#ifdef DEBUG
		perror ("ether_ip2e: ioctl (SIOCGARP)");
#endif
		goto failed;
	    }

	    /*
	     * Penalize for no entry yet (wait longer if incomplete found).
	     */

	    i++;
	}
	else if (arp.arp_flags & ATF_COM)
	{
	    bcopy (arp.arp_ha.sa_data, (char *) addr, sizeof (*addr));
	    (void) close (sockfd);
	    return (addr);
	}

	(void) select (0, (fd_set *) 0, (fd_set *) 0, (fd_set *) 0, &tv);
    }

    /*
     * As a last resort, since arp tables don't keep entries for the local
     * interfaces, we scan the local interfaces to check for a match.
     */

    ifc.ifc_len = sizeof (cbuf);
    ifc.ifc_buf = cbuf;
    if (ioctl (sockfd, SIOCGIFCONF, (char *) &ifc) < 0)
    {
#ifdef DEBUG
	perror ("libether SIOCGIFCONF:");
#endif
	goto failed;
    }

    (void) close (sockfd);

    numinter = ifc.ifc_len / sizeof (struct ifreq);
    for (i = 0; i < numinter; i++)
    {
	struct sockaddr_in *inaddr;

	if (ifc.ifc_req[i].ifr_addr.sa_family == AF_INET)
	{
	    inaddr = (struct sockaddr_in *) &ifc.ifc_req[i].ifr_addr;
	    if (ipaddr->s_addr == inaddr->sin_addr.s_addr)
	    {

		/*
		 * The IP address given corresponds to a local interface; get
		 * the ethernet address for that interface if possible;
		 */

		if (ether_intfaddr (ifc.ifc_req[i].ifr_name, addr) == 0)
		{
		    saved_errno = errno;
		    goto failed2open;
		}

		return (addr);		/* got the address - we're done */
	    }
	}
    }

    saved_errno = EHOSTDOWN;
    goto failed2open;

  failed:

    (void) close (sockfd);

  failed2open:

    if (allocated)
	free ((char *) addr);

    errno = saved_errno;
    return (0);
}
