/*
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
**
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
**
** Authors:  Allan Bricker and Michael J. Litzkow,
**           University of Wisconsin, Computer Sciences Dept.
**
*/


#include <stdio.h>
#include <sys/signal.h>
#include <sys/types.h>
#include <sys/param.h>

#include <sys/file.h>
#include <netinet/in.h>

#if defined(ultrix) || defined(ibm032)
#include <sys/dir.h>
#endif defined(ultrix) || defined(ibm032)

#include <sys/user.h>

#if defined(sun)
#include <sys/core.h>

static struct core C;
#if defined(SUNOS40)
int StackOffset;
#endif defined(SUNOS40)
#endif defined(sun)

#include "condor_sys.h"
#include "trace.h"
#include "ckpt.h"
#include "machdep.h"
#include "except.h"
#include "debug.h"
#include "clib.h"

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

#define DEBUG

static int CkptFD;
static int ObjFD;
static int CoreFD;

#ifdef CM_JOB
static int JobOffset, JobLen;
#endif CM_JOB

struct segsizes {
	int		ss_text;		/* Size of text segment */
	int		ss_data;		/* Size of data segment */
	int		ss_stck;		/* Size of stck segment */
};

#define TO_NET_ORDER(i) (int) htonl( (u_long)i ) 
#define TO_HOST_ORDER(i) (int) ntohl( (u_long)i ) 

_updateckpt( ckpt, obj, core )
char *ckpt, *obj, *core;
{
	register int i;
	struct exec x;
	struct segsizes s;
	CKPTMAP ckmap, coremap;
	int scm;

	/*
	**	Object and core files are definitely local.
	**	The new checkpoint file may be local or remote
	**	depending upon whether this is CONDOR or the check-
	**	pointing library.  We'll initialize the system
	**	call mode appropriately...
	*/
#ifdef REMOTE	/* Previous versions created the checkpoint file remotely,
				   not used currently -- instead we ckpt periodically on
				   the worker machine and transfer checkpoint file later */
	(void) SetSyscalls( SYS_REMOTE | SYS_UNRECORDED );
#else REMOTE
	(void) SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );
#endif REMOTE

	if( core == NULL ) {
		_mkckpt( ckpt, obj );
		return;
	}

#if defined(ibm032)
	EXCEPT("_updateckpt([%s], [%s], [%s]): not implemented for IBM032\n",
			ckpt, obj, core );
#endif defined(ibm032)

	/*
	**	Make sure to do local calls...
	*/
	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	ObjFD = open( obj, O_RDONLY, 0 );
	dprintf(D_CKPT, "_updateckpt: ObjFD = %d\n", ObjFD);
	if( ObjFD < 0 ) {
		EXCEPT(obj);
	}

	CoreFD = open( core, O_RDONLY, 0 );
	dprintf(D_CKPT, "_updateckpt: CoreFD = %d\n", CoreFD);
	if( CoreFD < 0 ) {
		EXCEPT(core);
	}

	/*
	**	Go back to the appropriate system call mode...
	*/
	(void) SetSyscalls( scm );

	CkptFD = open( ckpt, O_CREAT|O_TRUNC|O_WRONLY, 0777);
	dprintf(D_CKPT, "_updateckpt: CkptFD = %d\n", CkptFD);
	if( CkptFD < 0 ) {
		EXCEPT( "open( %s, CREAT|O_TRUNC|O_WRONLY ) -- euid = %d, egid = %d",
					ckpt, geteuid(), getegid() );
	}

	/*
	**	Make sure to do local calls...
	*/
	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	readexec( ObjFD, &x );
	readsegsizes( CoreFD, &s );
#ifdef CM_JOB
	readjobinfo();
#endif CM_JOB

	/*
	**	Go back to the appropriate system call mode...
	*/
	(void) SetSyscalls( scm );

	mkckptmap( &ckmap, &x, &s );
#ifdef DEBUG
	dumpsegmap( &ckmap, "ckpt map" );
#endif DEBUG

	mkcoremap( &coremap, &x, &s );
#ifdef DEBUG
	dumpsegmap( &coremap, "core map" );
#endif DEBUG

#ifdef CM_JOB
	for( i = CM_TEXT; i <= CM_JOB; i++ ) {
		copyseg( &coremap.cm_segs[i], &ckmap.cm_segs[i], i );
	}
#else CM_JOB
	for( i = CM_TEXT; i < NCKPTSEGS; i++ ) {
		copyseg( &coremap.cm_segs[i], &ckmap.cm_segs[i], i );
	}
#endif CM_JOB

	mkexec( &ckmap, &x );

	copyexec( CkptFD, &x );
#ifdef CM_JOB
	copyckptmap( CkptFD, &ckmap );
#else CM_JOB
	copystacksize( CkptFD, &ckmap );
#endif CM_JOB

	(void)close( CkptFD );

	/*
	**	Make sure to do local calls...
	*/
	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	(void)close( ObjFD );
	(void)close( CoreFD );

	/*
	**	Go back to the appropriate system call mode...
	*/
	(void) SetSyscalls( scm );
}


static
readexec( objfd, xp )
int objfd;
struct exec *xp;
{
	if( lseek(objfd, 0, L_SET) < 0 ) {
		EXCEPT("lseek");
	}

	if( read(objfd, (char *) xp, sizeof(*xp)) != sizeof(*xp) ) {
		EXCEPT("read");
	}
}

static
readsegsizes( corefd, sp )
int corefd;
struct segsizes *sp;
{
	struct user u;

	if( lseek(corefd, 0, L_SET) < 0 ) {
		EXCEPT("lseek(corefd, 0, L_SET)");
	}

#if defined(sun)
	if( read(corefd, (char *) &C, sizeof(C)) != sizeof(C) ) {
		EXCEPT("read struct core");
	}

	sp->ss_text = C.c_tsize;
	sp->ss_data = C.c_dsize;
#ifdef SUNOS40
	/*
	**	SunOS 4.0 sets c_ssize to be the limit.  We
	**	need to know how big the stack actually was.
	**	The stack pointer is stored in r_sp (r_o6).
	**
	**	What we really need to do here is to read through
	**	the core file until we find a non-zero page of
	**	stack.  The problem with using the hardware sp
	**	is that the program may have moved the stack pointer
	**	into the data area (ala mod2).  If that happens,
	**	we'd be hosed...
	*/

#define ReadCoreForStack
#ifdef ReadCoreForStack
	if( lseek(corefd, C.c_len+C.c_dsize, L_SET) < 0 ) {
		EXCEPT("lseek(corefd %d, %d, L_SET)",
			corefd, C.c_len+C.c_dsize);
	}

	StackOffset = 0;
	for(;;) {
		char buf[ PAGSIZ ];

		if( read(corefd, buf, sizeof(buf)) != sizeof(buf) ) {
			EXCEPT("read(corefd %d, buf 0x%x, %d)", corefd, buf, sizeof(buf));
		}

		if( ! is_zero(buf, sizeof(buf)) ) {
			break;
		}

		StackOffset += sizeof(buf);
	}

	sp->ss_stck = C.c_ssize - StackOffset;

	dprintf(D_CKPT, "Actual StackOffset = %d, ss_stck = %d\n",
		StackOffset, sp->ss_stck);
#else ReadCoreForStack
	sp->ss_stck = USRSTACK - C.c_regs.r_sp;
	StackOffset = C.c_ssize - sp->ss_stck;

	dprintf(D_CKPT, "reported stack size = %d, calculated ss = %d\n",
			C.c_ssize, sp->ss_stck);
	dprintf(D_CKPT, "StackOffset = %d\n", StackOffset );

#endif ReadCoreForStack
#else SUNOS40
	sp->ss_stck = C.c_ssize;
#endif SUNOS40
#else defined(sun)
	if( read(corefd, (char *) &u, sizeof(u)) != sizeof(u) ) {
		EXCEPT("read");
	}

	sp->ss_text = ctob(u.u_tsize);
	sp->ss_data = ctob(u.u_dsize);
	sp->ss_stck = ctob(u.u_ssize);
#endif defined(sun)

	dprintf(D_CKPT, "readsegsizes: ss_text %d, data %d, stck %d\n",
		sp->ss_text, sp->ss_data, sp->ss_stck);
}

/*
**	mkckptmap
**
**	Makes the segment map for the resulting checkpoint file.
*/
static
mkckptmap( ckmap, xp, sp )
CKPTMAP *ckmap;
struct exec *xp;
struct segsizes *sp;
{
	register int pos = N_TXTOFF(*xp);

	ckmap->cm_segs[CM_TEXT].cs_fd  = CkptFD;
	ckmap->cm_segs[CM_TEXT].cs_pos = pos;
#if defined(sequent)
	/*
	**	Text size is the size of text in memory, not
	**	in the a.out.  The first memory page is empty,
	**	but included in the text size.  This page is
	**	not in the a.out.
	*/
	ckmap->cm_segs[CM_TEXT].cs_len = sp->ss_text - ctob(1);
#else ! defined(sequent)
	ckmap->cm_segs[CM_TEXT].cs_len = sp->ss_text;
#endif defined(sequent)

	pos += ckmap->cm_segs[CM_TEXT].cs_len;

	ckmap->cm_segs[CM_DATA].cs_fd  = CkptFD;
	ckmap->cm_segs[CM_DATA].cs_pos = pos;
#if defined(sequent)
	ckmap->cm_segs[CM_DATA].cs_len = sp->ss_data - sp->ss_text;
#else ! defined(sequent)
	ckmap->cm_segs[CM_DATA].cs_len = sp->ss_data;
#endif defined(sequent)

	pos += ckmap->cm_segs[CM_DATA].cs_len;

	ckmap->cm_segs[CM_TRELOC].cs_fd  = CkptFD;
	ckmap->cm_segs[CM_TRELOC].cs_pos = pos;
	ckmap->cm_segs[CM_TRELOC].cs_len = xp->a_trsize;
	pos += ckmap->cm_segs[CM_TRELOC].cs_len;

	ckmap->cm_segs[CM_DRELOC].cs_fd  = CkptFD;
	ckmap->cm_segs[CM_DRELOC].cs_pos = pos;
	ckmap->cm_segs[CM_DRELOC].cs_len = xp->a_drsize;
	pos += ckmap->cm_segs[CM_DRELOC].cs_len;

	ckmap->cm_segs[CM_SYMS].cs_fd  = CkptFD;
	ckmap->cm_segs[CM_SYMS].cs_pos = pos;
	ckmap->cm_segs[CM_SYMS].cs_len = xp->a_syms;
	pos += ckmap->cm_segs[CM_SYMS].cs_len;

	if( xp->a_syms ) {
		ckmap->cm_segs[CM_STRTAB].cs_fd  = CkptFD;
		ckmap->cm_segs[CM_STRTAB].cs_pos = pos;
		ckmap->cm_segs[CM_STRTAB].cs_len = GetStrTabLen(xp, ObjFD);
	} else {
		ckmap->cm_segs[CM_STRTAB].cs_fd  = CkptFD;
		ckmap->cm_segs[CM_STRTAB].cs_pos = 0;
		ckmap->cm_segs[CM_STRTAB].cs_len = 0;
	}
	pos += ckmap->cm_segs[CM_STRTAB].cs_len;

	ckmap->cm_segs[CM_STACK].cs_fd  = CkptFD;
	ckmap->cm_segs[CM_STACK].cs_pos = pos;
	ckmap->cm_segs[CM_STACK].cs_len = sp->ss_stck;
	pos += ckmap->cm_segs[CM_STACK].cs_len;

#ifdef CM_JOB
	ckmap->cm_segs[CM_JOB].cs_fd  = CkptFD;
	ckmap->cm_segs[CM_JOB].cs_pos = pos;
	ckmap->cm_segs[CM_JOB].cs_len = JobLen;
	pos += ckmap->cm_segs[CM_JOB].cs_len;

	ckmap->cm_segs[CM_SEGMAP].cs_fd  = CkptFD;
	ckmap->cm_segs[CM_SEGMAP].cs_pos = pos;
	ckmap->cm_segs[CM_SEGMAP].cs_len = sizeof(CKPTMAP);
#endif CM_JOB
}

/*
**	Make a segment map.  The text will come from the object
**	file and the data and stack areas will come from the core file.
*/
static
mkcoremap( coremap, xp, sp )
CKPTMAP *coremap;
struct exec *xp;
struct segsizes *sp;
{
	coremap->cm_segs[CM_TEXT].cs_fd  = ObjFD;
	coremap->cm_segs[CM_TEXT].cs_pos = N_TXTOFF(*xp);
	coremap->cm_segs[CM_TEXT].cs_len = sp->ss_text;

	coremap->cm_segs[CM_DATA].cs_fd  = CoreFD;
#if defined(sun)
	coremap->cm_segs[CM_DATA].cs_pos = C.c_len;
#endif defined(sun)

#if defined(vax) | defined(sequent) | defined(bobcat)
	coremap->cm_segs[CM_DATA].cs_pos = ctob(UPAGES);
#endif defined(vax) | defined(sequent) | defined(bobcat)

#if defined(sequent)
	coremap->cm_segs[CM_DATA].cs_len = sp->ss_data - sp->ss_text;
#endif defined(sequent)

#if defined(vax) | defined(sun) | defined(bobcat)
	coremap->cm_segs[CM_DATA].cs_len = sp->ss_data;
#endif defined(vax) | defined(sun) | defined(bobcat)

	coremap->cm_segs[CM_STACK].cs_fd  = CoreFD;
	coremap->cm_segs[CM_STACK].cs_pos =   coremap->cm_segs[CM_DATA].cs_pos
									 	+ coremap->cm_segs[CM_DATA].cs_len;
#if defined(SUNOS40)
	coremap->cm_segs[CM_STACK].cs_pos += StackOffset;
#endif defined(SUNOS40)

	coremap->cm_segs[CM_STACK].cs_len = sp->ss_stck;

	coremap->cm_segs[CM_TRELOC].cs_fd  = ObjFD;
	coremap->cm_segs[CM_TRELOC].cs_pos = N_TROFF(*xp);
	coremap->cm_segs[CM_TRELOC].cs_len = xp->a_trsize;

	coremap->cm_segs[CM_DRELOC].cs_fd  = ObjFD;
	coremap->cm_segs[CM_DRELOC].cs_pos = N_DROFF(*xp);
	coremap->cm_segs[CM_DRELOC].cs_len = xp->a_drsize;

	coremap->cm_segs[CM_SYMS].cs_fd  = ObjFD;
	coremap->cm_segs[CM_SYMS].cs_pos = N_SYMOFF(*xp);
	coremap->cm_segs[CM_SYMS].cs_len = xp->a_syms;

	if( xp->a_syms ) {
		coremap->cm_segs[CM_STRTAB].cs_fd  = ObjFD;
		coremap->cm_segs[CM_STRTAB].cs_pos = N_STROFF(*xp);
		coremap->cm_segs[CM_STRTAB].cs_len = GetStrTabLen(xp, ObjFD);
	} else {
		coremap->cm_segs[CM_STRTAB].cs_fd  = ObjFD;
		coremap->cm_segs[CM_STRTAB].cs_pos = 0;
		coremap->cm_segs[CM_STRTAB].cs_len = 0;
	}

#ifdef CM_JOB
	coremap->cm_segs[CM_JOB].cs_fd  = ObjFD;
	coremap->cm_segs[CM_JOB].cs_pos = JobOffset;
	coremap->cm_segs[CM_JOB].cs_len = JobLen;

	coremap->cm_segs[CM_SEGMAP].cs_fd  = -1;
	coremap->cm_segs[CM_SEGMAP].cs_pos = -1;
	coremap->cm_segs[CM_SEGMAP].cs_len = sizeof(CKPTMAP);
#endif CM_JOB
}

GetStrTabLen( xp, fd )
register struct exec *xp;
register int fd;
{
	register int scm;
	int strtablen;

	/*
	**	Make sure to do local calls...
	*/
	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	dprintf(D_CKPT, "GetStrTabLen: N_STROFF(*xp) = %d, sizeof(int) = %d\n", N_STROFF(*xp), sizeof(int));

	if( lseek(fd, (off_t)N_STROFF(*xp), L_SET) < 0 ) {
		EXCEPT("lseek to string table");
	}

	if( read(fd, (char *) &strtablen, sizeof(int)) != sizeof(int) ) {
		EXCEPT("read string table length");
	}

	/*
	**	Go back to the appropriate system call mode...
	*/
	(void) SetSyscalls( scm );

	dprintf(D_CKPT, "GetStrTabLen: returning %d\n", strtablen);

	return( strtablen );
}

#ifdef CM_JOB
static
readjobinfo()
{
	register int scm;
	CKPTMAP cm;

	/*
	**	Make sure to do local calls...
	*/
	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	if( lseek(ObjFD, -sizeof(CKPTMAP), L_XTND) < 0 ) {
		EXCEPT("lseek(%d, -sizeof(CKPTMAP), L_XTND)", ObjFD);
	}

	if( read(ObjFD, (char *) &cm, sizeof(CKPTMAP)) != sizeof(CKPTMAP) ) {
		EXCEPT("read %d", ObjFD);
	}

	/*
	**	Go back to the appropriate system call mode...
	*/
	(void) SetSyscalls( scm );

	JobOffset = TO_HOST_ORDER(cm.cm_segs[CM_JOB].cs_pos);
	JobLen    = TO_HOST_ORDER(cm.cm_segs[CM_JOB].cs_len);
}
#endif CM_JOB

/*
**	Modify the exec structure to reflect changes
**	in the size of data, etc.
*/
static
mkexec( ckmap, xp )
CKPTMAP *ckmap;
struct exec *xp;
{
	xp->a_data   = ckmap->cm_segs[CM_DATA].cs_len;
	xp->a_bss    = 0;	/* bss is included in data */
}

static
copyseg( inseg, outseg, segno )
register CKPTSEG *inseg, *outseg;
int segno;
{
	register int scm;
	register int len, size;
	int rval;
	char buf[ 8 * 1024 ];

#ifdef DEBUG
	dprintf(D_CKPT,"copyseg: segno %d\n", segno);
	dprintf(D_CKPT,"         inseg  <pos %d, len %d, fd %d>\n",
		inseg->cs_pos, inseg->cs_len, inseg->cs_fd);
	dprintf(D_CKPT,"         outseg <pos %d, len %d, fd %d>\n",
		outseg->cs_pos, outseg->cs_len, outseg->cs_fd);
#endif DEBUG

	/*
	**	Make sure to do local calls...
	*/
	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	if( lseek(inseg->cs_fd, inseg->cs_pos, L_SET) < 0 ) {
		EXCEPT("lseek(%d, %d, L_SET)", inseg->cs_fd, inseg->cs_pos);
	}

	/*
	**	Go back to the appropriate system call mode...
	*/
	(void) SetSyscalls( scm );

	if( lseek(outseg->cs_fd, outseg->cs_pos, L_SET) < 0 ) {
		EXCEPT("lseek(%d, %d, L_SET)", outseg->cs_fd, outseg->cs_pos);
	}

	if( inseg->cs_len > sizeof(buf) ) {
		size = sizeof(buf);
	} else {
		size = inseg->cs_len;
	}

	for( len = 0; len < inseg->cs_len; ) {
		/*
		**	Make sure to do local calls...
		*/
		scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

		if( (len + size) > inseg->cs_len ) {
			size = inseg->cs_len - len;
		}

		errno = 0;
		rval = read(inseg->cs_fd, buf, size);
		if( rval != size ) {
			EXCEPT("read fd %d, segno %d, size %d returned %d",
						inseg->cs_fd, segno, size, rval);
		}

		/*
		**	Go back to the appropriate system call mode...
		*/
		(void) SetSyscalls( scm );

		if( write(outseg->cs_fd, buf, size) != size ) {
			EXCEPT("write outfd %d", outseg->cs_fd);
		}
		
		len += size;
	}
}

#ifdef CM_JOB
static
copyckptmap( ckptfd, ckmap )
int ckptfd;
CKPTMAP *ckmap;
{
	register int i;

	if( lseek(ckptfd, ckmap->cm_segs[CM_SEGMAP].cs_pos, L_SET) < 0 ) {
		EXCEPT("lseek");
	}

#ifdef DEBUG
	dumpsegmap( ckmap, "Final Checkpoint Map" );
#endif DEBUG

	/*
	**	Before writing out the checkpoint segment map,
	**	convert to network order.
	*/
	for( i = 0; i < NCKPTSEGS; i++ ) {
		ckmap->cm_segs[i].cs_fd  = TO_NET_ORDER(0);
		ckmap->cm_segs[i].cs_pos = TO_NET_ORDER(ckmap->cm_segs[i].cs_pos);
		ckmap->cm_segs[i].cs_len = TO_NET_ORDER(ckmap->cm_segs[i].cs_len);
	}

	if( write(ckptfd, (char *) ckmap, sizeof(CKPTMAP)) != sizeof(CKPTMAP) ) {
		EXCEPT("write");
	}
}
#else CM_JOB
static
copystacksize( ckptfd, ckmap )
int ckptfd;
CKPTMAP *ckmap;
{
	int ssize = htonl(ckmap->cm_segs[CM_STACK].cs_len);

	if( lseek(ckptfd, 0, L_XTND) < 0 ) {
		EXCEPT("lseek(%d, 0, L_XTND)", ckptfd);
	}

	if( write(ckptfd, (char *) &ssize, sizeof(ssize)) != sizeof(ssize) ) {
		EXCEPT("write stacksize");
	}
}
#endif CM_JOB

static
copyexec( ckptfd, xp )
int ckptfd;
struct exec *xp;
{
#ifdef DEBUG
	dprintf(D_CKPT, "Exec structure: a_text  0x%x\n", xp->a_text);
	dprintf(D_CKPT, "                a_data  0x%x\n", xp->a_data);
	dprintf(D_CKPT, "                a_bss   0x%x\n", xp->a_bss );
	dprintf(D_CKPT, "                a_syms  0x%x (%d)\n",
			xp->a_syms, xp->a_syms);
#endif DEBUG
	if( lseek(ckptfd, 0, L_SET) < 0 ) {
		EXCEPT("lseek");
	}

	if( write(ckptfd, (char *) xp, sizeof(*xp)) != sizeof(*xp) ) {
		EXCEPT("write");
	}
}

static
initjob( jobp )
JOB *jobp;
{
	bzero( (char *) jobp, sizeof(JOB) );

	jobp->j_version = TO_NET_ORDER(CKPT_VERS);
}

#ifdef SUNOS40
is_zero(buf, size)
char *buf;
register int size;
{
	register int *ibuf = (int *) buf;

	size /= sizeof(int);

	while( size-- > 0 ) {
		if( *ibuf++ != 0 ) {
			return( 0 );
		}
	}

	return( 1 );
}
#endif SUNOS40

#ifdef DEBUG
static char *segnames[] = {
	"TEXT:   ",
	"DATA:   ",
	"T_RELOC:",
	"D_RELOC:",
	"SYMS:   ",
	"STRTAB: ",
	"STACK:  ",
	"JOB:    ",
	"SEGMAP: "
};

static
dumpsegmap( segmap, name )
CKPTMAP *segmap;
char *name;
{
	int i;

	dprintf(D_CKPT, "\t\t%s\n", name);

	for( i = 0; i < NCKPTSEGS; i++ ) {
		dumpseg( &segmap->cm_segs[i], segnames[i] );
	}
}

static
dumpseg( seg, name )
CKPTSEG *seg;
char *name;
{
	dprintf(D_CKPT, "Segment %s fd %2d, pos %6d (0x%x),\n",
				name, seg->cs_fd, seg->cs_pos, seg->cs_pos);
	dprintf(D_CKPT, "                        len %6d (0x%x)\n",
				seg->cs_len, seg->cs_len );
}
#endif DEBUG

