/* 
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <ctype.h>
#include <pwd.h>
#include <errno.h>
#include <a.out.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/resource.h>
#include "debug.h"
#include "except.h"
#include "com.h"
#include "proc.h"
#include "config.h"
#include "trace.h"
#include "sched.h"
#include "expr.h"
#include "clib.h"

#ifdef NDBM
#include <ndbm.h>
#else NDBM
#include "ndbm_fake.h"
#endif NDBM

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

extern int	(*_EXCEPT_Cleanup)();	/* Cleanup function used by EXCEPT */
extern TABLE	ComTab[];

char	*OperatingSystem;
char	*Architecture;
char	*Spool;

EXPR	*scan();
char	*strdup(), *malloc(), *index(), *get_owner(),
		*check_requirements(), *param(), *condor_param();
XDR		*xdr_Init();
int		LineNo;
int		GotQueueCommand;

DBM		*Q, *OpenJobQueue();
PROC	Proc;

char	*IckptName;	/* Pathname of spooled initial ckpt file */

#define UPPER(x) (((x) >= 'a' && (x) <= 'z') ? toupper(x) : (x))

char	*MyName;
int		Quiet;

#define PROCVARSIZE	31
BUCKET *ProcVars[ PROCVARSIZE ];

/*
**	Names of possible CONDOR variables.
*/
char	*Cluster 		= "cluster";
char	*Process   		= "process";
char	*Priority		= "priority";
char	*Notification	= "notification";
char	*Executable		= "executable";
char	*Arguments		= "arguments";
char	*Environment	= "environment";
char	*Input			= "input";
char	*Output			= "output";
char	*Error			= "error";
char	*RootDir		= "rootdir";
char	*InitialDir		= "initialdir";
char	*Requirements	= "requirements";
char	*Preferences	= "preferences";

extern int	Terse;
extern int	DontDisplayTime;

main( argc, argv)
int		argc;
char	*argv[];
{
	FILE	*fp;
	int		DoCleanup();
	char	queue_name[MAXPATHLEN];
	char	**ptr;
	char	*cmd_file = NULL;

	MyName = argv[0];

	config( MyName, (CONTEXT *)0 );
	init_params();


	if( argc < 2 || argc > 3 ) {
		usage();
	}

	DebugFlags |= D_EXPR;
	Terse = 1;
	DontDisplayTime = TRUE;

	for( ptr=argv+1; *ptr; ptr++ ) {
		if( ptr[0][0] == '-' ) {
			if( ptr[0][1] == 'q' ) {
				Quiet++;
			} else {
				usage();
			}
		} else {
			cmd_file = *ptr;
		}
	}

	if( cmd_file == NULL ) {
		usage();
	}

	if( (fp=fopen(cmd_file,"r")) == NULL ) {
		EXCEPT( "fopen(%s,\"r\")", argv[1] );
	}

		/* Open job queue and initialize a new cluster */
	(void)sprintf( queue_name, "%s/job_queue", Spool );
	if( (Q=OpenJobQueue(queue_name,O_RDWR,0)) == NULL ) {
		EXCEPT( "OpenJobQueue(%s)", queue_name );
	}
	LockJobQueue( Q, WRITER );
	Proc.q_date = (int)time( (time_t *)0 );
	Proc.owner = get_owner();
	Proc.id.cluster = CreateCluster( Q );
	Proc.id.proc = 0;

	_EXCEPT_Cleanup = DoCleanup;

		/* Parse the file and queue the jobs */
	if( read_condor_file(fp) < 0 ) {
		EXCEPT( "CONDOR description file error" );
	}
	if( Proc.id.proc != 0 ) {
		reschedule();
	}
	if( !GotQueueCommand ) {
		fprintf( stderr, "\"%s\" doesn't contain any \"queue\"", cmd_file );
		fprintf( stderr, " commands -- no jobs queued\n" );
	}
}

/*
** Send the reschedule command to the local schedd to get the jobs running
** as soon as possible.
*/
reschedule()
{
	int			sock = -1;
	int			cmd;
	XDR			xdr, *xdrs = NULL;
	char		this_host[512];

	if( gethostname(this_host,sizeof this_host) < 0 ) {
		EXCEPT( "gethostname(0x%x,%d)", this_host, sizeof this_host );
	}

		/* Connect to the schedd */
	if( (sock = do_connect(this_host, "condor_schedd", SCHED_PORT)) < 0 ) {
		dprintf(D_ALWAYS, "Can't connect to condor_schedd on %s\n", this_host);
		exit( 1 );
	}
	xdrs = xdr_Init( &sock, &xdr );
	xdrs->x_op = XDR_ENCODE;

	cmd = RESCHEDULE;
	ASSERT( xdr_int(xdrs, &cmd) );
	ASSERT( xdrrec_endofrecord(xdrs,TRUE) );
}


SetExecutable()
{
	char	ickpt_name[MAXPATHLEN];
	char	*ename;
	static int exec_set = 0;

	if( exec_set ) {
		return;
	}

	ename = condor_param(Executable);

	if( ename == NULL ) {
		EXCEPT("No '%s' parameter was provided", Executable);
	}

	if( whitespace(ename) ) {
		EXCEPT("The '%s' takes exactly one argument (%s)",
				Executable, ename);
	}

	(void)sprintf( ickpt_name, "%s/job%06d.ickpt", Spool, Proc.id.cluster );
	_mkckpt(ickpt_name,ename);

	Proc.cmd = ename;

	exec_set = 1;
}

SetInput()
{
	char *in = condor_param(Input);

	if( in == NULL ) {
		Proc.in = "/dev/null";
	} else {
		if( whitespace(in) ) {
			EXCEPT("The '%s' takes exactly one argument (%s)",
					Input, in);
		}
		Proc.in = in;
	}
	check_open( Proc.in, O_RDONLY );
}

SetOutput()
{
	char *out = condor_param(Output);

	if( out == NULL ) {
		Proc.out = "/dev/null";
	} else {
		if( whitespace(out) ) {
			EXCEPT("The '%s' takes exactly one argument (%s)",
					Output, out);
		}
		Proc.out = out;
	}
	check_open( Proc.out, O_WRONLY|O_CREAT|O_TRUNC );
}

SetError()
{
	char *err = condor_param(Error);

	if( err == NULL ) {
		Proc.err = "/dev/null";
	} else {
		if( whitespace(err) ) {
			EXCEPT("The '%s' takes exactly one argument (%s)",
					Error, err);
		}
		Proc.err = err;
	}
	check_open( Proc.err, O_WRONLY|O_CREAT|O_TRUNC );
}

SetPriority()
{
	char *prio = condor_param(Priority);

	if( prio == NULL ) {
		Proc.prio = 0;
	} else {
		Proc.prio = atoi( prio );
		if( Proc.prio < -20 || Proc.prio > 20 ) {
			EXCEPT("Priority must be in the range -20 thru 20 (%d)",
					Proc.prio );
		}
	}
}

SetNotification()
{
	char *how = condor_param(Notification);

	if( (how == NULL) || (stricmp(how, "COMPLETE") == 0) ) {
		Proc.notification = NOTIFY_COMPLETE;
	} else if( stricmp(how, "NEVER") == 0 ) {
		Proc.notification = NOTIFY_NEVER;
	} else if( stricmp(how, "ALWAYS") == 0 ) {
		Proc.notification = NOTIFY_ALWAYS;
	} else if( stricmp(how, "ERROR") == 0 ) {
		Proc.notification = NOTIFY_ERROR;
	} else {
		EXCEPT("Notification must be 'Never', 'Always', 'Complete', or 'Error'");
	}
}

SetArguments()
{
	char *args = condor_param(Arguments);

	if( args == NULL ) {
		Proc.args = "";
	} else {
		Proc.args = args;
	}
}

SetEnvironment()
{
	char *env = condor_param(Environment);

	if( env == NULL ) {
		Proc.env = "";
	} else {
		Proc.env = env;
	}
}

SetRootDir()
{
	char *rootdir = condor_param(RootDir);

	if( rootdir == NULL ) {
		Proc.rootdir = "/";
	} else {
		if( access(rootdir, F_OK|X_OK) < 0 ) {
			EXCEPT("No such directory: %s", rootdir);
		}
		Proc.rootdir = rootdir;
	}
}

SetRequirements()
{
	char *requirements = condor_param(Requirements);

	if( requirements == NULL ) {
		Proc.requirements = "";
	} else {
		Proc.requirements = requirements;
	}

	Proc.requirements = check_requirements( Proc.requirements );
}

SetPreferences()
{
	char *preferences = condor_param(Preferences);

	if( preferences == NULL ) {
		Proc.preferences = "";
	} else {
		check_expr_syntax(  preferences );
		Proc.preferences = preferences;
	}
}

SetIWD()
{
	char	*shortname;
	char	iwd[ MAXPATHLEN ];
	char	cwd[ MAXPATHLEN ];

	shortname = condor_param( InitialDir );

	if( strcmp(Proc.rootdir,"/") != MATCH )	{	/* Rootdir specified */
		if( shortname ) {
			(void)strcpy( iwd, shortname );
		} else {
			(void)strcpy( iwd, "/" );
		}
	} else {
		if( shortname  ) {
			if( shortname[0] == '/' ) {
				(void)strcpy( iwd, shortname );
			} else {
				(void)getwd( cwd );
				(void)sprintf( iwd, "%s/%s", cwd, shortname );
			}
		} else {
			(void)getwd( iwd );
		}
	}

	compress( iwd );
		
	check_iwd( iwd );

	Proc.iwd = strdup( iwd );
}

check_iwd( iwd )
char	*iwd;
{
	char	pathname[ MAXPATHLEN ];

	(void)sprintf( pathname, "%s/%s", Proc.rootdir, iwd );
	compress( pathname );

	if( access(pathname, F_OK|X_OK) < 0 ) {
		EXCEPT("No such directory: %s", pathname);
	}
}

read_condor_file( fp )
FILE *fp;
{
	char	*name, *value, *getline(), *expand_macro();
	char	*ptr;

	LineNo = 0;

	for(;;) {
		name = getline(fp);
		if( name == NULL ) {
			break;
		}

			/* Skip over comments */
		if( *name == '#' || blankline(name) )
			continue;
		
		if( stricmp(name, "queue") == 0 ) {
			queue();
			continue;
		}

#define isop(c)		((c) == '=')

			/* Separate out the parameter name */
/*
		for( ptr=name ; *ptr && !isspace(*ptr) && !isop(*ptr); ptr++ );
*/

		ptr = name;
		while( *ptr ) {
			if( isspace(*ptr) || isop(*ptr) ) {
				break;
			} else {
				ptr++;
			}
		}

		if( !*ptr ) {
			(void)fclose( fp );
			return( -1 );
		}

		if( isop(*ptr) ) {
			*ptr++ = '\0';
		} else {
			*ptr++ = '\0';
			while( *ptr && !isop(*ptr) ) {
				ptr++;
			}

			if( !*ptr ) {
				(void)fclose( fp );
				return( -1 );
			}
			ptr += 1;

		}

			/* Skip to next non-space character */
		while( *ptr && isspace(*ptr) ) {
			ptr++;
		}

		value = ptr;

			/* Expand references to other parameters */
		name = expand_macro( name, ProcVars, PROCVARSIZE );
		if( name == NULL ) {
			(void)fclose( fp );
			return( -1 );
		}

		lower_case( name );

			/* Put the value in the Configuration Table */
		insert( name, value, ProcVars, PROCVARSIZE );

		free( name );
		free( value );
	}

	(void)fclose( fp );
	return 0;
}

char *
condor_param( name )
char *name;
{
	extern char *lookup_macro(), *expand_macro();
	char *pval = lookup_macro(name, ProcVars, PROCVARSIZE);

	if( pval == NULL ) {
		return( NULL );
	}

	pval = expand_macro(pval, ProcVars, PROCVARSIZE);

	return( pval );
}

set_condor_param( name, value )
char *name, *value;
{
	char *tval = strdup( value );
	insert( name, tval, ProcVars, PROCVARSIZE );
}


queue()
{
	char tmp[ BUFSIZ ];

	/*
	**	Insert the current idea of the cluster and
	**	process number into the hash table.
	*/
	GotQueueCommand = 1;
	(void)sprintf(tmp, "%d", Proc.id.cluster);
	set_condor_param(Cluster, tmp);
	(void)sprintf(tmp, "%d", Proc.id.proc);
	set_condor_param(Process, tmp);

	SetExecutable();
	SetRootDir();
	SetIWD();
	SetPriority();
	SetArguments();
	SetEnvironment();
	SetNotification();
	SetRequirements();
	SetPreferences();
	SetInput();
	SetOutput();
	SetError();

	if( StoreProc(Q,&Proc) < 0 ) {
		EXCEPT( "StoreProc failed" );
	}
	if( !Quiet ) {
		display_proc_long( &Proc );
	}
	Proc.id.proc += 1;
}

char *
check_requirements( orig )
char	*orig;
{
	int		has_opsys = FALSE;
	int		has_arch = FALSE;
	char	*ptr;
	static char	answer[2048];


	for( ptr = orig; *ptr; ptr++ ) {
		if( strincmp("Arch",ptr,4) == MATCH ) {
			has_arch = TRUE;
			break;
		}
	}

	for( ptr = orig; *ptr; ptr++ ) {
		if( strincmp("OpSys",ptr,5) == MATCH ) {
			has_opsys = TRUE;
			break;
		}
	}

	(void)strcpy( answer, orig );
	if( !has_arch ) {
		if( answer[0] ) {
			(void)strcat( answer, " && (Arch == \"" );
		} else {
			(void)strcpy( answer, "(Arch == \"" );
		}
		(void)strcat( answer, Architecture );
		(void)strcat( answer, "\")" );
	}

	if( !has_opsys ) {
		(void)strcat( answer, " && (OpSys == \"" );
		(void)strcat( answer, OperatingSystem );
		(void)strcat( answer, "\")" );
	}

	if( !has_opsys && !has_arch ) {
		magic_check();
	}

	check_expr_syntax(  answer );
	return answer;
}

check_open( name, flags )
char	*name;
int		flags;
{
	int		fd;
	char	pathname[MAXPATHLEN];

	if( name[0] == '/' ) {	/* absolute wrt whatever the root is */
		(void)sprintf( pathname, "%s%s", Proc.rootdir, name );
	} else {	/* relative to iwd which is relative to the root */
		(void)sprintf( pathname, "%s/%s/%s", Proc.rootdir, Proc.iwd, name );
	}
	compress( pathname );

	if( (fd=open(pathname,flags,0664)) < 0 ) {
		EXCEPT( "Can't open \"%s\"  with flags 0%o", pathname, flags );
	}
	(void)close( fd );
}

		


usage()
{
	fprintf( stderr, "Usage: %s [-q] cmdfile\n", MyName );
	exit( 1 );
}

DoCleanup()
{
	TerminateCluster( Q, Proc.id.cluster, SUBMISSION_ERR );
	if( IckptName ) {
		(void)unlink( IckptName );
	}
}


SetSyscalls( foo )
int		foo;
{
	return foo;
}

char *
get_owner()
{
	struct passwd	*pwd, *getpwuid();

	if( (pwd=getpwuid(getuid())) == NULL ) {
		EXCEPT( "Can't get passwd entry for uid %d\n", getuid() );
	}
	return strdup(pwd->pw_name);
}

init_params()
{
	Architecture = param( "ARCH" );
	if( Architecture == NULL ) {
		EXCEPT( "ARCH not specified in config file" );
	}

	OperatingSystem = param( "OPSYS" );
	if( OperatingSystem == NULL ) {
		EXCEPT( "OPSYS not specified in congig file" );
	}

	Spool = param( "SPOOL" );
	if( Spool == NULL ) {
		EXCEPT( "SPOOL not specified in congig file" );
	}

}

whitespace(str)
register char *str;
{
	while( *str ) {
		if( isspace(*str++) ) {
			return( 1 );
		}
	}

	return( 0 );
}

compress( str )
char	*str;
{
	char	*src, *dst;

	src = str;
	dst = str;

	while( *dst ) {
		*dst++ = *src++;
		while( *(src - 1) == '/' && *src == '/' ) {
			src++;
		}
	}
}

#ifdef NOTDEF
char *Notifications[] = {
	"Never", "Always", "Complete", "Error"
};

display_proc_long( proc )
PROC	*proc;
{
	(void)putchar( '\n' );
	printf( "Id: %d.%d\n", proc->id.cluster, proc->id.proc );
	printf( "Owner: %s\n", proc->owner );
	printf( "Queue Date: %s", ctime( (time_t *)&proc->q_date ) );
	printf( "Status: %d\n", proc->status );
	printf( "Priority: %d\n", proc->prio );
	printf( "Notification: %s\n", Notifications[proc->notification]);
	printf( "Cmd: %s\n", proc->cmd );
	printf( "Args: %s\n", proc->args );
	printf( "Env: %s\n", proc->env );
	printf( "In: %s\n", proc->in );
	printf( "Out: %s\n", proc->out );
	printf( "Err: %s\n", proc->err );
	printf( "RootDir: %s\n", proc->rootdir );
	printf( "Initial Working Directory: %s\n", proc->iwd );
	printf( "Requirements: %s\n", proc->requirements );
	printf( "Preferences: %s\n", proc->preferences );
}
#endif NOTDEF

CONTEXT	*MachineContext;

check_expr_syntax( expr )
char	*expr;
{
	CONTEXT	*context, *create_context(), *fake_machine_context();
	char	line[1024];
	ELEM	*dummy, *eval();

	if( MachineContext == NULL ) {
		MachineContext = fake_machine_context();
	}

	(void)sprintf( line, "DUMMY = %s && (Disk >= 0)", expr );
	context = create_context();
	store_stmt( scan(line), context );

	dummy = eval( "DUMMY", context, MachineContext );
	if( !dummy || dummy->type != BOOL ) {
		fprintf( stderr, "Syntax error: \"%s\"\n", expr );
		DoCleanup();
		exit( 1 );
	}

	free_elem( dummy );
	free_context( context );
}

CONTEXT	*
fake_machine_context()
{
	CONTEXT	*answer, *create_context();
	char	line[1024];

	answer = create_context();

	(void)sprintf( line, "Arch = \"fake\"" );
	store_stmt( scan(line), answer );

	(void)sprintf( line, "OpSys = \"fake\"" );
	store_stmt( scan(line), answer );

	(void)sprintf( line, "Disk = 0" );
	store_stmt( scan(line), answer );

	(void)sprintf( line, "Memory = 0" );
	store_stmt( scan(line), answer );

	(void)sprintf( line, "Machine = \"nobody\"" );
	store_stmt( scan(line), answer );

	return answer;
}

#ifdef MIPS
magic_check()
{
	return;
}
#else MIPS
magic_check()
{
	int		fd;
	struct exec	exec;

	if( (fd=open(Proc.cmd,O_RDONLY,0)) < 0 ) {
		EXCEPT( "open(%s,O_RDONLY,0)", Proc.cmd );
	}

	if( read(fd,(char *)&exec,sizeof(struct exec)) != sizeof(struct exec) ) {
		EXCEPT( "read(%d,0x%x,%d)", fd, &exec, sizeof(struct exec) );
	}

#ifdef vax
	if( exec.a_magic != ZMAGIC ) {
		fprintf( stderr, "\"%s\": BAD MAGIC NUMBER\n", Proc.cmd );
		DoCleanup();
		exit( 1 );
	}
#endif

#ifdef sequent
	if( exec.a_magic != ZMAGIC ) {
		fprintf( stderr, "\"%s\": BAD MAGIC NUMBER\n", Proc.cmd );
		DoCleanup();
		exit( 1 );
	}
#endif sequent

#ifdef sparc
	if( exec.a_magic != ZMAGIC ) {
		fprintf( stderr, "\"%s\": BAD MAGIC NUMBER\n", Proc.cmd );
		DoCleanup();
		exit( 1 );
	}
	if( exec.a_machtype != M_SPARC ) {
		fprintf( stderr, "\"%s\": NOT COMPILED FOR SPARC ARCHITECTURE\n",
															Proc.cmd );
		DoCleanup();
		exit( 1 );
	}
	if( exec.a_dynamic ) {
		fprintf( stderr, "\"%s\": LINKED FOR DYNAMIC LOADING\n", Proc.cmd );
		DoCleanup();
		exit( 1 );
	}
#endif
		
#ifdef mc68020
	if( exec.a_magic != ZMAGIC ) {
		fprintf( stderr, "\"%s\": BAD MAGIC NUMBER\n", Proc.cmd );
		DoCleanup();
		exit( 1 );
	}
	if( exec.a_machtype != M_68020 ) {
		fprintf( stderr, "\"%s\": NOT COMPILED FOR MC68020 ARCHITECTURE\n",
																Proc.cmd );
		DoCleanup();
		exit( 1 );
	}
#endif
		

	(void)close( fd );
}
#endif MIPS
