/* 
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/signal.h>

#define debug
#define log printf

#define MINUTES	* 60
#define HOURS	* (60 MINUTES)
#define DAYS	* (24 HOURS)

#define CKPT_INTRVL	(1 HOURS)

int CkptInterval = CKPT_INTRVL;
struct itimerval Timer;

int CkptPid = -1;
char *CkptName;
char *CkptTmpName;
char *CoreName;

char *StdIn = NULL;
char *StdOut = NULL;
char *StdErr = NULL;
char *LogFile = NULL;

int StopRunning = 0;
char *MyName;

extern errno;

main( argc, argv )
int argc;
char **argv;
{
	register char *arg;
	int Ckpt(), Quit();

	MyName = *argv;

	while( argc--, *++argv ) {
		arg = *argv;
		if( *arg != '-' ) {
			break;
		}
		switch( *++arg ) {
#ifdef debug
		case 's':
			CkptInterval = atoi(++arg);
			break;
#endif debug
		case 'm':
			CkptInterval = atoi(++arg) MINUTES;
			break;
		case 'h':
			CkptInterval = atoi(++arg) HOURS;
			break;
		case 'd':
			CkptInterval = atoi(++arg) DAYS;
			break;
		
		case 'L':	/* Specify log file */
			LogFile = ++arg;
			break;
		case 'I':	/* Specify stdin    */
			StdIn = ++arg;
			break;
		case 'O':	/* Specify stdout   */
			StdOut = ++arg;
			break;
		case 'E':	/* Specify stderr   */
			StdErr = ++arg;
			break;
		default:
			Usage( MyName );
		}
	}

	if( (argc < 1) || (CkptInterval <= 0) ) {
		Usage( MyName );
	}

#ifdef NOTDEF
	if( LogFile != NULL ) {
		initlog( LogFile );
	}
#endif NOTDEF

	if( StdIn != NULL ) {
		if( freopen(StdIn, "r", stdin) == NULL ) {
			perror(StdIn);
			exit( 1 );
		}
	}

	if( StdOut != NULL ) {
		if( freopen(StdOut, "a", stdout) == NULL ) {
			perror(StdOut);
			exit( 1 );
		}
		setlinebuf(stdout);
	}

	if( StdErr != NULL ) {
		if( freopen(StdErr, "a", stderr) == NULL ) {
			perror(StdErr);
			exit( 1 );
		}
		setbuf(stderr, (char *)0);
	}

	(void)signal( SIGALRM, Ckpt );
	(void)signal( SIGTSTP, Quit );

	if( strcmp(MyName, "rerun") == 0 ) {
		rerun( argv );
	} else {
		run( argv );
	}

	Timer.it_interval.tv_sec  = CkptInterval;
	Timer.it_interval.tv_usec = 0;
	Timer.it_value.tv_sec     = CkptInterval;
	Timer.it_value.tv_usec    = 0;
	(void)setitimer(ITIMER_REAL, &Timer, (struct itimerval *)0);

	WaitForChild();
}

Usage( name )
char *name;
{
	fprintf(stderr, "Usage: %s [Time Specifier] [Output Specifier] prog args\n",
			name);
	fprintf(stderr, "   Time Specifiers:   -mMinutes -hHours -dDays\n");
	fprintf(stderr, "   Output Specifiers: -IStdIn -OStdOut -EStdErr -LLog\n");
	
	exit( 1 );
}

/*
**	Cause a checkpoint to occur.
*/
Ckpt()
{
	log("Killing CkptPid %d\n", CkptPid);
	(void)kill( CkptPid, SIGTSTP );
}

Quit()
{
	StopRunning = 1;
	log("Killing CkptPid %d, then StopRunning\n", CkptPid);
	(void)kill( CkptPid, SIGTSTP );
}

WaitForChild()
{
	union wait status;
	struct rusage rusage;
	int pid;

	for(;;) {
		pid = wait3( &status, 0, &rusage );

		if( pid < 0 ) {
			log("wait returned %d, errno = %d\n", pid, errno);
			perror("wait");
			exit( 1 );
		}

		if( pid != CkptPid ) {
			log("wait returned %d (not %d)\n", pid, CkptPid);
			exit( 1 );
		}

		log("pid %d returned by wait.  CkptName is '%s'\n", pid, CkptName);

		if( status.w_termsig == 0 ) {
			fprintf(stderr, "%s: exited status %d\n",
					CkptName, status.w_retcode);
			exit( (int)status.w_retcode );
		}

		if( status.w_termsig != SIGILL ) {
			fprintf(stderr, "%s: exited due to signal %d%s\n",
				CkptName, status.w_termsig,
				(status.w_coredump ? ", core dump produced" : "") );
			exit( 4 );
		}

		_updateckpt( CkptTmpName, CkptName, CoreName );

		(void)rename( CkptTmpName, CkptName );
		(void)unlink( CoreName );

		if( StopRunning ) {
			log("Request to StopRunning\n");
			exit( 0 );
		}

		Start( CkptName, (char **)NULL );
	}
}

Start( name, argv )
char *name, **argv;
{
	CkptPid = fork();
	if( CkptPid < 0 ) {
		perror("fork");
		exit( 2 );
	}

	if( CkptPid == 0 ) {
		/*
		**	Make sure that the child starts up with TSTP blocked
		*/
		(void)sigsetmask( sigmask(SIGTSTP) );

		if( argv == 0 ) {
			(void)execl(name, name, 0);
		} else {
			(void)execv(name, argv);
		}
		perror(name);
		exit( 3 );
	}

	log("Started process %d\n", CkptPid);
}

run( argv )
char **argv;
{
	char *strdup();
	char buf[ 1024 ];

	(void)sprintf(buf, "%s.ckpt", *argv);
	CkptName = strdup( buf );

	(void)sprintf(buf, "%s.tmp", CkptName);
	CkptTmpName = strdup( buf );

	CoreName = "core";

	_mkckpt( CkptName, *argv );

	*argv = CkptName;
	Start( CkptName, argv );
}

rerun( argv )
char **argv;
{
	char *strdup();
	char buf[ 1024 ];

	(void)sprintf(buf, "%s.ckpt", *argv);
	CkptName = strdup( buf );

	(void)sprintf(buf, "%s.tmp", CkptName);
	CkptTmpName = strdup( buf );

	CoreName = "core";

	*argv = CkptName;
	Start( CkptName, (char **)NULL );
}
