/* 
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <netinet/in.h>
#include <setjmp.h>

#include "condor_sys.h"
#include "trace.h"
#include "ckpt.h"
#include "machdep.h"
#include "except.h"
#include "debug.h"

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

#define DEBUG

#define STACKSIZ	(32*1024)

char	 TmpStack[ STACKSIZ ];

char	 CWD[ MAXPATHLEN ];
char	 CkptName[ MAXPATHLEN ];

RESTREC	 RestartInfo;

int CkptInProgress;
int SyscallInProgress;
int CkptWanted;

#ifdef VAX
#include <frame.h>

#define _CJBLEN	4
typedef long condor_jmp_buf[_CJBLEN];

#define MGIC	0
#define APTR	1
#define FPTR	2
#define PCTR	3

#define MAGIC	0xfeedface

#define FP_OFFSET	24

condor_setjmp( cenv )
u_long *cenv;
{
	struct frame *fp = (struct frame *) ((int) &cenv - FP_OFFSET);

	dprintf(D_CKPT, "setjmp: fp = 0x%x, &cenv = 0x%x, diff = %d\n",
			fp, &cenv, (int) fp - (int) &cenv );

	dprintf(D_CKPT, "setjmp: sp 0x%x, savap 0x%x, savfp 0x%x, savpc 0x%x\n",
		fp, fp->fr_savap, fp->fr_savfp, fp->fr_savpc );
	
	cenv[MGIC] = MAGIC;
	cenv[APTR] = fp->fr_savap;
	cenv[FPTR] = fp->fr_savfp;
	cenv[PCTR] = fp->fr_savpc;

	return( 0 );
}

condor_longjmp( cenv )
u_long *cenv;
{

	struct frame *fp = (struct frame *) ((int) &cenv - FP_OFFSET);

	if( cenv[MGIC] != MAGIC ) {
		EXCEPT("Condor long jump botch");
	}

	fp->fr_savap = cenv[APTR];
	fp->fr_savfp = cenv[FPTR];
	fp->fr_savpc = cenv[PCTR];

	return( 1 );
}
#else VAX
#define condor_jmp_buf			jmp_buf
#define condor_setjmp(cenv)		setjmp(cenv)
#define condor_longjmp(cenv)	longjmp((cenv), 1)
#endif VAX

condor_jmp_buf LongJumpBuf;

ckpt()
{
	int scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );
	int pid = getpid();

	dprintf(D_CKPT,
		"CKPT: CkptWanted %d, CkptInProgress %d, SyscallInProgress %d\n",
			CkptWanted, CkptInProgress, SyscallInProgress);

	dprintf(D_CKPT, "ckpt: about to kill(%d, SIGTSTP)\n", pid);

	kill( pid, SIGTSTP );

	(void) SetSyscalls( scm );
}

#if defined(ULTRIX) || defined(SUNOS40)
#define sighandler_t	void
#else defined(ULTRIX) || defined(SUNOS40)
#define sighandler_t	int
#endif defined(ULTRIX) || defined(SUNOS40)

sighandler_t
CKPT( sig, code, scp )
int					sig, code;
struct sigcontext	*scp;
{

	extern int CurrentSysCall;


	dprintf(D_CKPT,
		"CKPT: CkptWanted %d, CkptInProgress %d, SyscallInProgress %d\n",
			CkptWanted, CkptInProgress, SyscallInProgress);

	if( CkptInProgress || SyscallInProgress ) {
		if( SyscallInProgress ) {
			dprintf(D_ALWAYS, "CKPT: [%s] in progress, checkpoint wanted\n",
					CONDOR_SYSCALLNAME(CurrentSysCall));
		}
		CkptWanted = 1;
		return;
	}

	do_CKPT();
}

static int	CkptSCM;

do_CKPT()
{
#ifdef VAX
	/* Note: these registers are not saved by the signal handling
	   mechanism.  We force them to be saved in our call frame instead. */
	register int r11, r10, r9, r8, r7, r6;
#endif VAX
	int scm, rval;

	CkptInProgress = 1;
	CkptWanted = 0;

	/*
	**	Save the current offset of all open files.
	*/
	SetFileInfo();

	CkptSCM = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	dprintf(D_CKPT, "do_CKPT: CkptSCM = 0x%x\n", CkptSCM);
	rval = condor_setjmp(LongJumpBuf);

	if( rval != 0 ) {
		dprintf(D_CKPT, "do_CKPT: condor_setjmp returned %d, CkptSCM 0x%x\n",
				rval, CkptSCM);
		(void) SetSyscalls( CkptSCM );
		return;
	} else {
		dprintf(D_CKPT,"do_CKPT: condor_setjmp returned zero, CkptSCM 0x%x\n",
				CkptSCM);
	}

	dump_core();
}

dump_core()
{
	(void) SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	kill( getpid(), SIGQUIT );
}

struct sigvec	SigVec, oUSR2Vec;
struct sigstack	SigStack, oSigStack;
int				SigMask;


restart()
{

	sighandler_t restart2();


	SigStack.ss_sp = &TmpStack[ STACKSIZ ];
	SigStack.ss_onstack = 0;

	SigVec.sv_handler = restart2;
	SigVec.sv_mask    = 0;
	SigVec.sv_onstack = 1;

	(void) SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	if( sigstack(&SigStack, &oSigStack) < 0 ) {
		EXCEPT("sigstack(&SigStack [0x%x], &oSigStack [0x%x), errno = %d",
			&SigStack, &oSigStack, errno);
	}

	if( sigvec(SIGUSR2, &SigVec, &oUSR2Vec) < 0 ) {
		EXCEPT("sigvec(SIGUSR2, restart2), errno = %d", errno );
	}

	dprintf(D_CKPT, "restart: about to kill SIGUSR2\n");

	kill( getpid(), SIGUSR2 );
	sigpause( 0 );
}

sighandler_t
restart2()
{
	dprintf(D_CKPT,
		"restart2: CkptWanted %d,CkptInProgress %d, SyscallInProgress %d\n",
			CkptWanted, CkptInProgress, SyscallInProgress);

	/*
	**	Open the checkpoint file to extract any information
	**	that we need (eg. the saved stack).
	*/
	ReadRestartInfo();

	/*
	**	Reopen and reposition all active files.  The open file table
	**	does not need to be read from the checkpoint file because it
	**	resides in memory.  The checkpoint version is there for
	**	initialization and for other programs to examine.
	*/
#ifdef CONDOR
	(void) SetSyscalls( SYS_REMOTE | SYS_UNRECORDED );
#else CONDOR
	(void) SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );
#endif CONDOR

	dprintf( D_CKPT, "CkptSCM = 0x%x\n", CkptSCM );

	ReopenFiles();

	/*
	**	Change to the previous current working directory
	*/
	if( chdir(CWD) < 0 ) {
		EXCEPT(CWD);
	}

	SigVec.sv_handler = CKPT;
	SigVec.sv_mask    = 0;
	SigVec.sv_onstack = 0;

	(void) SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	if( sigvec(SIGTSTP, &SigVec, (struct sigvec *) 0) < 0 ) {
		EXCEPT("sigvec(SIGTSTP, CKPT), errno = %d", errno );
	}

	if( sigvec(SIGUSR2, &oUSR2Vec, (struct sigvec *) 0) < 0 ) {
		EXCEPT("sigvec(SIGUSR2, &oUSR2Vec [0x%x]), errno = %d",
			&oUSR2Vec, errno );
	}

	if( sigstack(&oSigStack, (struct sigstack *) 0) < 0 ) {
		EXCEPT("sigstack(&oSigStack [0x%x], 0), errno = %d",
			&oSigStack, errno );
	}

	sigsetmask( 0 );

	dprintf(D_CKPT, "restart2: about to condor_longjmp!\n");
	CkptInProgress = 0;
	condor_longjmp( LongJumpBuf );
	EXCEPT("restart2: what happened to condor_longjmp????");
}

SetFileInfo()
{
	register FINFO *fi;
	register int i;
	register int flags;
	register int scm;

#ifdef CONDOR
	scm = SetSyscalls( SYS_REMOTE | SYS_UNRECORDED );
#else CONDOR
	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );
#endif CONDOR

	for( i = 0; i < NOFILE; i++ ) {
		fi = &RestartInfo.rr_file[i];
		flags = fi->fi_flags;

		if( ((flags & FI_OPEN) == FI_OPEN) && ((flags & FI_DUP) == 0) ) {
			fi->fi_pos = lseek(fi->fi_fdno,0,L_INCR);
		}
	}

	(void) SetSyscalls( scm );
}

ReadRestartInfo()
{
	register int fd;
	int scm;	/* System call mode (local or remote */
	int ssize;

	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	fd = open(CkptName, O_RDONLY, 0);
	if( fd < 0 ) {
		EXCEPT( "Could not open '%s', errno = %d",
				CkptName, errno );
	}
	dprintf( D_CKPT, "Opened checkpoint file \"%s\"\n", CkptName );

	ssize = ReadStackSize( fd );
	ReadStack( fd, ssize );

	close( fd );

	(void) SetSyscalls( scm );
}

ReadStackSize( fd )
register int fd;
{
	long ssize;
	int	 where;

	if( (where=lseek(fd, -sizeof(ssize), L_XTND)) < 0 ) {
		EXCEPT("Could not lseek to stack size value");
	}
	dprintf(D_CKPT, "Seeked to %d\n", where );

	if( read(fd, (char *) &ssize, sizeof(ssize)) != sizeof(ssize) ) {
		EXCEPT("Could not read stack size");
	}
	dprintf(D_CKPT, "Read %d bytes\n", sizeof(ssize) );

	dprintf(D_CKPT, "ReadStackSize: ssize = %d ntohl(ssize) = %d\n",
		ssize, ntohl(ssize) );

	if( ntohl(ssize) <= 0 ) {
		EXCEPT( "Stack Size Botch (stack size = %d)\n", ntohl(ssize) );
	}

	return( ntohl(ssize) );
}

ReadStack(fd, ssize)
register int fd;
register long ssize;
{
	register char *TStack = (char *) USRSTACK;

	if( lseek(fd, -(ssize+sizeof(ssize)), L_XTND) < 0 ) {
		EXCEPT( "lseek(%d,%d,L_XTND)", fd, -(ssize+sizeof(ssize)) );
		goto error;
	}

	TStack -= ssize;	/* Set TStack pointing at the base of the stack  */

	dprintf(D_CKPT, "ReadStack: USRSTACK = 0x%x, TStack = 0x%x\n",
			USRSTACK, TStack );

	/*
	**	The kernel (probably) won't grow the user stack in the middle
	**	of a system call, so lets grow it now...
	**	Make sure to account for any extra stack the system may give us...
	*/

	*(int *) (TStack+STACKGROW) = 0;

	dprintf(D_CKPT, "ReadStack: Just set *(TStack 0x%x + STACKGROW 0x%x)\n",
					TStack, STACKGROW);

	dprintf(D_CKPT, "ReadStack: About to read(%d, 0x%x, %d)\n",
					fd, TStack, ssize);

	if( read(fd, TStack, ssize) != ssize ) {
		goto error;
	}

	dprintf(D_CKPT, "ReadStack: Just read %d bytes of stack\n", ssize);

	return;

error:
	EXCEPT("Could not read Checkpoint Stack, errno = %d",
				errno );
}


ReopenFiles()
{
	register FINFO *fi;
	register int i, fd;
	register int flags;

	for( i = 0; i < NOFILE; i++ ) {
		fi = &RestartInfo.rr_file[i];
		flags = fi->fi_flags;

		if( (flags & FI_PREOPEN) == FI_PREOPEN ) {
			if( lseek(i, fi->fi_pos, L_SET) < 0 ) {
				EXCEPT("re-lseek(%d, %d, L_SET)", i, fi->fi_pos);
			}
			continue;
		}

		if( (flags & FI_OPEN) == FI_OPEN ) {
			if( (flags & FI_DUP) == FI_DUP ) {
				if( dup2(fi->fi_fdno, i) < 0 ) {
					EXCEPT("dup2");
				}
			} else {
				fd = open(fi->fi_path, fi->fi_priv, 0);
				if( fd < 0 ) {
					EXCEPT("open(%s, 0x%x, 0)", fi->fi_path, fi->fi_priv);
				}

				if( fd != i ) {
					if( dup2(fd, i) < 0 ) {
						EXCEPT("dup2 (again)");
					}
					if( close(fd) < 0 ) {
						EXCEPT("close");
					}
				}

				if( lseek(i, fi->fi_pos, L_SET) < 0 ) {
					EXCEPT("re-lseek");
				}
			}
		}
	}
}
