/* 
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <signal.h>
#include <netdb.h>
#include <pwd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include "condor_types.h"
#include "sched.h"
#include "debug.h"
#include "trace.h"
#include "except.h"
#include "expr.h"
#include "manager.h"
#include "clib.h"

#ifndef LINT
static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */
#endif LINT

CONTEXT		*create_context();
EXPR		*build_expr();
char		*calloc(), *strdup();
MACH_REC	*find_mach_rec(), *create_mach_rec();
bool_t 		xdr_int();


MACH_REC	*MachineList;

STATUS_LINE	*StatusLines[1024];
int			N_StatusLines;
int			MachineUpdateInterval;

negotiator_info( xdrs )
XDR		*xdrs;
{
	PRIO_REC	record, *rec = &record;

	for(;; ) {
		bzero( (char *)rec, sizeof(PRIO_REC) );
		if( !xdr_prio_rec(xdrs,rec) ) {
			dprintf( D_ALWAYS, "Can't read prio_rec from negotiator\n" );
			return;
		}
		if( rec->name == NULL ) {
			return;
		}
		if(  rec->name[0] == '\0' ) {
			free( rec->name );
			return;
		}
		update_prio( rec );
		free( rec->name );
	}
}

update_prio( rec )
PRIO_REC	*rec;
{
	int	i;

	for( i=0; i<N_StatusLines; i++ ) {
		if( strcmp(StatusLines[i]->name,rec->name) == MATCH ) {
			StatusLines[i]->prio = rec->prio;
			return;
		}
	}
	dprintf( D_ALWAYS, "Can't find status line for \"%s\"\n", rec->name );
}

STATUS_LINE	Endmarker;
give_status_lines( xdrs )
XDR	*xdrs;
{
	int		i;

	if( !xdrrec_skiprecord(xdrs) ) {
		dprintf( D_ALWAYS, "Can't send mach_rec to client\n" );
		return;
	}

	xdrs->x_op = XDR_ENCODE;

	for( i=0; i<N_StatusLines; i++ ) {
		if( !xdr_status_line(xdrs,StatusLines[i]) ) {
			dprintf( D_ALWAYS, "Can't send StatusLine to client\n" );
			return;
		}
	}
	if( !xdr_status_line(xdrs,&Endmarker) ) {
		dprintf( D_ALWAYS, "Can't send Endmarker to client\n" );
		return;
	}

	(void)xdrrec_endofrecord( xdrs, TRUE );
}

machine_info( xdrs, from, kind )
XDR		*xdrs;
struct sockaddr_in	*from;
int		kind;
{
	MACH_REC	*rec;

	if( (rec = find_mach_rec(from)) == NULL ) {
		rec = create_mach_rec( from );
		insque( (struct qelem *)rec, (struct qelem *)MachineList );
		dprintf( D_ALWAYS, "Creating new record for \"%s\"\n", rec->name );
	} else {
		/*
		*/
		if( kind == STARTD_INFO ) {
			dprintf( D_ALWAYS, "STARTD update for \"%s\"\n", rec->name );
		} else {
			dprintf( D_ALWAYS, "SCHEDD update for \"%s\"\n", rec->name );
		}
	}

	if( rec->machine_context == NULL ) {
		rec->machine_context = create_context();
	}
	if( !xdr_context(xdrs,rec->machine_context) ) {
		dprintf( D_ALWAYS, "Can't read mahcine context from client\n" );
		free_context( rec->machine_context );
		rec->machine_context = NULL;
		return;
	}
	rec->time_stamp = time( (time_t *)0 );

	if( kind == STARTD_INFO ) {
		rec->busy = FALSE;
	}

	update_status_line( rec );

}

update_status_line( rec )
MACH_REC	*rec;
{
	int			idle;
	STATUS_LINE	*line = rec->line;

	if( line->name == NULL ) {
		line->name = strdup( rec->name );
	}
	(void) evaluate_int( "Running", &line->run, rec->machine_context,
															(CONTEXT *)0 );
	(void) evaluate_int( "Idle", &idle, rec->machine_context, (CONTEXT *)0 );
	line->tot = line->run + idle;
	free( line->state );
	(void) evaluate_string( "State", &line->state, rec->machine_context,
																(CONTEXT *)0 );
	(void) evaluate_float( "LoadAvg", &line->load_avg, rec->machine_context,
																(CONTEXT *)0 );
	(void) evaluate_int( "KeyboardIdle", &line->kbd_idle, rec->machine_context,
																(CONTEXT *)0);
	if( line->arch == NULL ) {
		(void) evaluate_string( "Arch", &line->arch, rec->machine_context,
																(CONTEXT *)0 );
	}
	if( line->op_sys == NULL ) {
		(void) evaluate_string( "OpSys", &line->op_sys, rec->machine_context,
																(CONTEXT *)0 );
	}
}

MACH_REC	*
find_mach_rec( from )
struct sockaddr_in	*from;
{
	MACH_REC	*ptr;

	for( ptr = MachineList->next; ptr->name; ptr = ptr->next ) {
		if( ptr->net_addr.s_addr == from->sin_addr.s_addr ) {
			return ptr;
		}
	}
	return NULL;
}

STATUS_LINE	*
create_status_line()
{
	STATUS_LINE	*line;

	line = (STATUS_LINE *)calloc( 1, sizeof(STATUS_LINE) );
	StatusLines[ N_StatusLines++ ] = line;
	return line;
}

MACH_REC	*
create_mach_rec( from )
struct sockaddr_in	*from;
{
	MACH_REC	*answer;
	ELEM		tmp;
	struct hostent	*hp, *gethostbyaddr();

	answer = (MACH_REC *)calloc( 1, sizeof(MACH_REC) );

	answer->next = answer;
	answer->prev = answer;
	answer->machine_context = create_context();
	if( !from ) {
		/*
		dprintf( D_ALWAYS, "Created NULL machine record\n" );
		*/
		return answer;
	}

	answer->net_addr = from->sin_addr;
	answer->net_addr_type = from->sin_family;
	answer->time_stamp = time( (time_t *)0 );

	if( (hp=gethostbyaddr((char *)&from->sin_addr,sizeof(struct in_addr),
												from->sin_family)) == NULL ) {
		dprintf( D_ALWAYS, "Can't find host name for \"%s\"\n",
											inet_ntoa(from->sin_addr) );



		free_mach_rec( answer );
		return NULL;
	}
	answer->name = strdup( hp->h_name );

	tmp.type = INT;
	tmp.i_val = 0;
	store_stmt( build_expr("Users",&tmp), answer->machine_context );
	store_stmt( build_expr("Running",&tmp), answer->machine_context );
	store_stmt( build_expr("Idle",&tmp), answer->machine_context );
	store_stmt( build_expr("Prio",&tmp), answer->machine_context );

	answer->line = create_status_line();
	return answer;
}

free_mach_rec( rec )
MACH_REC	*rec;
{
	if( rec->name ) {
		free( rec->name );
	}

	if( rec->machine_context ) {
		free_context( rec->machine_context );
	}
	free( (char *)rec );
}

give_status( xdrs )
XDR		*xdrs;
{
	MACH_REC	*ptr;


	if( !xdrrec_skiprecord(xdrs) ) {
		dprintf( D_ALWAYS, "Can't send mach_rec to client\n" );
		return;
	}

	xdrs->x_op = XDR_ENCODE;
	for( ptr=MachineList->next; ptr->name; ptr = ptr->next ) {
		if( (int)time( (time_t *)0 ) - ptr->time_stamp >
												MachineUpdateInterval ) {
				/* Doesn't belong here, but it's the only place we
				   traverse the entire machine list. */
			free( ptr->line->state );
			ptr->line->state = strdup( "(DOWN)" );
		}

		if( !xdr_mach_rec(xdrs,ptr) ) {
			dprintf( D_ALWAYS, "Can't send mach_rec to client\n" );
			return;
		}
	}
	if( !xdr_mach_rec(xdrs,ptr) ) {
		dprintf( D_ALWAYS, "Can't send mach_rec to client\n" );
		return;
	}

	xdrrec_endofrecord( xdrs, TRUE );
}
