/*
 * Copyright (c) 1980 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifndef lint
char copyright[] =
"@(#) Copyright (c) 1980 Regents of the University of California.\n\
 All rights reserved.\n";
#endif /* not lint */

/* comsat - daemon to notify registered users of new mail
   Usage: comsat 2> /tmp/comsat.errors

   Receives one line messages of the form
   user@mailbox-offset\n
   on a named pipe, and if the user is logged on and "biff y" (owner
   execute bit of tty is turned on), prints a message summarizing the new
   mail on the user's screen.

   Converted for System V with FIFO
   by David MacKenzie <edf@rocky2.rockefeller.edu>
   with additional changes
   by Jim Mattson <mattson%cs@ucsd.edu>

   Latest revision: 08/23/89 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <utmp.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <termio.h>
#include <string.h>
#include <varargs.h>
#include <sys/utsname.h>

/* BSD-compatible constants for lseek. */
#define L_SET 0
#define L_INCR 1
#define L_XTND 2

/* The directory where system mailboxes are located. */
#define SYSMAILDIR "/usr/mail"

/* Path of the named pipe used to send messages to this program. */
#define FIFO "/etc/comsat.fifo"

/* The number of seconds between checks of the utmp. */
#define ALARM_INTERVAL 15

/* If this is defined, no blank line will be printed between the
   header lines (e.g., "Subject:") and the head of the message text.
   The blank line is present in the 4.3BSD comsat but absent in the
   4.3BSD-tahoe comsat from which this version is derived. */
/* #define JOIN_HEADER_AND_TEXT */

char *malloc ();
char *realloc ();
off_t atol ();
off_t fseek ();
off_t lseek ();
time_t time ();

char *xmalloc ();
char *xrealloc ();
int read_utmp ();
int note_alarm ();
void mail_for ();
void msg_perror_fatal ();
void notify ();
void summarize_new_mail ();

/* This machine's host name, used in the notification message. */
char hostname[10];

/* Contents of the utmp. */
struct utmp *utmp;

/* If nonzero, the alarm rang while sending a biff message. */
int alarm_rang;

/* Number of entries in `utmp'. */
int nutmp;

/* File descriptor for reading the utmp. */
int utfd;

/* The name this program was run with, for error messages. */
char *program_name;

/* ARGSUSED */
int
main (argc, argv)
     int argc;
     char **argv;
{
  FILE *fifp;
  char msgbuf[100];

  program_name = argv[0];

  signal (SIGINT, SIG_IGN);
  signal (SIGHUP, SIG_IGN);

  /* Don't tie up a filesystem. */
  if (chdir ("/") == -1)
    msg_perror_fatal ("Cannot chdir to /");

  if (daemon_running (FIFO))
    {
      fprintf (stderr, "%s: Daemon is already running\n", program_name);
      exit (1);
    }
  switch (fork ())
    {
    case -1:
      msg_perror_fatal ("Cannot fork");
    case 0:			/* Child. */
      break;
    default:			/* Parent. */
      _exit (0);
    }

  fclose (stdin);		/* Don't need these anymore. */
  fclose (stdout);
  setpgrp ();			/* Detach from parent's process group. */

  fifp = fdopen (open_fifo (FIFO), "r");
  if (fifp == NULL)
    msg_perror_fatal ("Cannot fdopen FIFO %s", FIFO);

  utfd = open (UTMP_FILE, O_RDONLY);
  if (utfd == -1)
    msg_perror_fatal ("Cannot read %s", UTMP_FILE);

  utmp = NULL;
  nutmp = 0;
  gethostname (hostname, sizeof (hostname));
  signal (SIGCLD, SIG_IGN);	/* Prevent zombie process creation. */

  read_utmp ();

  while (1)
    {
      while (fgets (msgbuf, sizeof msgbuf, fifp) == NULL)
	sleep (1);
      if (nutmp == 0)
	continue;		/* No one has logged in yet. */
      /* Don't let automatic utmp updating corrupt the in-core copy while
         we're using it. */
      alarm_rang = 0;
      signal (SIGALRM, note_alarm);
      mail_for (msgbuf);
      /* If we missed a utmp update while biffing the user, do the
         update manually and set the alarm again. */
      if (alarm_rang)
	read_utmp ();
      else
	signal (SIGALRM, read_utmp);
    }
  /* NOTREACHED */
}

/* SIGALRM handler for while mail_for is running. */

int
note_alarm ()
{
  alarm_rang = 1;
}

/* Normal SIGALRM handler.  Every ALARM_INTERVAL seconds, read a current
   copy of the utmp into `utmp'. */

int
read_utmp ()
{
  static unsigned utmp_size = 0;/* Bytes allocated for `utmp'. */
  static unsigned utmp_mtime = 0;	/* Last modification time of utmp. */
  struct stat stats;

  if (fstat (utfd, &stats) == -1)
    msg_perror_fatal ("Cannot fstat utmp");
  if (stats.st_mtime > utmp_mtime)
    {
      utmp_mtime = stats.st_mtime;
      if (stats.st_size > utmp_size)
	{
	  utmp_size = stats.st_size + 10 * sizeof (struct utmp);
	  utmp = (struct utmp *) xrealloc ((char *) utmp, utmp_size);
	}
      if (lseek (utfd, 0L, L_SET) < 0)
	msg_perror_fatal ("Cannot seek to beginning of utmp");
      nutmp = read (utfd, utmp, (unsigned) stats.st_size);
      if (nutmp == -1)
	msg_perror_fatal ("Cannot read utmp");
      nutmp /= sizeof (struct utmp);
    }
  alarm ((unsigned) ALARM_INTERVAL);
  signal (SIGALRM, read_utmp);
}

/* `name' has the form "user@mailbox-offset\n".  Check whether "user" is
   logged on; if so, try to notify them of the new mail. */

void
mail_for (name)
     char *name;
{
  struct utmp *utp;
  char *cp;
  off_t offset;

  cp = strchr (name, '@');
  if (cp == NULL)
    {
      fprintf (stderr, "%s: Invalid message: %s\n", program_name, name);
      return;
    }
  *cp++ = '\0';
  offset = atol (cp);
  utp = &utmp[nutmp];
  while (--utp >= utmp)
    {
      if (!strncmp (utp->ut_user, name, sizeof (utmp[0].ut_user)))
	notify (utp, offset);
    }
}

/* The carriage return character needed for the terminal being notified;
   it will be the null string if the terminal driver or the terminal
   is supplying a carriage return automatically with each newline. */
static char *cr;

/* If the user described in `utp' is logged on and "biff y", notify them
   of the new mail in their system mailbox at offset `offset'. */

void
notify (utp, offset)
     struct utmp *utp;
     off_t offset;
{
  static char tty[20] = "/dev/";
  struct termio termio;
  FILE *tp;
  char name[sizeof (utmp[0].ut_user) + 1];
  struct stat stats;
  int i;

  strncpy (tty + 5, utp->ut_line, sizeof (utp->ut_line));
  while ((i = stat (tty, &stats)) == -1 && errno == EINTR)
    /* Do nothing. */ ;
  if (i == -1 || !(stats.st_mode & S_IEXEC))
    return;
  switch (fork ())
    {
    case -1:
      msg_perror_fatal ("Cannot fork");
    case 0:			/* Child. */
      break;
    default:			/* Parent. */
      return;
    }
  signal (SIGALRM, SIG_DFL);
  alarm ((unsigned) 30);
  tp = fopen (tty, "w");
  if (tp == NULL)
    _exit (1);
  ioctl (fileno (tp), TCGETA, &termio);
  cr = (termio.c_oflag & OPOST) && (termio.c_oflag & ONLCR)
    || (termio.c_oflag & ONLRET) ? "" : "\r";
  strncpy (name, utp->ut_user, sizeof (utp->ut_user));
  name[sizeof (name) - 1] = '\0';
  fprintf (tp, "%s\n\007New mail for %s@%.*s\007 has arrived:%s\n----%s\n",
	   cr, name, sizeof (hostname), hostname, cr, cr);
  summarize_new_mail (tp, name, offset);
  fclose (tp);
  _exit (0);
}

/* Print the first 7 lines or 560 characters (whichever comes first) of
   the new mail message that starts at byte `offset' in the system
   mailbox for user `name' to stream `tp'.  Skip header lines other than
   From, Subject, [To, and Date]. */

void
summarize_new_mail (tp, name, offset)
     FILE *tp;
     char *name;
     off_t offset;
{
  char *cp;
  FILE *fi;
  int linecnt;
  int charcnt;
  int inheader;
  char line[BUFSIZ];

  cp = xmalloc (sizeof (SYSMAILDIR) + strlen (name) + 2);
  sprintf (cp, "%s/%s", SYSMAILDIR, name);
  fi = fopen (cp, "r");
  free (cp);
  if (fi == NULL)
    return;
  if (fseek (fi, offset, L_SET))
    return;
  linecnt = 7;
  charcnt = 560;
  inheader = 1;

  while (fgets (line, sizeof (line), fi) != NULL)
    {
      if (inheader)
	{
	  if (line[0] == '\n')
	    {
	      inheader = 0;
#ifdef JOIN_HEADER_AND_TEXT
	      continue;
#endif
	    }
	  else if (line[0] == ' ' || line[0] == '\t'
		   || strncmp (line, "From:", 5)
		   && strncmp (line, "Subject:", 8))
	    /* Skip header continuation lines and non-essential header lines. */
	    continue;
	}
      if (linecnt <= 0 || charcnt <= 0)
	{
	  fprintf (tp, "...more...%s\n", cr);
	  return;
	}
      cp = strchr (line, '\n');
      if (cp)
	*cp = '\0';
      fprintf (tp, "%s%s\n", line, cr);
      charcnt -= strlen (line);
      linecnt--;
    }
  fprintf (tp, "----%s\n", cr);
}

/* Simulate the BSD gethostname(2) system call on System V.  */

int
gethostname (name, length)
     char *name;
     int length;
{
  struct utsname uts;

  if (uname (&uts) < 0)
    return -1;
  strncpy (name, uts.nodename, length);
  return 0;
}

static void
memory_out ()
{
  fprintf (stderr, "%s: Virtual memory exhausted\n", program_name);
  exit (1);
}

/* Allocate `n' bytes of memory dynamically, with error checking.  */

char *
xmalloc (n)
     unsigned n;
{
  char *p;

  p = malloc (n);
  if (p == 0)
    memory_out ();
  return p;
}

/* Change the size of an allocated block of memory `p' to `n' bytes,
   with error checking.
   If `p' is NULL, run xmalloc.
   If `n' is 0, run free and return NULL.  */

char *
xrealloc (p, n)
     char *p;
     unsigned n;
{
  if (p == 0)
    return xmalloc (n);
  if (n == 0)
    {
      free (p);
      return 0;
    }
  p = realloc (p, n);
  if (p == 0)
    memory_out ();
  return p;
}

/* ANSI C function. */

char *
strerror (n)
     int n;
{
  extern char *sys_errlist[];
  extern int sys_nerr;

  return n >= 0 && n < sys_nerr ? sys_errlist[n] : "Unknown error";
}

/* Print "program_name: str_and_optional_args: perror_message" on stderr,
   then exit with error status. */
/* VARARGS */
void
msg_perror_fatal (str, va_alist)
     char *str;
     va_dcl
{
  va_list args;
  extern int errno;
  int save_errno;

  save_errno = errno;
  fprintf (stderr, "%s: ", program_name);
  va_start (args);
  vfprintf (stderr, str, args);
  va_end (args);
  fprintf (stderr, ": %s\n", strerror (save_errno));
  exit (1);
}

/* Open `path' for reading as a mode 0600 FIFO, creating it if necessary.
   Return the file descriptor. */

int
open_fifo (path)
     char *path;
{
  int fifd;

  if (mknod (path, 010600, 0) == -1 && errno != EEXIST)
    msg_perror_fatal ("Cannot create FIFO %s", path);
  if (chmod (path, 0600))
    msg_perror_fatal ("Cannot change mode of FIFO %s", path);
  while ((fifd = open (path, O_RDONLY | O_TRUNC)) == -1 && errno == EINTR)
     /* Do nothing. */ ;
  if (fifd == -1)
    msg_perror_fatal ("Cannot open FIFO %s for reading", path);
  return fifd;
}

/* Return nonzero if there is another process already reading `fifo'.
   If there isn't, open will fail with EPIPE (write on broken pipe). */

int
daemon_running (fifo)
     char *fifo;
{
  int fifd;

  fifd = open (fifo, O_WRONLY | O_NDELAY);
  if (fifd == -1)
    return 0;
  else
    {
      close (fifd);
      return 1;
    }
}
