/* mail-round-robiner
   Version 1.0 of 3/21/90 by Max Hailperin <Hailperin@CS.Stanford.edu>
   Copyright (C) 1990 by Max Hailperin
   Permission to copy all or part of this work is granted, provided
   that the copies are not made or distributed for resale, and that
   the copyright notice and this notice are retained.
   THIS WORK IS PROVIDED ON AN "AS IS" BASIS.  THE AUTHOR PROVIDES NO
   WARRANTY WHATSOEVER, EITHER EXPRESS OR IMPLIED, REGARDING THE WORK,
   INCLUDING WARRANTIES WITH RESPECT TO ITS MERCHANTABILITY OR FITNESS
   FOR ANY PARTICULAR PURPOSE.

   This program can be used on a BSD-derived system to cause mail received at a
   mailbox to be forwarded in round-robin rotation to one of a list of
   mailboxes.  This is useful for providing help, for example.  To use it, the
   incoming mailbox should be made an alias for something like
   "|/a/mxh/LaTeX-help/mail-round-robiner /a/mxh/LaTeX-help/log
    /a/mxh/LaTeX-help/latex-help-helpers.alias
    /a/mxh/LaTeX-help/round-robin-state
    LaTeX-help-coordinator@sumex-aim.stanford.edu".
   (This is a concrete example, namely the one for which this program was
   written.)  The first command-line argument is the name of a log file,
   useful for debugging purposes and figuring out who it was who didn't
   respond to some piece of mail.  The second is the list of outgoing
   mailboxes, in a format documented below.  The third is a file (which should 
   be initialized to "0") which keeps the round-robiners place.  The fourth is
   the mail address from which the mail is sent.  There can optionally be a
   fifth argument, to specify a non-local SMTP server host.

   Note that file permissions must be set appropriately for world access.
   The log file and state file must be world writable.  The state and address
   files must be world readable.  The executable of this program must be
   world executable, and so must the directories which are ancestors of
   all the files (where executable means searchable).

   The format of the addresses file must be scrupulously maintained, as it is
   used by two programs (the mailer and the round-robiner), neither of which
   is very flexible about format.  Why does the mailer read it?  Because it
   is also useful as an alias file for all the members of the round-robin.
   1) Make sure there is exactly one newline at the end of the file.
   2) There must be no blank lines.  Each line should either have a
      mail address, a suspended mail address, or a /dev/null place holder
      on it.
   3) Mail addresses must begin with a non-blank character other than /.
   4) Suspended mail addresses must begin with a blank.
   5) Each line must end with a comma, except for the last line, which mustn't.
   6) NEVER delete a line from this file, or the round-robiner will
      lose its place.  Instead, replace the address with a /dev/null, if
      it is a permanent deletion, or precede it with a space if it is a
      temporary suspension (e.g. for a vacation).  (That way the round-robiner
      will skip the person on vacation, but they will still get all general
      mail.)  Actually, that NEVER isn't quite absolute, but its easier to
      pretend it is, and the file is unlikely to get very junked up.
   7) New addresses can either be added at the end or replace a /dev/null.
      Someone returning from vacation should, of course, merely have the space
      before their name removed.  If you add a new line at the end, be sure
      to respect rules 1 and 5.

   Please note that this program doesn't enforce sender authentication, but
   neither does it provide a means for defeating it if there wasn't already
   one.

   Known bugs:
     - prepending X- to the return-path header line may make it exceed the
       RFC 821 limit of 1000 characters.
   Known weaknesses (small subset, please extend):
     - relies way too heavily on the input complying with RFCs 821, 822
   */

#include <stdio.h>
#include <ctype.h>
#include <sys/file.h>
#include <sysexits.h>
#include <strings.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <varargs.h>

#define HELO_553_OK /* workaround sendmail bug */
#define SCRAP_UNIX_FROM_LINE /* non-RFC 821 line some unix systems prepend */

#define DEFAULT_SMTP_SERVER_HOST "localhost"
#define SMTP_SERVER_HOST ((argc < 6) ? DEFAULT_SMTP_SERVER_HOST : argv[5])

/* The OLD_HEADER_PREFIX is used to "comment out" any header fields that are
   obsoleted in the forwarding--currently only the Return-Path.  "X-" has two
   advantages: 1) it is guaranteed not to begin any current or future official
   header field name 2) it is mnemonic for "ex-". */
#define OLD_HEADER_PREFIX "X-"

#define peekc(f) (ungetc(getc((f)),(f)))
#define peekchar() (peekc(stdin))

extern int sys_nerr;
extern int errno;
extern char *sys_errlist[];
static char errmsg_buf[256];
#define ERRMSG ((errno<sys_nerr)?sys_errlist[errno]:\
                                 (sprintf(errmsg_buf,"Error number %d",errno),\
                                  errmsg_buf))

static FILE *log_stream, *server_stream_w, *server_stream_r;

static int    /* these are from RFC821, section 4.3 */
  connect_success[] = {220, 0},
  connect_failure[] = {421, 0},
  connect_error[] = {0},
  helo_success[] = {
#ifdef HELO_553_OK
    553,
#endif
    250, 0},
  helo_failure[] = {0},
  helo_error[] = {500, 501, 553, 504, 421, 0},
  mail_success[] = {250, 0},
  mail_failure[] = {552, 451, 452, 0},
  mail_error[] = {500, 501, 421, 0},
  rcpt_success[] = {250, 251, 0},
  rcpt_failure[] = {550, 551, 552, 553, 450, 451, 452, 0},
  rcpt_error[] = {500, 501, 503, 421, 0},
  data_success[] = {354, 0},
  data_failure[] = {451, 554, 0},
  data_error[] = {500, 501, 503, 421, 0},
  eod_success[] = {250, 0}, /* eod = end of data = dot after the text */
  eod_failure[] = {552, 554, 451, 452, 0},
  eod_error[] = {0},
  quit_success[] = {221, 0},
  quit_failure[] = {0},
  quit_error[] = {500, 0};

static struct hostent *hep;

main(argc, argv)
     int argc;
     char **argv;
{
  int state, c;
  FILE *state_stream;
  char address[257], *cannonical_hostname, *my_strdup();
  FILE *fopen(), *fdopen();
  long ftell();

  if((argc < 5) || (argc > 6)){
    fprintf(stderr,
            "Usage: %s log_file addresses_file state_file errors_address [smpt_server_host]\n",
            argv[0]);
    exit(EX_USAGE);
  }

  if((state_stream = fopen(argv[3], "r+")) == NULL)
    error("Error opening state file", argv[3], NULL);
  if(flock(fileno(state_stream), LOCK_EX) == -1) /* against siblings */
    error("Error locking state file", argv[3], ERRMSG);
  if(fscanf(state_stream, "%d", &state) != 1)
    error("Error reading state file", argv[3], NULL);
  
  {
    FILE *addresses_stream;

    if((addresses_stream = fopen(argv[2], "r")) == NULL)
      error("Error opening addresses file", argv[2], NULL);
    if(flock(fileno(addresses_stream), LOCK_SH) == -1) /* against emacs */
      error("Error locking addresses file", argv[2], ERRMSG);
    {
      int i;
      for(i = 0; i < state; i++){
	while(((c = getc(addresses_stream)) != EOF) && (c != '\n'))
	  ;
	if(c == EOF){
	  rewind(addresses_stream);
	  i = state = 0;
	  break;
	}
      }
      {
	int rewound = 0;
	while(((c = peekc(addresses_stream)) == '/') || (c == ' ')
	      || (c == EOF)){
	  if(rewound && (state == i))
	    error("No addresses in address file", argv[2], NULL);
	  state++;
	  while(((c = getc(addresses_stream)) != EOF) && (c != '\n'))
	    ;
	  if(c == EOF){
	    rewind(addresses_stream);
	    state = 0;
	    rewound = 1;
	  }
	}
      }
    }
    if(fscanf(addresses_stream, "%[^,\n]", address) != 1)
      error("Error reading address from address file", argv[2], NULL);
    fclose(addresses_stream);
  }

  {
    char hostname[MAXHOSTNAMELEN + 1];

    if(gethostname(hostname, MAXHOSTNAMELEN + 1) == -1)
      error("Error getting hostname", "gethostname", ERRMSG);
    if((hep = gethostbyname(hostname)) == NULL)
      error("Error cannonicalizing hostname", hostname, NULL);
  }
  if((cannonical_hostname = my_strdup(hep->h_name)) == NULL)
    error("Can't malloc space to copy cannonical hostname", hep->h_name, NULL);

  {
    int server_socket;
    {
        struct protoent *pep, *getprotobyname();

	if((pep = getprotobyname("tcp")) == NULL)
	  error("Error looking up protocol", "tcp", NULL);
	if((server_socket = socket(AF_INET, SOCK_STREAM, pep->p_proto)) < 0)
	  error("Error creating a socket", "server_socket", ERRMSG);
      }
    if((hep = gethostbyname(SMTP_SERVER_HOST)) == NULL)
      error("Error looking up server host", SMTP_SERVER_HOST, NULL);
    {
          struct sockaddr_in server_address;

	  bcopy(hep->h_addr, &server_address.sin_addr, hep->h_length);
	  server_address.sin_family = AF_INET;
	  {
	      struct servent *sep, *getservbyname();

	      if((sep = getservbyname("smtp", "tcp")) == NULL)
		error("Error looking up server port", "smpt/tcp", NULL);
	      server_address.sin_port = sep->s_port;
	    }
	  if(connect(server_socket, &server_address, sizeof(server_address))
	      < 0)
	    error("Error connecting to server", hep->h_name, NULL);
	}
    if((server_stream_r = fdopen(server_socket, "r")) == NULL)
      error("Error making read stream of server socket", hep->h_name, NULL);
    if((server_stream_w = fdopen(server_socket, "w")) == NULL)
      error("Error making write stream of server socket", hep->h_name, NULL);
  }
  setbuf(server_stream_r, NULL); /* setlinebuf should suffice for these, but */
  setbuf(server_stream_w, NULL); /* doesn't under Ultrix */

  if((log_stream = fopen(argv[1], "a")) == NULL)
    error("Error opening log file", argv[1], NULL);
  if(flock(fileno(log_stream), LOCK_EX) == -1)    /* against emacs */
    error("Error locking log file", argv[1], ERRMSG);
  setbuf(log_stream, NULL); /* worth the performance penalty to know
			       what happened we we get a fault */

  servwait(connect_success, connect_failure, connect_error);
  servprintf("HELO %s\r\n", cannonical_hostname);
  servwait(helo_success, helo_failure, helo_error);
  servprintf("MAIL From: <%s>\r\n", argv[4]);
  servwait(mail_success, mail_failure, mail_error);
  servprintf("RCPT To: <%s>\r\n", address);
  servwait(rcpt_success, rcpt_failure, rcpt_error);
  servprintf("DATA\r\n");
  servwait(data_success, data_failure, data_error);

  {
#ifdef SCRAP_UNIX_FROM_LINE
    int headers = 0;
    char s[2];
#endif

    while(((c = peekchar()) != '\n') && (c != EOF)){ /* should never be EOF */
      char hdrbuf[1001], hdrbuf2[1001];

      if(fgets(hdrbuf2, 1001, stdin) == NULL)
	error("Error reading header line", "standard input", NULL);
#ifdef SCRAP_UNIX_FROM_LINE
      if(headers || !sscanf(hdrbuf2, "From %1[^:]", s))
#endif
	{
	if(sscanf(hdrbuf2, "%[^: \t]", hdrbuf) != 1)
	  error("Error parsing header field", "standard input", hdrbuf2);
	if(hfeql(hdrbuf, "return-path"))
	  servprintf("%s", OLD_HEADER_PREFIX);
	hdrbuf2[strlen(hdrbuf2) - 1] = '\r';
	servprintf("%s", hdrbuf2);
	servputchar('\n');
	while(((c = peekchar()) == ' ') || (c == '\t')){
	  if(fgets(hdrbuf, 1001, stdin) == NULL)
	    error("Error reading header continuation line", "standard input",
		  NULL);
	  hdrbuf[strlen(hdrbuf) - 1] = '\r';
	  servprintf("%s", hdrbuf);
	  servputchar('\n');
	}
      }
    }
  }

  {
    int bol = 1;
    while((c = getchar()) != EOF){
      if(bol && c == '.'){
	bol = 0;
	servputchar('.');
      } else if(c == '\n'){
	bol = 1;
        servputchar('\r');
      } else
	bol = 0;
      servputchar(c);
    }
    if(!bol)
      servprintf("\r\n");
  }

  servprintf(".\r\n");
  servwait(eod_success, eod_failure, eod_error);
  servprintf("QUIT\r\n");
  servwait(quit_success, quit_failure, quit_error);
  fclose(log_stream);
  fclose(server_stream_w);
  fclose(server_stream_r);

  rewind(state_stream);
  fprintf(state_stream, "%d\n", state + 1);
  fflush(state_stream);
  if(ftruncate(fileno(state_stream), ftell(state_stream)) == -1)
    error("Error reseting state file length", argv[3], ERRMSG);
  fclose(state_stream);
}

static error(s1, s2, s3)
     char *s1, *s2, *s3;
{
  if(s3 == NULL)
    fprintf(stderr, "%s: %s\n", s1, s2);
  else
    fprintf(stderr, "%s: %s (%s)\n", s1, s2, s3);
  exit(EX_SOFTWARE);
}

static int hfeql(hf1, hf2)  /* header-field name equality; hf2 lower case */
     char *hf1, *hf2;
{
  for( ; *hf1 && *hf2; hf1++, hf2++)
    if(((isupper(*hf1))?tolower(*hf1):*hf1) != *hf2)
      return(0);
  return(*hf1 == *hf2);
  }

static servwait(s, f, e)
     int *s, *f, *e;
{
  int code, first_line = 1, is_not_success = 0;
  char c, line[513];

  do{
    if(fgets(line, 513, server_stream_r) == NULL)
      error("SMTP server read error or unexpected EOF", hep->h_name, NULL);
    fputs(line, log_stream);
    if(sscanf(line, "%d%c", &code, &c) != 2)
      error("SMTP server malformed reply", hep->h_name, line);
    if(is_not_success || (first_line && (is_not_success = !memq(code, s)))){
      if(first_line){
	if(memq(code, f))
	  fprintf(stderr, "SMTP server %s failure:\n", hep->h_name);
	else if(memq(code, e))
	  fprintf(stderr, "SMTP server %s error:\n", hep->h_name);
	else
	  fprintf(stderr, "SMTP server %s protocol error, illegal reply:\n",
		  hep->h_name);
	first_line = 0;
      }
      fputs(line, stderr);
    }
  } while(c == '-');
  if(c != ' ')
    error("SMTP server malformed reply", hep->h_name, line);
  if(is_not_success){
    servprintf("QUIT\r\n");
    error("Closing connection to", hep->h_name, NULL);
  }
}

static servputchar(c)
     int c;
{
  putc(c, server_stream_w);
  putc(c, log_stream);
}

static servprintf(va_alist)
     va_dcl
{
  va_list l;
  char *fmt;

  va_start(l);
  fmt = va_arg(l, char *);
  vfprintf(server_stream_w, fmt, l);
  va_end(l);
  va_start(l);
  fmt = va_arg(l, char *);
  vfprintf(log_stream, fmt, l);
  va_end(l);
}

static int memq(i, ip)
     int i, *ip;
{
  for(; *ip; ip++)
    if(i == *ip)
      return(1);
  return(0);
}

static char *my_strdup(old) /* for systems without strdup, e.g. Ultrix */
     char *old;
{
  char *new, *malloc();

  if((new = malloc(strlen(old))) != NULL)
    strcpy(new, old);
  return(new);
}
