/*
** COMSAT
**
** A network version of the BSD biff server.
**
** Written by Steven Grimm (koreth@ssyx.ucsc.edu), 12-27-88.
*/
#include <sys/types.h>
#include <utmp.h>
#include <stdio.h>
#include <netdb.h>
#include <ctype.h>
#include <strings.h>
#include <signal.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifndef MAXPATHLEN
#include <sys/param.h>
#endif
#include "config.h"

#ifndef FD_SETSIZE	/* for 4.2BSD */
#define FD_SETSIZE      (sizeof(fd_set) * 8)
#define FD_SET(n, p)    (((fd_set *) (p))->fds_bits[0] |= (1 << ((n) % 32)))
#define FD_CLR(n, p)    (((fd_set *) (p))->fds_bits[0] &= ~(1 << ((n) % 32)))
#define FD_ISSET(n, p)  (((fd_set *) (p))->fds_bits[0] & (1 << ((n) % 32)))
#define FD_ZERO(p)      bzero((char *)(p), sizeof(*(p)))
#endif

extern	char *sys_errlist[];
extern	int errno;

int	udp_fd, tcp_fd;
struct	sockaddr_in tcp_host[FD_SETSIZE];
fd_set	readset;

/*
** A linked list of these structures is used to keep track of remote users
** who are watching mailboxes on the local host.
*/
struct ruser {
	struct	sockaddr_in host;	/* watcher's host */
	char	watcher[64];		/* watcher's name */
	char	watchee[64];		/* watchee's name */
	int	local;			/* set if host==localhost */
	int	flag;			/* flag for temporary use */
	struct	ruser *next;		/* link to next entry */
};

struct ruser *rlist = NULL;

/*
** Add a remote user to the watch list.  Returns 0 on success, 1 on
** malloc failure, 2 if that user is repeating an existing request,
** or 3 if the remote user authentication fails.
*/
r_add(host, watcher, watchee)
struct sockaddr_in *host;
char *watcher, *watchee;
{
	struct	ruser *entry;
	struct	hostent *hostent;
	int	ruok;
	char	hbuf[64];

	for (entry = rlist; entry != NULL; entry = entry->next)
		if ((!strcmp(watcher, entry->watcher)) &&
				(!strcmp(watchee, entry->watchee)))
			return 2;

	if ((hostent = gethostbyaddr(&host->sin_addr, sizeof(host->sin_addr),
				AF_INET)) == NULL)
		strcpy(hbuf, inet_ntoa(host->sin_addr));
	else
		strncpy(hbuf, hostent->h_name, 63);
	hbuf[63] = '\0';
	if (! host->sin_addr.s_addr)
		strcpy(hbuf, "localhost");

	ruok = ruserok(hbuf, 0, watcher, watchee);
	setreuid(0,0);		/* make up for some Sun bugs */
	if (ruok)
		return 3;

	entry = (struct ruser *)malloc(sizeof(struct ruser));
	if (entry == NULL)
		return 1;

	entry->next = rlist;
	rlist = entry;
	strncpy(entry->watcher, watcher, 64);
	strncpy(entry->watchee, watchee, 64);
	bcopy(host, &entry->host, sizeof(struct sockaddr_in));
	entry->local = localhost(host);

	return 0;
}

/*
** Compare a host-username pair with the remote host and username in a
** watch list entry.  Returns 0 if they're the same.  Watche[er] may be empty
** in which case it's ignored; host is always significant.
*/
r_cmp(host, watcher, watchee, entry)
struct sockaddr_in *host;
char *watcher, *watchee;
struct ruser *entry;
{
	if (watcher[0] && strcmp(watcher, entry->watcher))
		return 1;
	if (watchee[0] && strcmp(watchee, entry->watchee))
		return 1;
/* we have to check for localhost, since watchees registered without */
/* machine names will be watched by 'localhost', and checking for */
/* user@realmachinename will fail unless we do the special check. */
	if (entry->local)
	{
		if (!localhost(host))
			return 1;
	}
	else if (bcmp(&host->sin_addr, &entry->host.sin_addr,
			sizeof(host->sin_addr)))
		return 1;
	return 0;
}

/*
** Delete a remote user from the watch list.
*/
r_del(host, watcher, watchee)
struct sockaddr_in *host;
char *watcher, *watchee;
{
	struct ruser *cur, *tmp;
	int delflg = 0;

#ifdef DEBUG
	printf("r_del(%s, %s, %s)\n", inet_ntoa(host->sin_addr.s_addr),
					watcher, watchee);
#endif

	while (rlist != NULL)
		if (! r_cmp(host, watcher, watchee, rlist))
		{
			tmp = rlist;
			rlist = rlist->next;
			free(tmp);
			delflg = 1;
		}
		else
			break;

	if (rlist == NULL)
		return delflg;

	cur = rlist;

	while (cur->next != NULL)
		if (! r_cmp(host, watcher, watchee, rlist))
		{
			tmp = cur->next;
			cur->next = cur->next->next;
			free(tmp);
			delflg = 1;
		}
		else
			cur = cur->next;

	return delflg;
}

/*
** Open and bind the necessary sockets.  One socket listens for datagrams,
** the other for stream connection requests.
*/
void
getsocks()
{
	struct servent *tcpsvc, *udpsvc;
	struct sockaddr_in addr;

	tcpsvc = getservbyname("rbiff", "tcp");

	tcp_fd = socket(PF_INET, SOCK_STREAM, 0);
	if (tcp_fd < 0)
		panic("tcp socket()");

	udp_fd = socket(PF_INET, SOCK_DGRAM, 0);
	if (udp_fd < 0)
		panic("udp socket()");

	bzero(&addr, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;

	if (tcpsvc == NULL)
		addr.sin_port = htons(TCP_PORT);
	else
		addr.sin_port = tcpsvc->s_port;
	if (bind(tcp_fd, &addr, sizeof(addr)) < 0)
		panic("tcp bind()");

	udpsvc = getservbyname("biff", "udp");

	if (udpsvc == NULL)
		addr.sin_port = htons(UDP_PORT);
	else
		addr.sin_port = udpsvc->s_port;
	if (bind(udp_fd, &addr, sizeof(addr)) < 0)
		panic("udp bind()");

	if (listen(tcp_fd, 5) < 0)
		panic("listen (weird)");

	if (fcntl(tcp_fd, F_SETFL, FIONBIO) == -1)
		panic("fcntl tcp_fd");
	if (fcntl(udp_fd, F_SETFL, FIONBIO) == -1)
		panic("fcntl udp_fd");

	FD_ZERO(&readset);
	FD_SET(tcp_fd, &readset);
	FD_SET(udp_fd, &readset);
}

/*
** Send a message to a user, once to each of his ttys that has the owner
** execute bit set.
*/
void
biff(user, msg)
char *user, *msg;
{
	char filename[MAXPATHLEN];
	struct utmp uent;
	struct stat st;
	FILE *fp;
	int tty;

	if ((fp = fopen(UTMP, "r")) == NULL)
		return;

	while (! feof(fp))
	{
		if (! fread(&uent, sizeof(uent), 1, fp))
			break;
		if (strcmp(uent.ut_name, user))
			continue;
		sprintf(filename, "%s/%s", DEVDIR, uent.ut_line);
		if (stat(filename, &st) < 0)
			continue;
		if ((st.st_mode & S_IEXEC)
#ifndef DEBUG
			&& !fork()
#endif
			)
		{
			tty = open(filename, O_RDWR);
			if (tty)
				write(tty, msg, strlen(msg));
#ifdef DEBUG
			close(tty);
#else
			exit(0);
#endif
		}
	}
	fclose(fp);
}

/*
** Is a header line valid?  This routine finds out.  Returns 0 if the
** line shouldn't be included in the biff message.
*/
hdrline(buf)
char *buf;
{
	int	i;

	for (i = 0; i < Nfields; i++)
		if (!strncmp(buf, fields[i], strlen(fields[i])))
			return 1;
	return 0;
}

/*
** Build a message to send out to everyone who's waiting for mail
** to a specific user.  Pass the username, offset in his mailbox of
** the new mail, and the address of a buffer at least 800 bytes long.
*/
void
buildmsg(name, offset, buf)
char *name, *buf;
int offset;
{
	struct	ruser *cur;
	struct	stat st;
	struct	timeval tvp[2];
	FILE	*fp;
	int	bpos, body;
	char	filename[MAXPATHLEN];

	gethostname(filename, 79);
	filename[79] = '\0';

	sprintf(buf, "\n\r\007New mail for %.20s@%.30s\007 has arrived:\n\r----\n\r",
		name, filename);
	bpos = strlen(buf);

	sprintf(filename, "%s/%s", MAILDIR, name);

	stat(filename, &st);
	if ((fp = fopen(filename, "r")) == NULL)
	{
		sprintf(buf+bpos, "(%s: %s)\n\r----\n\r", filename,
			sys_errlist[errno]);
		return;
	}

	body = 0;	/* we are looking at the message header. */

	fseek(fp, offset, 0);

	while (! feof(fp))
	{
		if (fgets(buf+bpos, 780-bpos, fp) == NULL)
		{
			strcpy(buf+bpos, "----\n\r");
			break;
		}
		if (! body)
		{
			if (buf[bpos] == '\n')
			{
				bpos++;
				body++;
			}
			else if (hdrline(buf+bpos))
			{
				bpos += strlen(buf+bpos);
				buf[bpos++] = '\r';
			}
		}
		else
		{
			if (buf[bpos] == '\n')
				continue;
			if (++body == 4)
			{
				strcpy(buf+bpos, "...more...\n\r");
				break;
			}
			bpos += strlen(buf+bpos);
			buf[bpos++] = '\r';
		}
	}
	fclose(fp);

/* login's "you have new mail" message won't work unless mtime > atime */
	bzero(tvp, sizeof(tvp));
	tvp[0].tv_sec = st.st_atime;
	tvp[1].tv_sec = st.st_mtime;
	utimes(filename, tvp);
}

/*
** Determine whether or not a host address points to the local host.  Returns
** 0 if the host isn't local.  There's probably a much better way of doing
** this: first we see if the host is called "localhost", then see if its
** name is the same one that gethostname() returns, then see if gethostname()
** returns a machine name whose address is equal to the host address we're
** checking.  Yucko.
*/
localhost(addr)
struct sockaddr_in *addr;
{
	struct	hostent *host;
	char	hname[64];
	int	i;

	if (! addr->sin_addr.s_addr)
		return 1;

	gethostname(hname, 64);

	if ((host = gethostbyaddr(&addr->sin_addr, sizeof(addr->sin_addr),
			AF_INET)) != NULL)
	{
		if (! stricmp(host->h_name, "localhost"))
			return 1;
		if (! stricmp(host->h_name, hname))
			return 1;
	}

	if ((host = gethostbyname(hname)) != NULL)
#ifndef h_addr
		if (!bcmp(host->h_addr,&addr->sin_addr,host->h_length))
			return 1;
#else /* if we have multiple host addresses, check all of 'em */
		for (i = 0; host->h_addr_list[i] != NULL; i++)
			if (! bcmp(host->h_addr_list[i], &addr->sin_addr,
					sizeof(addr->sin_addr)))
				return 1;
#endif
	return 0;
}

/*
** See if a user is logged in.  Return 1 if he is.
*/
logged_in(user)
char *user;
{
	struct	utmp uent;
	int	sawhim = 0;
	FILE	*fp;

	fp = fopen(UTMP, "r");
	if (fp == NULL)
		return 0;

	while (! feof(fp))
	{
		if (! fread(&uent, sizeof(uent), 1, fp))
			break;
		if (strcmp(uent.ut_name, user))
			continue;
		sawhim++;
		break;
	}
	fclose(fp);

	return(sawhim);
}

/*
** Open up a dialogue with a remote comsat, and send it a biff for someone.
*/
void
rbiff(cur, buf)
struct ruser *cur;
char *buf;
{
	char	biffcmd[90];
	int	fd, port, len;

	sprintf(biffcmd, "B%s%c%d", cur->watcher, '\0', strlen(buf) + 1);

/* connect to the remote, as specified in the rlist entry.  make sure we */
/* are coming from a secure port else the other guy will hang up on us.  */
	port = 1023;
	fd = rresvport(&port);
	if (fd < 0)
		return;
	if (connect(fd, &cur->host, sizeof(cur->host)))
	{
		close(fd);
		return;
	}

	len = strlen(biffcmd);
	len += strlen(biffcmd+len+1)+2;
	write(fd, biffcmd, len);

	if (len = gack(fd))
	{
		shutdown(fd, 2);
		close(fd);
		return;
	}

	write(fd, buf, strlen(buf)+1);

	gack(fd);

	shutdown(fd, 2);
	close(fd);
}

/*
** Handle an incoming datagram.  This should be a message of the form
** "user@offset", where user is the user who has new mail and offset
** is the byte offset into his mailbox of the new message.  If SECUREBIFF
** is defined in config.h, the datagram must come from a reserved (<1024)
** port on the local host.
*/
void
udp()
{
	struct	ruser *cur;
	struct	sockaddr_in from;
	char	buf[800], user[64], *cp;
	int	offset;

	offset = sizeof(from);
	if ((offset = recvfrom(udp_fd, buf, 800, 0, &from, &offset)) < 3)
		return;
	buf[offset] = '\0';

#ifdef SECUREBIFF
	if (ntohs(from.sin_port) >= IPPORT_RESERVED)
		return;
	if (! localhost(&from))
		return;
#endif

	if ((cp = index(buf, '@')) == NULL)
		return;
	*cp++ = '\0';

	strncpy(user, buf, 63);
	user[63] = '\0';

	if (! isdigit(*cp))
		return;
	offset = atoi(cp);

	buildmsg(user, offset, buf);
	biff(user, buf);

	for (cur = rlist; cur != NULL; cur = cur->next)
		if (! strcmp(cur->watchee, user))
/* if the watcher is on the local host, just biff him. */
			if (cur->local)
			{
/* if the watcher is watching himself, don't biff him again. */
				if (strcmp(user, cur->watcher))
					biff(cur->watcher, buf);
			}
			else
				rbiff(cur, buf);
}

/*
** Handle a pending TCP connection.  If it's not coming from a secure port
** on the remote host, close it immediately.
*/
void
tcp()
{
	struct sockaddr_in host;
	int len, tcp_fd_c;

	len = sizeof(host);
	tcp_fd_c = accept(tcp_fd, &host, &len);
	if (tcp_fd_c < 0)
		return;

	if (ntohs(host.sin_port) >= IPPORT_RESERVED)
	{
		shutdown(tcp_fd_c, 2);
		close(tcp_fd_c);
		return;
	}

	len = 1;
	setsockopt(tcp_fd_c, SOL_SOCKET, SO_KEEPALIVE, &len, sizeof(len));
	fcntl(tcp_fd_c, F_SETFL, FIONBIO);

	bcopy(&host, &tcp_host[tcp_fd_c], sizeof(host));
	FD_SET(tcp_fd_c, &readset);
}

/*
** Handle incoming data on a connected socket.  This should be in the form
** of a single-character command byte followed by null-terminated parameters.
** There probably isn't enough error checking here.
*/
void
tcpc(fd)
int fd;
{
	char	buf[800], user[64];
	int	nbytes;

	ioctl(fd, FIONREAD, &nbytes);
	if (! nbytes)		/* 0 bytes available means disconnect */
	{
		shutdown(fd, 2);
		close(fd);
		FD_CLR(fd, &readset);
		return;
	}

	if (nbytes > sizeof(buf))
		nbytes = sizeof(buf);

	read(fd, buf, nbytes);
	buf[nbytes] = '\0';

	if (buf[0] == 'W')	/* Watch watcher\0watchee\0 */
		ack(fd, r_add(&tcp_host[fd], buf+1, buf+strlen(buf) + 1));
	else if (buf[0] == 'D')	/* Delete watcher\0[watchee]\0 */
		ack(fd, r_del(&tcp_host[fd], buf+1, buf+strlen(buf) + 1));
	else if (buf[0] == 'V') /* Verify watcher\0 */
		ack(fd, logged_in(buf+1));
	else if (buf[0] == 'P')	/* Port portnum\0 */
	{
		tcp_host[fd].sin_port = htons(atoi(buf+1));
		ack(fd, 0);
	}
	else if (buf[0] == 'B')	/* Biff watcher\0length\0 */
	{
		char	user[64];
		int	len, nread;

		strcpy(user, buf+1);
		len = atoi(buf+strlen(buf) + 1);
		if (len > sizeof(buf))	/* don't wanna overflow our buffer */
		{
			ack(fd, 1);
			return;
		}

		ack(fd, 0);	/* tell him to start sending */

		nbytes = 0;
		while (nbytes < len)
		{
			fd_set readbits;
			struct timeval tout;

			FD_ZERO(&readbits);
			FD_SET(fd, &readbits);
			timerclear(&tout);
			tout.tv_sec = 20;	/* biff msg must arrive fast */

			select(fd+1, &readbits, NULL, NULL, &tout);
			nread = read(fd, buf+nbytes, len-nbytes);
			if (nread < 1)	/* if timed out, we get EWOULDBLOCK */
			{
				close(fd);
				FD_CLR(fd, &readset);
				return;
			}
			nbytes += nread;
		}
		biff(user, buf);
		ack(fd, 0);
	}
}

/*
** Send a byte out on a file descriptor.
*/
ack(fd, ch)
int fd, ch;
{
	char c;

	c = (char) ch;
	write(fd, &c, 1);
}

/*
** Read a byte from a file descriptor.
*/
gack(fd)
{
	char c;

	read(fd, &c, 1);
	return (int) c;
}

/*
** Do a perror() and exit.
*/
panic(string)
char *string;
{
	perror(string);
	exit(-1);
}

/*
** A custom bcopy whose behavior is known for overlapping regions
*/
bcopy(src, des, len)
register char *src, *des;
register int len;
{
	while (len--)
		*des++ = *src++;
}

/*
** Case-insensitive compare.  This should use lookup tables for fast case
** conversion.
*/
stricmp(str1, str2)
register char *str1, *str2;
{
	while (*str1 || *str2)
	{
		if ((islower(*str1) ? toupper(*str1) : *str1) !=
		    (islower(*str2) ? toupper(*str2) : *str2))
			return 1;
		str1++;
		str2++;
	}
	return 0;
}

/*
** See which watchers are still online, and delete the entries of those
** who have logged off.  This probably scans through the list a lot more
** times than is necessary.  Oh well.
*/
void
rollcall()
{
	struct	ruser *cur, *user, *place;
	int	port, fd, uflag;
	char	buf[66], code;

/* if there are no watchers, this is pretty easy. */

	if (rlist == NULL)
		return;

/* first, see who's logged on locally.  while we're going through the linked */
/* list, set all the remote-host entries' flag fields to 0 and all the local */
/* ones to 1, to simplify things later. */

	for (cur = rlist; cur != NULL; cur = cur->next)
		if (cur->local)
			if (logged_in(cur->watcher))
				cur->flag = 1;
			else
				cur->flag = -1;
		else
			cur->flag = 0;

/* now go through and look at the remote hosts.  each time we find a remote */
/* host, we will connect to it then scan forward in the linked list so that */
/* we only have to connect to each host once. */

	buf[0] = 'V';

	for (place = rlist; place != NULL; place = place->next)
	{
		if (place->flag)
			continue;

		port = 1023;
		fd = rresvport(&port);
		if (fd < 0)
			break;

/* if we can't connect to a host, toast all entries from that host. */

		if (connect(fd, &place->host, sizeof(place->host)))
		{
			for (cur = place; cur != NULL; cur = cur->next)
				if (! cur->flag)
					if (! r_cmp(&place->host, "", "", cur))
						cur->flag = -1;
		}

/* ask about each watcher on the current host. */

		else for (user = place; user != NULL; user = user->next)
		{
			if (user->flag)
				continue;
			if (r_cmp(&place->host, "", "", user))
				continue;
			strcpy(buf+1, user->watcher);
			write(fd, buf, strlen(buf)+1);
			read(fd, &code, 1);
			uflag = code ? 1 : -1;

/* toast or keep all the watcher's entries. */

			for (cur = user; cur != NULL; cur = cur->next)
				if (!(cur->flag || r_cmp(&user->host,
						user->watcher, "", cur)))
					cur->flag = uflag;
		}
		close(fd);
	}

/* now go through and delete all the entries that are marked as toast. */

	while (rlist != NULL)
		if (rlist->flag == -1)
		{
			cur = rlist;
			rlist = rlist->next;
			free(cur);
		}
		else
			break;

	if (rlist == NULL)
		return;

	cur = rlist;
	while (cur->next != NULL)
		if (cur->next->flag == -1)
		{
			place = cur->next;
			cur->next = cur->next->next;
			free(place);
		}
		else
			cur = cur->next;
}

#ifdef DEBUG
/*
** Dump the ruser list to stdout.
*/
dumplist()
{
	struct ruser *cur;

	if (rlist == NULL)
		printf("No users in rlist\n");
	else
		for (cur = rlist; cur != NULL; cur = cur->next)
			printf("%s@%s watching %s\n", cur->watcher,
				inet_ntoa(cur->host.sin_addr.s_addr),
				cur->watchee);
}
#endif

reap()
{
	wait(0);
}

main()
{
	int	fd;
	fd_set	myread, other;
	struct	timeval timer;

	getsocks();

	setreuid(0, 0);
#ifndef DEBUG
	if (fork())
		exit(0);
	fd = open("/dev/tty", O_RDWR);
	ioctl(fd, TIOCNOTTY, 0);
	close(fd);
	close(0);
	close(1);
	close(2);
#else
	signal(SIGQUIT, dumplist);
#endif

	signal(SIGCHLD, reap);

	FD_ZERO(&other);

	while (1)
	{
		timerclear(&timer);
		timer.tv_sec = ROLLCALL;
		bcopy(&readset, &myread, sizeof(fd_set));

		if (! select(FD_SETSIZE, &myread, &other, &other, &timer))
			rollcall();
		else
		{
			if (FD_ISSET(udp_fd, &myread))
			{
				udp();
				FD_CLR(udp_fd, &myread);
			}
			if (FD_ISSET(tcp_fd, &myread))
			{
				tcp();
				FD_CLR(tcp_fd, &myread);
			}
/* we have to do this in a loop; ffs() won't work for FD_SETSIZE > 32 */
			for (fd = 0; fd < FD_SETSIZE; fd++)
				if (FD_ISSET(fd, &myread))
					tcpc(fd);
		}
	}
}
