/*
** BIFF
**
** Monitor remote mailboxes.
*/

#include <stdio.h>
#include <netdb.h>
#include <ctype.h>
#include <strings.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include "config.h"

int tcp_port;

main(argc, argv)
char **argv;
{
	char code, *tty, *ttyname();
	int fd, port;
	struct sockaddr_in in;
	struct servent *sent;
	struct stat st;

	if (! isatty(1))
	{
		fprintf(stderr, "Can't biff to a pipe/socket\n");
		exit(-1);
	}

	tty = ttyname(1);
	if (stat(tty, &st))
	{
		perror("statting tty");
		exit(-1);
	}

	if (argc == 1)
	{
		printf("is %c\n", st.st_mode & S_IEXEC ? 'y' : 'n');
		exit(0);
	}

/* you can't biff when su-ed to another account, unless it's to root */
	if (geteuid() && geteuid() != st.st_uid)
	{
		fprintf(stderr, "%s: Not owner\n", tty);
		exit(-1);
	}

	switch(argv[1][0])
	{
		case 'y': case 'Y':
			st.st_mode |= S_IEXEC;
			chmod(tty, st.st_mode);
		case 'a': case 'A':
			if (argc > 2)
				addusers(argc-2, argv+2);
			break;
		case 'n': case 'N':
			st.st_mode &= ~S_IEXEC;
			chmod(tty, st.st_mode);
		case 'd': case 'D':
			if (argc > 2)
				delusers(argc-2, argv+2);
			break;
		default:
			fprintf(stderr,
				"Usage: %s [y|n|a|d [[user][@machine]...]]\n",
					argv[0]);
			exit(-1);
	}
}

/*
** Sort an argument list by machine name.  This makes the load on remote
** comsats (not to mention the network) a bit lighter, since we can easily
** service all the requests for each host individually.
*/
sortargs(num, vec)
int num;
char **vec;
{
	int mcomp();

	qsort(vec, num, sizeof(char *), mcomp);
}

/*
** Compare the hostnames of two user@host entries.
*/
mcomp(a1, a2)
char *a1, *a2;
{
	a1 = index(a1, '@');
	a2 = index(a2, '@');

	if (a1 == NULL)
		if (a2 == NULL)
			return 0;
		else
			return -1;
	if (a2 == NULL)
		return 1;

	return(strcmp(a1, a2));
}

init_tcp()
{
	struct servent *sent;

	if ((sent = getservbyname("rbiff", "tcp")) != NULL)
		tcp_port = sent->s_port;
	else
		tcp_port = htons(TCP_PORT);
}

/*
** Look up a host address, or translate numeric (a.b.c.d) notation to a
** valid inet address.  Puts the host address in *buf.  Returns 0 if
** the host wasn't found.
*/
haddr(host, buf)
char *host;
unsigned long *buf;
{
	struct hostent *hent;

	if (isdigit(host[0]))
	{
		*buf = inet_addr(host);
		return 1;
	}
	hent = gethostbyname(host);
	if (hent == NULL)
		return 0;
	bcopy(hent->h_addr, buf, hent->h_length);
	return 1;
}

int sockfd;

/*
** Initiate a connection with a host.  Returns 0 on failure.
*/
getcon(hostname)
char *hostname;
{
	char	buf[10];
	struct	sockaddr_in in;
	int	port;

	port = 1023;
	sockfd = rresvport(&port);
	if (sockfd < 0)
	{
		perror("rresvport");
		return 0;
	}

	bzero(&in, sizeof(in));
	if (! haddr(hostname, &in.sin_addr.s_addr))
	{
		fprintf("%s: Host unknown\n", hostname);
		return 0;
	}
	in.sin_port = tcp_port;
	in.sin_family = AF_INET;

	if (connect(sockfd, &in, sizeof(in)))
	{
		perror(hostname);
		return 0;
	}

	sprintf(buf, "P%d", tcp_port);
	write(sockfd, buf, strlen(buf)+1);
	read(sockfd, buf, 1);
	return 1;
}

/*
** Extract a hostname from a user@host string.  If there's no @host, use
** localhost.
*/
char *hname(str)
char *str;
{
	char *at;

	at = index(str, '@');
	if (at == NULL)
		return("localhost");
	else
		return(at+1);
}

char *addcodes[] = {
"",
"Out of memory",
"Already monitoring",
"Permission denied"
};

addusers(num, vec)
int num;
char **vec;
{
	int	i, npos;
	char	*curhost, *tmp, buf[130], *getlogin(), code;

	sortargs(num, vec);
	curhost = vec[0];
	i = 0;

	init_tcp();

	sprintf(buf, "W%s", getlogin());
	npos = strlen(buf)+1;	/* position to place second name */
	
	while(i < num) {
		if (! getcon(hname(vec[i])))
		{
			fprintf(stderr, "Couldn't register %s\n", vec[i]);
			while (! mcomp(curhost, vec[i]))
				if (++i == num)
					break;
			if (i < num)
				curhost = vec[i];
		}
		else
		{
			while (! mcomp(curhost, vec[i]))
			{
				if (vec[i][0] == '@')
					strcpy(buf+npos, getlogin());
				else
				{
					strncpy(buf+npos, vec[i], 64);
					buf[npos+64] = '\0';
					if ((tmp=index(buf+npos,'@')) != NULL)
						*tmp = '\0';
				}
				write(sockfd, buf, npos+strlen(buf+npos)+1);
				read(sockfd, &code, 1);
				if (code)
					fprintf(stderr, "%s: %s\n", vec[i],
						addcodes[code]);
				if (++i == num)
					break;
			}
			if (i < num)
				curhost = vec[i];
			close(sockfd);
		}
	}
}

/*
** Delete users from watch lists.  This looks mostly identical to
** addusers(), but is just different enough to warrant its own
** routine.
*/
delusers(num, vec)
int num;
char **vec;
{
	int	i, npos;
	char	*curhost, *tmp, buf[130], *getlogin(), code;

	sortargs(num, vec);
	curhost = vec[0];
	i = 0;

	init_tcp();

	sprintf(buf, "D%s", getlogin());
	npos = strlen(buf)+1;	/* position to place second name */
	
	while(i < num) {
		if (! getcon(hname(vec[i])))
		{
			fprintf(stderr, "Couldn't delete %s\n", vec[i]);
			while (! mcomp(curhost, vec[i]))
				if (++i == num)
					break;
			if (i < num)
				curhost = vec[i];
		}
		else
		{
			while (! mcomp(curhost, vec[i]))
			{
				if (vec[i][0] == '@')
					buf[npos] = '\0';
				else
				{
					strncpy(buf+npos, vec[i], 64);
					buf[npos+64] = '\0';
					if ((tmp=index(buf+npos,'@')) != NULL)
						*tmp = '\0';
				}
				write(sockfd, buf, npos+strlen(buf+npos)+1);
				read(sockfd, &code, 1);
				if (!code)
					printf("error deleting %s\n", vec[i]);
				if (++i == num)
					break;
			}
			if (i < num)
				curhost = vec[i];
			close(sockfd);
		}
	}
}

