/*++
/* NAME
/*      sendwork 3
/* SUMMARY
/*      send local work to remote system
/* PROJECT
/*      pc-mail
/* PACKAGE
/*      cico
/* SYNOPSIS
/*      #include "work.h"
/*
/*      void sendwork(wrk)
/*      work *wrk;
/* DESCRIPTION
/*      sendwork converts names and contents of local work files,
/*	sends them to the remote system and deletes the files after 
/*	successfull transfer.
/*
/*	In particular, it generates appropriate "From " lines at the
/*	beginning of an outgoing mail message.
/* SEE ALSO
/*      scanwork(3)     locates work in the spool directory
/* DIAGNOSTICS
/*	sendwork() returns via longjmp(systrap,errorcode) in case
/*	of unrecoverable problems.
/*
/*	The error codes are: E_CONFUSED (unexpected work type),
/*	E_LOST (timed out), E_READERR (file read error).
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*      Thu Mar 26 11:32:23 GMT+1:00 1987
/* LAST MODIFICATION
/*	90/01/22 13:02:35
/* VERSION/RELEASE
/*	2.1
/*--*/

#include <stdio.h>
#include <time.h>

#include "defs.h"
#include "work.h"
#include "logs.h"
#include "status.h"
#include "params.h"
#include "comm.h"

extern struct tm *localtime();		/* std C library */

 /*
  * A pc-mail system can connect to the UNIX net in at least two modes:
  * 
  * 1. As a real UUCP node, with it's own node name. This node name will have to
  * appear in the "From " lines of outgoing mail. A consequence is that the
  * pc mail node name should be known in mailer routing tables. Obviously
  * this implies some administrative work when a pc mail node is added to the
  * net or taken out of operation.  This mode has not been tested by the
  * author.
  * 
  * 2. As an ordinary user. The program lets the UNIX host believe that mail
  * messages come from an ordinary user. Recipients of mail will not be able
  * to see that the mail came from the pc. Only the UNIX host knows it should
  * forward mail for unixhost!xyz to the pc-mail node. This approach has the
  * advantage that adding/deleting pc-mail nodes is simpler.
  */

#ifdef	UUCP_NODE			/* case 1 */
#   define UUSER	"root"		/* use current user's name */
#   define UHOST	LOGIN_NAME	/* use pc host name */
#else					/* case 2 */
#   define UUSER	LOGIN_NAME	/* use remote login name */
#   define UHOST	rmthost		/* use remote host name */
#endif

/* sendwork - adapt file contents for remote host */

public  sendwork(wrk)
work   *wrk;
{
    long    secs;
    char    buf[MAXLINE];		/* recipient addresses */

    switch (wrk->type) {

	/*
	 * Local D files contain the mail message. Except for the addition of
	 * a UUCP-style "From " line (with originator/date/system), D files
	 * are sent without modification.
	 */

    case 'd':
    case 'D':
	secs = time((long *) 0);
	say(strcons("From %s %.24s remote from %s\n",
		    UUSER, asctime(localtime(&secs)), UHOST));
	send_file(wrk->fp);
	break;

	/*
	 * The first line of local X files contains the destination address.
         * Real UUCP expects something entirely different.
         *
	 * We make up some extra info to make the remote uuxqt program happy.
	 */

    case 'x':
    case 'X':
	say(strcons("U %s %s\n", UUSER, UHOST));/* U user system */
	say(strcons("F %s\n",
		    rmtname('D', wrk->seqno)));	/* F D.rmtsysGnumber */
	say(strcons("I %s\n",
		    rmtname('D', wrk->seqno)));	/* I D.rmtsysGnumber */
	say("C rmail ");			/* C rmail */
	(void) fgets(buf, sizeof(buf), wrk->fp);/* read destinations */
	say(buf);				/* send destinations */
	say("");				/* send EOF */
	break;

    default:
	trap(E_CONFUSED, "INTERNAL ERROR (unexpected work type: %c)", wrk->type);
    }
}

/* say - write string to host */

hidden  say(str)
char   *str;
{
    if (CALL(Write) (ttfd, str, strlen(str)) < 0)
	trap(E_LOST, "FAILED (link lost)");
}

/* send_file - do the nitty-gritty of file transfer; traps on all errors */

hidden  send_file(fp)
register FILE *fp;
{
    register int nread;
    register int nwrite = 0;
    char    buf[BUFSIZ];
    register int rerror;

    while ((nread = fread(buf, sizeof(*buf), sizeof(buf), fp)) > 0
	   && (nwrite = CALL(Write) (ttfd, buf, nread)) == nread)
	 /* void */ ;
    rerror = ferror(fp);
    fclose(fp);

    if (rerror) {
	trap(E_READERR, "FILE READ ERROR (%s)", sys_errlist[errno]);
	/* NOTREACHED */
    } else if (nwrite < 0 || CALL(Write) (ttfd, buf, 0) != 0) {
	trap(E_LOST, "FAILED (link lost)");
	/* NOTREACHED */
    }
}
