/*++
/* NAME
/*      mbox 3
/* SUMMARY
/*      display and manipulate one non-work message
/* PROJECT
/*      pc-mail
/* PACKAGE
/*      mail
/* SYNOPSIS
/*      int mbox(meta,msgid)
/*	int meta;
/*	unsigned msgid;
/* DESCRIPTION
/*      mbox() is invoked when the user has selected a non-work mail message.
/*	It instructs the pager to display the selected mail message. 
/*      The user has the usual options for manipulating the message
/*	being displayed.
/*
/*	The meta parameter indicates the message type,
/*	and msgid is the numerical message id. If the message file is being
/*	read for the first time (meta == NEW_META), it will be marked as read
/*	by renaming the meta file.
/*
/*	Message header lines can be suppressed selectively (see setup).
/* FILES
/*      $MAILDIR/?nnnnn, message and meta files
/* SEE ALSO
/*      pager(3), pager(5), kbdinp(3)
/* DIAGNOSTICS
/*      If a selected mail message could not be found an error message
/*      is displayed instead.
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*      Sun Apr  5 13:01:12 GMT+1:00 1987
/* LAST MODIFICATION
/*	90/01/22 13:02:11
/* VERSION/RELEASE
/*	2.1
/*--*/

#include <stdio.h>
#include <ctype.h>

#include "defs.h"
#include "path.h"
#include "pager.h"
#include "screen.h"
#include "mail.h"
#include "ascf.h"
#include "params.h"
#include "ms_parse.h"

 /*
  * Storage for header lines to be ignored. Both the strings with header
  * names, and the pointers to these strings, are kept together.
  */

struct ignore {
    char    strs[BUFSIZ];		/* null-terminated strings */
    char   *ptrs[BUFSIZ / 2];		/* null-terminated list of pointers */
};

hidden void ign_init();			/* forward declarations */
hidden int ign_header();
hidden int mbox_filter();

hidden File *letter = 0;		/* pager file */

/* show_letter - display selected mail message */

hidden int show_letter()
{
    set_pager(letter);				/* select message display */
    ds_pager();					/* put it on the screen */
    return (0);					/* say screen is ok */
}

/* mbox - user has selected a message file */

public int mbox(meta, id)
int     meta;
unsigned id;
{
    static Screen screen[] = {
	'C',	"Close",0,		initscreen,
	'D',	"Delete",delete,	delcurr,
	'M',	"Mail",	mailfile,	"Mail a copy of this message",
	'P',	"Print",print,		printcurr,
	'R',	"Reply",reply,		"Create reply to sender",
	'S',	"Save",	save,		"Save this message to ordinary file",
	'W',	"Work",	makework,	"Save this message to work file",
	'|',	"|",	filter,		"Filter this message through command",
	PGUP,	PgUp,	pu_pager,	pageup,
	PGDN,	PgDn,	pd_pager,	pagedn,
	UP,	"Up",	up_pager,	csrup,
	DOWN,	"Down",	dn_pager,	csrdn,
	0,	0,	show_letter,	"(Reading a mail message)",
    };
    char   *seen;

    /* 
     * Mail being read for the first time is renamed to reflect the
     * status change.
     */

    if (mbox_filter(letter = open_pager(), message)) {
	mesg_pager(letter, m_msgread);		/* no file or read error */
    } else if (meta != NEW_META) {		/* unread message? */
	 /* void */ ;				/* no */
    } else if (rename(comment, seen = old_meta(id)) == 0) {
	strcpy(comment, seen);			/* mark message as read */
	junk_desk();				/* say desk-top outdated */
    }
    kbdinp(screen);				/* look at the screen */
    close_pager(letter), letter = 0;		/* destroy the display */
    return (S_REDRAW);				/* force screen redrawing */
}

/* mbox_filter - suppress some message-header lines */

hidden int mbox_filter(pp, path)
File   *pp;
char   *path;
{
#if (defined(lint) && defined(iAPX286))
    static
#endif
    struct ignore ignore;
    FILE   *fp;

    if ((fp = ascopen(path, "r")) == 0) {
	return (1);
    } else {
	char    buf[BUFSIZ];
	int     ig_flag = 0;
	int     ret;
	int     context = MS_UUCP;

	ign_init(&ignore);			/* initialize filter */

	/*
	 * The header-line suppression algorithm is effective for RFC822-like
	 * headers lines only. Its main use is to get rid of the "Received:"
	 * lines that frequently show up in non-local mail.
	 */

	while (ascgets(buf, sizeof(buf), fp)) {
	    switch (context = ms_parse(context, buf)) {
	    case MS_UUCP:
		app_pager(pp, buf);
		break;
	    case MS_HEADER:
		if ((ig_flag = (ign_header(buf, &ignore))) == 0)
		    app_pager(pp, buf);
		break;
	    case MS_CONT:
		if (ig_flag == 0)
		    app_pager(pp, buf);
		break;
	    case MS_BODY:
		app_pager(pp, buf);
		break;
	    }
	}
	ret = ferror(fp);
	ascclose(fp);
	return (ret);
    }
}

/* ign_init - setup header lines to be ignored */

hidden void ign_init(ig)
struct ignore *ig;
{
    Info   *ip = getparams() + P_IGNORE;/* what to ignore */
    char   *sp = ":, ";			/* string separators */
    char  **lp = ig->ptrs;		/* separated strings */

    /*
     * This function takes, from the pc-mail setup file, a (blank or
     * comma)-separated list with names of mail headers to be ignored when a
     * message is displayed. The list of names is broken up into separate
     * strings. The result, a null-terminated list of string pointers, is
     * stored in the ig argument. We use strtok() for string splitting. Since
     * that function destroys its input, and since the user may change the
     * setup at any time, we keep in the ig argument a copy of the relevant
     * setup information.
     */

    if (ip->strval == 0) {			/* nothing to ignore */
	*lp = 0;
    } else {					/* copy, then split */
	(void) strncpy(ig->strs, ip->strval, sizeof(ig->strs));
	for (*lp = strtok(ig->strs, sp); *lp; *lp = strtok((char *) 0, sp))
	    lp++;
    }
}

/* ign_header - do we ignore this header line */

hidden int ign_header(buf, ig)
register char *buf;
struct ignore *ig;
{
    register int l;			/* header name length */
    register char **list;		/* ptr to ignored header names */

    /* Make sure that the header name is followed by a colon */

    for (list = ig->ptrs; *list; list++) {
	if (buf[l = strlen(*list)] == ':' && istrncmp(buf, *list, l) == 0)
	    return (1);
    }
    return (0);
}
