/*++
/* NAME
/*      errdisp 3
/* SUMMARY
/*      produce an error message display
/* PROJECT
/*      pc-mail
/* PACKAGE
/*      mail
/* SYNOPSIS
/*      int errdisp(errcode);
/*      int errcode;
/* DESCRIPTION
/*      errdisp() produces an error display in the middle screen
/*      window. The error code must be one defined in status.h.
/*	Otherwise, an "unknown error" message is displayed.
/*	errdisp() gives the user a chance to read it, before resuming
/*      the program. errdisp() returns its argument, or the
/*	value E_UNKNOWN if an unknown error code was specified.
/* FUNCTIONS AND MACROS
/*      kbdinp(), open_pager(), app_pager(), ds_pager(), close_pager()
/* BUGS
/*      Cannot force the user to really stop. It just recommends
/*      to do so.
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*      Thu Apr  9 21:19:05 GMT+1:00 1987
/* LAST MODIFICATION
/*	90/01/22 13:01:36
/* VERSION/RELEASE
/*	2.1
/*--*/

#include "defs.h"
#include "screen.h"
#include "pager.h"
#include "status.h"
#include "window.h"

/* forward declarations */

hidden int ds_error();

/* A list of error mesages */

/* when the spool directory is not found */

hidden char *e_nospool[] = {
    "",
    "The mail program cannot access essential data files.",
    "",
    "Please check whether the spool directory is present",
    "and whether the environment variable MAILDIR has been",
    "set correctly.",
    0,
};

/* when we cannot open as many files as we need */

hidden char *e_fileno[] = {
    "",
#ifdef  MSDOS
    "Check your CONFIG.SYS file. It should have a line",
    "stating (for example)",
    "    FILES = 20 .",
    "While you're at it, also make sure that file also",
    "has a line (for example)",
    "    DEVICE = ANSI.SYS .",
    "If your system does not have such a file create",
    "one with an editor and place the file in the root",
    "directory.",
    "Reboot after making the changes and try again.",
#endif
#ifdef  unix
    "Your UNIX is crippled (not enough file descriptors",
    "per process). Rebuild the system or buy a better one.",
#endif
    0,
};

/* when a program is not found */

hidden char *e_noprog[] = {
    "",
    "The mail program cannot execute an essential program file.",
    "",
    "Please make sure that the environment variable PATH has",
    "been set to include the directories with the SMAIL, NMAIL,",
#ifdef	DAEMON
    "and EDITOR programs.",
#else
    "CICO and EDITOR programs.",
#endif
    "Also, make sure that the environment variable EDITOR has",
    "been set to the editor of your preference, and that the",
    "machine has enough free memory",
    0,
};

/* when a read error ocurred, or a file could not be opened */

hidden char *e_readerr[] = {
    "",
    "The mail program cannot read a data file.",
    "",
    "Please check your disk. This may be a serious problem.",
    0,
};

/* when a write error occurred */

hidden char *e_writerr[] = {
    "",
    "The mail program cannot write a data file.",
    "",
    "Please make sure that the disk is not write protected",
    "and that there is sufficient free space.",
    0,
};

/* when the program is confused */

hidden char *e_confused[] = {
    "",
    "The mail program has detected a serious internal problem.",
    "",
    "You are requested to leave the mail system as it is and",
    "call an expert.",
    0,
};

/* an unknown error (perhaps the editor returned a nonzero status) */

hidden char *e_unknown[] = {
    "",
    "The mail program has detected a unknown error. This may",
    "not be a serious problem.",
    0,
};

/* when the UNIX host does not ask for login/password */

hidden char *e_noline[] = {
    "",
    "The mail program is not able to exchange mail with the",
    "network.",
    "",
    "Please check the password, or try again later.",
    0,
};

/* when the UNIX host does not send initial protocol messages */

hidden char *e_noresp[] = {
    "",
    "The mail program is not able to exchange mail with the",
    "network. Probably the password was incorrect.",
    0,
};

/* when the UNIX host does not want to talk to us */

hidden char *e_reject[] = {
    "",
    "The mail program is not able to exchange mail with the",
    "network. Please contact the system administrator who is",
    "responsible for the nearest UNIX system.",
    0,
};

/* when the link is lost (timeout,...) */

hidden char *e_lost[] = {
    "",
    "The mail program has lost contact with the network.",
    "",
    "We suggest that you try again at a later time.",
    0,
};

/* the resources are exhausted */

hidden char *e_sysfail[] = {
    "",
    "The mail program cannot proceed due to the fact",
    "that the computer system is overloaded.",
    "",
    "Consult your local systems manager.",
    0,
};

/* bad setup parameter */

hidden char *e_badsetup[] = {
    "",
    "The mail program cannot communicate with the network",
    "due to missing or invalid data in the setup.",
    "",
    "Check the Setup command in the main menu.",
    0,
};

/* when a file could not be removed */

hidden char *e_unlink[] = {
    "",
    "The mail program cannot remove a data file.",
    "",
    "Please make sure that your disk is not write protected",
    0,
};

/* when a file could not be printed */

hidden char *e_printerr[] = {
    "",
    "The mail programm cannot write to the printer",
    "",
    "Please check your default printer device",
    0,
};

hidden char *e_nouser[] = {
    "",
    "An unknown local user name was specified",
    "",
    "Please check the list of user names",
    0,
};

hidden char *e_ovalias[] = {
    "",
    "There probably is an error in your alias data base,",
    "such that an alias is defined in terms of itself.",
    "",
    "Please check your alias data base, and then try to",
    "send this message again.",
    0,
};

hidden char *e_toolong[] = {
    "",
    "The list of recipients is too long.",
    "",
    "There is no easy solution to this problem; you might",
    "ask your UNIX system administrator to set up a mailing",
    "list, and use the name of that mailing list instead.",
    0,
};

/* The folowing data structure links an error code to an error message */

typedef struct {
    short code;				/* error return value */
    char **msg;				/* associated message */
} msgmap;

hidden msgmap errmap[] = {		/* lookup table for error codes */
    E_UNKNOWN,	e_unknown,		/* don't remove this one! */
    E_SYSFAIL,	e_sysfail,
    E_NOPROG,	e_noprog,		/* and error messages */
    E_NOSPOOL,	e_nospool,
    E_READERR,	e_readerr,
    E_WRITERR,	e_writerr,
    E_CONFUSED,	e_confused,
    E_FILENO,	e_fileno,
    E_NOLINE,	e_noline,
    E_NORESP,	e_noresp,
    E_REJECT,	e_reject,
    E_LOST,	e_lost,
    E_BADSETUP,	e_badsetup,
    E_UNLINK,	e_unlink,
    E_PRINTERR,	e_printerr,
    E_NOUSER,	e_nouser,
    E_OVALIAS,	e_ovalias,
    E_TOOLONG,	e_toolong,
    0,		0,			/* terminator */
};

/* errdisp - set up error pager file, return error code */

public int errdisp(code)
int code;
{
    static Screen screen[] = {
	ENTER,	"Enter",        0,      "Resume the mail program",
	0,	0,              ds_error,"Press ENTER to continue",
    };
    register msgmap *mp;

    /* linear table search; this code is not used heavily */

    for (mp = errmap; mp->code && mp->code != code; mp++)
	/* void */ ;				/* look it up in table */
    if (mp->code == 0) {
	return(errdisp(E_UNKNOWN));		/* unknown error code */
    } else {
	File *pp = open_pager();		/* open pager file */
	mesg_pager(pp,mp->msg);			/* copy to pager file */
	beep();					/* red alert! red alert! */
	kbdinp(screen);				/* give a chance to read it */
	close_pager(pp);			/* forget screen display */
	return(code);				/* return error code */
    }
}

/* ds_error - display error window */

hidden int ds_error()
{
    ds_pager();					/* use "current" pager file */
    return(0);					/* say screen up-to-date */
}
