/*++
/* NAME
/*	email 3
/* SUMMARY
/*	manipulate one message in preparation
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	mail
/* SYNOPSIS
/*	#include "mail.h"
/*
/*	int work()
/*
/*	int work_disp(def_addr)
/*	char *def_addr;
/* DESCRIPTION
/*      The functions in this module are responsible for manipulations 
/*	on mail messages in preparation.
/*
/*	work() should be invoked when the user has selected an existing
/*	message in preparation. It does some initializations and invokes
/*	the work_disp() function.
/*
/*      work_disp() allows the user to specify the disposition of a
/*	mail message in preparation. It should be used after the user has
/*	created a message, or after the user has selected a message in 
/*	preparation.
/*
/*	The message file is displayed on the screen and user the 
/*	can choose to print, mail, edit or delete etc. the message.
/*
/*	The def_addr argument contains a default mail destination: for example,
/*	an address extracted from a message being replied to. It should be an
/*	e-mail address or an empty string.
/*
/*	The code in this module is a little tricky, to avoid "orphan" work
/*	files (message file without a metafile).
/* COMMANDS
/*	The program specified in the EDITOR environment variable,
/*	or a system-dependent default.
/* FILES
/*	mail.msg, file being edited in the current directory
/*      $MAILDIR/ennnnn, message file (body)
/*	$MAILDIR/cnnnnn, meta file (summary)
/*	$MAILDIR/header, template mail header file
/*	$MAILDIR/trailer, template signature file
/* SEE ALSO
/*      pager(3), pager(5), kbdinp(3), edit(3)
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Tue May 12 15:35:20 GMT+1:00 1987
/* LAST MODIFICATION
/*	90/01/22 13:01:35
/* VERSION/RELEASE
/*	2.1
/*--*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <errno.h>
#include "defs.h"
#include "path.h"
#include "pager.h"
#include "screen.h"
#include "mail.h"
#include "status.h"

/* forward declarations */

hidden void junk_work();
hidden int edit_work();
hidden int show_work();
hidden int hold_work();
hidden int send_work();
hidden int queue_work();

public char address[MAXLINE];		/* default destination */
hidden File *workfile = 0;		/* pager file */

/* work - user selected message in preparation */

public int work()
{
    return (work_disp(""));
}

/* work_disp - ask disposition of a message in preparation */

public int work_disp(def_addr)
char *def_addr;
{
    static Screen screen[] = {
	'C',	"Close",hold_work,"Send message later, return to message-selection menu",
#ifdef	ATTACH
	'A',	"Attach",attach", "Attach file to message",
#endif
	'D',	"Delete",delete,  delcurr,
	'E',	"Edit",	edit_work,"Edit this message",
	'M',	"Mail",	send_work,"Send this message to destination",
	'P',	"Print",print,	  printcurr,
	PGUP,	PgUp,	pu_pager, pageup,
	PGDN,	PgDn,	pd_pager, pagedn,
	UP,	"Up",	up_pager, csrup,
	DOWN,	"Down",	dn_pager, csrdn,
	0,	0,	show_work,
	"(Reading a message in preparation)",
    };
    struct stat s;

    strcpy(address, def_addr);			/* set up default address */
    kbdinp(screen);				/* ask disposition */
    junk_work();				/* destroy mail pager file */
    return (S_REDRAW);				/* say screen was changed */
}

/* show_work - show message in preparation or error message in middle window */

hidden int show_work()
{
    if (workfile) {				/* check pager file exists */
	set_pager(workfile);			/* select existing display */
    } else if (rd_pager(workfile = open_pager(), message)) {
	mesg_pager(workfile, m_msgread);	/* cannot display message */
    }
    ds_pager();					/* (re)draw display */
    return (0);					/* screen is up-to-date */
}

/* junk_work - destroy message in preparation display */

hidden void junk_work()
{
    if (workfile) {				/* no-op if no display */
	close_pager(workfile);			/* release memory */
	workfile = 0;				/* say it is gone */
    }
}

/* edit_work - edit a message in preparation */

hidden int edit_work()
{
    register int stat;

    if (stat = edit(message, MAILFILE))		/* try to edit the message */
	errdisp(stat);				/* edit() had a problem */
    junk_work();				/* force new message display */
    return (S_REDRAW);				/* say screen has changed */
}

hidden int label_work();

/* hold_work - stop editing but do not yet mail a message in preparation */

hidden int hold_work()
{
    static Screen screen[] = {
	STRING,	0, label_work, int_error,
	0,	0, 0,
	getsummary,
    };
    struct stat s;

    /*
     * The user does not yet want to send the message off. The purpose of the
     * following code is to ask for a one-line summary, but only if such a
     * comment does not yet exist. The summary is used to identify the
     * message in preparation in the message- selection display.
     */

    if (stat(message, &s) || !stat(comment, &s)) {
	return (S_BREAK);			/* we are done here */
    } else {
	return (kbdinp(screen) | S_REDRAW);	/* ask for a summary */
    }
}

/* label_work - save summary line to meta file */

hidden  label_work(string)
char   *string;
{
    register int stat;

    if (stat = metafile(comment, string, (char *) 0)) {
	errdisp(stat);				/* oops, notify the user */
	return (S_REDRAW);			/* say screen has changed */
    } else {
	chmod(comment, 0444);			/* make comments read-only */
	junk_desk();				/* say mail box has changed */
	return (S_BREAK);			/* say no more work */
    }
}

/* send_work - user wants to send message in preparation, ask for destination */

hidden int send_work()
{
    static Screen screen[] = {
	EDIT,	0,	queue_work,	address,
	0,	0,	when,
	"Press ESC to cancel. Send message to:",
    };

    return (kbdinp(screen) | S_REDRAW);
}

/* queue_work - spool mail, delete message in preparation and meta file */

hidden int queue_work(to)
char   *to;
{
    register int stat;

    if (stat = submit(message, to)) {
	errdisp(stat);				/* cannot queue message */
	return (S_REDRAW);			/* say screen has changed */
    } else {
	return (unspool() | S_BREAK);		/* remove work and meta file */
    }
}
