/*++
/* NAME
/*	cmail 1
/* SUMMARY
/*	report if there are unread messages
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	cmail
/* SYNOPSIS
/*	cmail [-p password]
/* DESCRIPTION
/*	cmail reports if there are unread mail messages in the
/*	pc-mail spool directory.
/*
/*	If the -p option is present, cmail first invokes the cico
/*	program to contact the mail server host.
/*
/*	Typically cmail is run immediately after system startup,
/*	while you are getting your first cup of coffee.
/*
/*	The program returns a nonzero exit status if it could not
/*	find any mail.
/* COMMANDS
/*	cico	file transfer program
/*	nmail	processes new mail
/* FILES
/*	Various files in the spool directory
/*
/*	LOGFILE system status messages
/*	n<seqno> mail messages
/*	h<seqno> header line of new mail
/*	o<seqno> header line of old mail
/* DIAGNOSTICS
/*	Error messages in case the environment is not properly set up.
/* BUGS
/*	Invites people to put their mail password into the autoexec file.
/* AUTHOR(S)
/*	W.Z. Venema
/*	Eindhoven University of Technology
/*	Department of Mathematics and Computer Science
/*	Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Sun Apr  3 19:34:57 MET 1988
/* LAST MODIFICATION
/*	90/01/22 13:01:19
/* VERSION/RELEASE
/*	2.1
/*--*/

#include <stdio.h>
#include <signal.h>
#include <varargs.h>

#include "defs.h"
#include "ndir.h"
#include "path.h"
#include "status.h"

hidden void parse_args();		/* forward declarations */
hidden void usage();
hidden void error();
hidden int newmail();

hidden char *password = 0;		/* set by the -p option */

public char *progname = "cmail";	/* for diagnostics */

main(argc, argv)
int     argc;
char  **argv;
{
    signal(SIGINT, SIG_IGN);			/* disable ctrl-c */
    parse_args(argc, argv);			/* parse command args */
    if (pathinit())				/* check path info */
	error("cmail: bad MAILDIR environment variable");
    if (password && *password &&
	invokelp(CICO, "-p", password, (char *) 0) == E_NOPROG)
	error("cmail: cannot execute the CICO program");
    if (invokelp(NMAIL, (char *) 0) == E_NOPROG)
	error("cmail: cannot execute the NMAIL program");
    exit(newmail() == 0);			/* look for new mail */
}

/* parse_args - process command-line arguments */

hidden void parse_args(argc, argv)
int     argc;
char  **argv;
{
    while (--argc && *++argv && **argv == '-') {/* process options */
	switch (*++*argv) {
	case 'p':				/* call cico first */
	    if (--argc == 0)
		usage("missing password");
	    password = *++argv;
	    break;
	default:				/* unknown option */
	    usage("invalid option: -%s", *argv);
	    break;
	}
    }

    /* check for extraneous arguments */

    if (argc > 0)
	usage("unexpected argument: %s", *argv);
}

/* scan for new unread mail */

hidden int newmail()
{
    DIR    *dp;
    struct direct *de;
    FILE   *fp;
    char    from[MAXLINE];
    int     msgcount = 0;
    unsigned msgno;

    /*
     * Scan the spool directory for unread messages and extract the
     * originator address from the corresponding meta file.
     */

    if ((dp = opendir(maildir)) == 0) {
	error("cmail: cannot access the mail directory");
    } else {
	while (de = readdir(dp)) {
	    if (de->d_name[0] == NEW_MESG
	    && (msgno = seqno(de->d_name))
	    && (fp = fopen(new_meta(msgno), "r")) != 0) {
		if (fgets(from, sizeof(from), fp))
		    printf("You have new mail from %s", from);
		msgcount++;
		fclose(fp);
	    }
	}
	closedir(dp);
    }
    return (msgcount);
}

hidden void error(str)
char   *str;
{
    fprintf(stderr, "%s\n", str);
    exit(1);
}

/* VARARGS */

hidden void usage(va_alist) 
va_dcl
{
    va_list ap;
    char   *fmt;

    va_start(ap);
    fmt = va_arg(ap, char *);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    fprintf(stderr, "\nusage: cmail [-p password]\n");
    exit(1);
}
