/*++
/* NAME
/*	spoolfil,metafile 3
/* SUMMARY
/*	create message file and meta file
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	general utilities
/* SYNOPSIS
/*	int spoolfil(path,meta_info,mesg_prefix,meta_prefix)
/*	char *path;
/*	char *meta_info;
/*	int mesg_prefix;
/*	int meta_prefix;
/*
/*	int metafile(path,string,...,(char *) 0)
/*	char *path;
/*	char *string;
/* DESCRIPTION
/*	spoolfil() creates a (message, meta) file pair in the spool
/*	directory.
/*
/*	"path" should be null-terminated string with the name of an existing
/*	file. The contents of that file are filtered to produce a
/*	clean ASCII file.
/*
/*	"meta-info" is a string with additional information that is written to
/*	the first line of a meta file (usually name of mail recipient, mail
/*	originator or a comment describing the contents of the message file).
/*	There should not be any embedded newline characters in this string.
/*
/*	"mesg_prefix," "meta_prefix" are prefixes that will be used for building
/*	names for files in the spool directory.
/*
/*	If the message file given to spoolfil() looks like a mail message, a
/*	line with Subject: information, extracted from the message, will be
/*	appended to the meta file.
/*
/*	metafile() creates a file, writes the successive string arguments to
/*	that file, each followed by a newline character. This function is
/*	typically used to create a file with meta information (originator,
/*	subject etc.).
/* FUNCTIONS AND MACROS
/*	ascopen(), ascget(), ascclose(), newseqno()
/* DIAGNOSTICS
/*	A nonzero return value indicates an error condition (see status.h)
/* SEE ALSO
/*	status(5)	return values
/*	ascf(3)		ASCII filter
/* BUGS
/*	Wordprocessor control codes etc will be lost when spoolfil copies
/*	a file.
/* AUTHOR(S)
/*	W.Z. Venema
/*	Eindhoven University of Technology
/*	Department of Mathematics and Computer Science
/*	Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Mon May 18 18:45:10 GMT+1:00 1987
/* LAST MODIFICATION
/*	90/01/22 13:02:40
/* VERSION/RELEASE
/*	2.1
/*--*/

#include <stdio.h>
#include <varargs.h>

#include "defs.h"
#include "path.h"
#include "ascf.h"
#include "status.h"
#include "ms_parse.h"

/* metafile - create a meta file */

/* VARARGS */

public int metafile(va_alist) 
va_dcl
{
    char   *path;
    char   *string;
    FILE   *fp;
    va_list ap;
    int     ret = 0;

    va_start(ap);
    path = va_arg(ap, char *);

    if ((fp = fopen(path, "w")) == 0) {
	ret = E_WRITERR;
    } else {
	while ((string = va_arg(ap, char *)) != 0)
	    (void) fprintf(fp, "%s\n", string);
	if (fflush(fp) || ferror(fp))
	    ret = E_WRITERR;
	(void) fclose(fp);
    }
    va_end(ap);
    return (ret);
}

/* spoolfil - make arbitrary spool file */

public int spoolfil(fname, meta, msgpfx, auxpfx)
char   *fname;
char   *meta;
int     msgpfx;
int     auxpfx;
{
    register int newid = newseqno();	/* new message id */
    register int stat;			/* some status */
    char   *msgpath;			/* new message file */
    char   *auxpath;			/* new meta file */
    char    subj[BUFSIZ];		/* subject: info */

    msgpath = mesg_file(msgpfx, newid);		/* message file name */
    auxpath = meta_file(auxpfx, newid);		/* meta file name */

    /* copy input file to spool file, try to get subject, check for errors */

    if (stat = copyfile(fname, msgpath, subj)) {/* read/write error */
	(void) unlink(msgpath);			/* delete msg file */
	return (stat);				/* notify caller */
    }
    /* create file for meta information, check for errors */

    if (stat = metafile(auxpath, meta, subj, (char *) 0)) {
	(void) unlink(msgpath);			/* delete msg file */
	(void) unlink(auxpath);			/* delete meta file */
	return (stat);				/* notify caller */
    }
    /* when nothing went wrong */

    (void) chmod(msgpath, 0444);		/* make message read-only */
    (void) chmod(auxpath, 0444);		/* make metafile read-only */
    return (0);					/* own error handling */
}

/* copyfile - copy a file, filter it, optionally extract subject: info */

public int copyfile(from, to, subj)
char   *from;
char   *to;
char   *subj;				/* may be a null pointer */
{
    register FILE *in,
           *out;			/* file pointers */
    int     ret = 0;			/* error status */

    if ((in = ascopen(from, "r")) == 0) {	/* cannot read ?? */
	return (E_READERR);
    } else if ((out = fopen(to, "w")) == 0) {	/* cannot write ?? */
	(void) ascclose(in);
	return (E_WRITERR);
    } else {
	char    buf[BUFSIZ];
	register int context = MS_UUCP;

	if (subj)
	    *subj = 0;

	while (ascgets(buf, sizeof(buf), in)) {
	    (void) fputs(buf, out);
	    (void) putc('\n', out);

	    /* extract the subject "on the fly" */

	    if (subj != 0 && *subj == 0
	    && (context = ms_parse(context, buf)) == MS_HEADER)
		(void) hscanf(buf, "Subject:", " %[^\n]", subj);
	}

	/* perform the usual error checking */

	if (ferror(in))
	    ret = E_READERR;
	else if (fflush(out) || ferror(out))
	    ret = E_WRITERR;
	(void) ascclose(in);
	(void) fclose(out);
	return (ret);
    }
}
