/*++
/* NAME
/*	params 3
/* SUMMARY
/*	communication parameter access
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	library
/* SYNOPSIS
/*	#include "params.h"
/*
/*	Info *getparams();
/* DESCRIPTION
/*	getparams() returns a pointer to a table with communications
/*	parameters. Usually communications parameters are set with the
/*	"setup" option in the main menu of the interactive mail program.
/*
/*	First getparams() attempts to read from the setup file.
/*	If that fails it creates an empty parameter table with
/*	null string pointers as parameter values.
/* FUNCTIONS AND MACROS
/*	myalloc()
/* BUGS
/*	getparams() silently ignores any information in the
/*	parameter file that it does not recognize.
/*	getparams() will read the parameter file upon each call, even
/*	if nothing has changed since the last read. Let us say that
/*	it anticipates on multi-user environments.
/* AUTHOR(S)
/*	W.Z. Venema
/*	Eindhoven University of Technology
/*	Department of Mathematics and Computer Science
/*	Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Wed Apr  8 15:39:23 GMT+1:00 1987
/* LAST MODIFICATION
/*	90/01/22 13:02:24
/* VERSION/RELEASE
/*	2.1
/*--*/

#include <stdio.h>
#include <ctype.h>

#include "defs.h"
#include "path.h"
#include "params.h"

/* Storage area for setup parameters */

hidden Info params[] = {
    /* name */	/* name length */	/* value */	/* default */
    S_IGNORE,	sizeof(S_IGNORE) -1,	0,		D_IGNORE,
#ifndef	DAEMON
    S_PORT,	sizeof(S_PORT) - 1,	0,		0,
    S_BAUD,	sizeof(S_BAUD) - 1,	0,		0,
    S_HOST,	sizeof(S_HOST) - 1,	0,		0,
    S_LOGIN,	sizeof(S_LOGIN) - 1,	0,		0,
    S_DIAL,	sizeof(S_DIAL) - 1,	0,		0,
    S_DISC,	sizeof(S_DISC) - 1,	0,		D_DISC,
#endif
    0,		0,			0,		0,
};

hidden char *hackstr();			/* forward declaration */

/* getparams - try to get info from file, else make empty table */

public Info *getparams()
{
    char    line[BUFSIZ];
    register Info *ip;
    FILE   *fp;

    /* for cleanliness, we first clear all table entries */

    for (ip = params; ip->ident; ip++) {
	if (ip->strval)
	    free(ip->strval);
	ip->strval = 0;
    }

    /* then, try to copy parameter file info to the table */

    if (fp = fopen(parm_file(), "r")) {
	while (fgets(line, sizeof(line), fp)) {
	    for (ip = params; ip->ident; ip++) {
		if (strncmp(ip->ident, line, ip->length) == 0) {
		    ip->strval = hackstr(line + ip->length);
		    break;
		}
	    }
	}
	(void) fclose(fp);
    }

    /* set defaults for undefined values */

    for (ip = params; ip->ident; ip++)
	if (ip->strval == 0 && ip->defval != 0)
	    ip->strval = hackstr(ip->defval);

    return (params);
}

/* hackstr - cut away blanks around string and make copy */

hidden char *hackstr(s)
register char *s;
{
    register char *r;
    int     len;

    /* strip leading and trailing blanks */

    while (*s && isspace(*s))
	s++;
    for (r = s + strlen(s); r > s && isspace(r[-1]); r--)
	/* void */ ;

    /*
     * s is at the terminator or first non-blank char. r is at the terminator
     * or first blank after the last non-blank char. Thus, the actual string
     * length is r-s. We add one for the terminator. We don't allocate memory
     * if the string is empty.
     */

    if (len = r - s) {
	char   *cp = strncpy(myalloc(len + 1), s, len);

	cp[len] = '\0';
	return (cp);
    } else {
	return (0);
    }
}
