/*++
/* NAME
/*	deskutil 3
/* SUMMARY
/*	utility functions
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	mail
/* SYNOPSIS
/*	#include "mail.h"
/*
/*	void patience()
/*
/*	int when()
/*
/*	int delete()
/*
/*	int unspool()
/*
/*	int print()
/*
/*	int save()
/*
/*	int filter()
/*
/*	char *tstamp(ltime)
/*	long *ltime()
/* DESCRIPTION
/*	tstamp() converts absolute time to a string. If called with a
/*	recent time argument (less than 100 days ago) the string will
/*	be of the form "Sun Apr 17 12:50", otherwise "Sun Apr 17  1988".
/*
/*      delete() gives the user another chance before a message is deleted.
/*
/*      unspool() actually deletes a message. As a side effect it destroys
/*      the current mail box display so that the next display will
/*	reflect the actual status of the spool directory.
/*	The affected message and meta file names are taken from the
/*	global "message" and "commant" string variables.
/*
/*	print() copies a pager file to the printer.
/*
/*	save() asks where the pager file should be saved.
/*
/*	filter() asks the command through which the file being shown should be
/*	piped. The original file is used.
/*
/*	when() should be called after the user has entered a mail destination
/*	address. It informs the user that messages are not sent right away, 
/*	but after selection of the Network option in the main menu. This
/*	function is a no-op on systems that use daemons for message transport.
/*
/*	patience() prints a 'one moment please' message in the middle
/*	screen window. As a side effect, the current pager file is set
/*	to none.
/* FILES
/*      mail header files in the spool directory
/* SEE ALSO
/*      pager(3), pager(5), kbdinp(3)
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Tue May 12 15:35:20 GMT+1:00 1987
/* LAST MODIFICATION
/*	90/01/22 13:01:32
/* VERSION/RELEASE
/*	2.1
/*--*/

#include <stdio.h>
#include <errno.h>
#include <time.h>

#include "defs.h"
#include "pager.h"
#include "mail.h"
#include "screen.h"
#include "status.h"

extern struct tm *localtime();		/* std C library */

hidden int save_util();
hidden int pipe_util();

/* patience - say this will take some time */

public void patience()
{
    static char *m_wait[] = {
	"",
	"One moment please...",
	0,
    };

    register File *pp = open_pager();		/* create pager file */

    mesg_pager(pp,m_wait);			/* write pager file */
    ds_pager();					/* show on middle window */
    close_pager(pp);				/* forget pager file */
}

/* delete - user wants to delete a message; ask for confirmation */

public int delete()
{
    static Screen screen[] = {
	ESCCR,	"Enter",	unspool,int_error,
	0,	0,		0,
	"Press ESC to cancel. Confirm with ENTER",
    };

    return(kbdinp(screen)|S_REDRAW);
}

/* unspool - actually delete a message; force mail box display rebuild */

public int unspool()
{
    if (((chmod(message,0666) || unlink(message)) && errno != ENOENT)
    || ((chmod(comment,0666) || unlink(comment)) && errno != ENOENT)) {
	errdisp(E_UNLINK);			/* notify user of problem */
	return(S_REDRAW);			/* say screen has changed */
    } else {
	junk_desk();				/* say mail box has changed */
	return(S_BREAK);			/* no more work to do */
    }
}

/* print - print pager display on default printer */

public int print()
{
    return(pr_pager() ? (errdisp(E_PRINTERR),S_REDRAW) : 0);
}

/* save - ask where pager display should be copied to */

public int save()
{
    static Screen screen[] = {
	STRING,	0,              save_util,int_error,
	0,	0,              0,
	"Press ESC to cancel. Save to file:",
    };

    kbdinp(screen);			/* prompt for file name, then copy */
    return(S_REDRAW);			/* force screen repaint */
}

/* save_util - copy pager file to ordinary file */

hidden int save_util(to)
char *to;
{
    if (cp_pager(to)) {			/* if file copy failed */
	unlink(to);			/* remove that file */
	errdisp(E_WRITERR);		/* notify the user */
	return(S_BREAK|S_REDRAW);	/* redisplay, terminate caller */
    } else {
	junk_file();			/* say file display maybe outdated */
	return(S_BREAK);		/* terminate caller */
    }
}

/* filter - ask command through which the pager display should be piped */

public int filter()
{
    static Screen screen[] = {
	STRING,	0,              pipe_util,int_error,
	0,	0,              0,
	"Press ESC to cancel. Filter through command:",
    };

    kbdinp(screen);			/* prompt for file name, then copy */
    return(S_REDRAW);			/* force screen repaint */
}

/* redirect - make old refer to new, close redundant file descriptors */

hidden int redirect(old, new)
int     old;
int     new;
{
    register int ret;

    (void) close(old);
    ret = dup(new);
    (void) close(new);
    return (ret != old);
}

/* pipe_util - pipe external file through command */

hidden int pipe_util(cmd)
char   *cmd;
{
    register int fd_stdin;
    register int fd_msg;

    /*
     * Run command with standard input redirected to the file displayed, and
     * clean up. Note that the original file is used, not the contents of the
     * pager. We use system(3), so that the command can be a pipeline. This
     * code is a bit messy, since it has to run both with UNIX and MS-DOS.
     */

    if ((fd_msg = open(message, 0)) < 0) {
	errdisp(E_READERR);
    } else {
	kbdrest();				/* cooked-mode input */
	if ((fd_stdin = dup(0)) < 0) {		/* save standard input */
	    kbdinit();				/* raw-mode input */
	    errdisp(E_SYSFAIL);
	} else {
	    (void) putchar('\n');
	    if (redirect(0, fd_msg))		/* re-direct standard input */
		fatal("cannot re-direct standard input");
	    (void) system(cmd);			/* try to execute command */
	    if (redirect(0, fd_stdin))		/* restore standard input */
		fatal("cannot restore standard input");
	    kbdinit();				/* raw-mode input */
	    (void) printf(anykey);
	    (void) getkey();
	    junk_file();			/* file display outdated */
	}
    }
    return (S_BREAK | S_REDRAW);		/* terminate caller */
}

/* when - say when mail will actually be sent */

public int when()
{
#ifndef	DAEMON
    static char *msg[] = {
	"",
	"To send messages through the network, use the Network",
	"option in the main menu.",
	0,
    };
    File *pp = open_pager();			/* open new pager file */

    mesg_pager(pp,msg);				/* fill pager file */
    ds_pager();					/* display the file */
    close_pager(pp);				/* forget pager file */
#endif
    return(0);					/* don't care value */
}

/* tstamp - time format as produced by the ls(1) command */

public char *tstamp(ltime)
long   *ltime;
{
    static char buf[25];

    /*
     * Output from asctime() is of the form
     * 
     * "Sun Apr 17 13:34:35 1988\n"
     * 
     * Depending on how recent the time in question is, we supress the time or
     * year field. The weekday field is always suppressed.
     */

    (void) strcpy(buf, asctime(localtime(ltime)));
    if (time((long *) 0) - *ltime > 60L * 60L * 24L * 100L) {
	buf[24] = '\0';				/* remove the \n */
	(void) strcpy(buf + 11, buf + 19);	/* old, show year */
    } else
	buf[16] = '\0';				/* recent, show time */
    return (buf + 4);
}
