/*
 *
 * This program is Copyright 1989, by David H. Wolfskill.
 *
 * Anyone may freely copy it and use it for any purpose, as long as:
 * 1) it is not sold, and
 * 2) no one else attempts to claim authorship of the program.
 * (In all humility and honesty, I can't imagine why anyone would want to do
 * either of the above, but human nature is a truly marvelous thing....)
 *
 * [First part of above is freely adapted from "rn," by Larry Wall.... dhw]
 *
 * Disclaimer: Although the author finds the program useful, there is no
 *             guarantee that anyone else will find it useful for any given
 *             purpose.  Caveat Emptor, and all that.
 *             
 * I cannot, of course, promise to fix things that some folks may think are
 * broken; nor can I promise to implement enhancements.  However, I have a
 * strong desire to do what I can to help improve email, especially over dialup
 * 'phone lines; accordingly, it is my present intent to support this program
 * as a tool that can be helpful in some circumstances... as long as what it
 * reports isn't trusted too much....  :-)
 *
 * In all seriousness, comments & suggestions may be sent to me via
 *
 * InterNet: david@dhw68k.cts.com
 * uucp: ...{spsd,zardoz,felix}!dhw68k!david	
 *
 * and -- unless you have been explicitly informed otherwise by someone
 * in a position to know (such as me) -- I welcome suggestions for improvement.
 *
 * David H. Wolfskill	16 April, 1989
 *
 */

/*
 *	Program to list statistics, some of which may be of interest,
 *	regarding the output of "pathalias," as well as do a basic
 *	"reality check" on that output.

 *	By its nature, this program is intended to be used in conjunction
 *	with pathalias.  I really don't think you should spend any time
 *	or other resources on this program if you neither run nor have
 *	any intent to run pathalias.

 *	This version can read the pathalias output either in the
 *	original order, or either of the 2 rotations:

 *	"pathalias":	<cost>	<destination>	<path>
 *	"smail":	<destination>	<path>	<cost>
 *	unknown:	<path>	<cost>	<destination>

 *	It does *not* (presently) handle "<destination>	<cost>	<path>"
 *	(or any of its rotations); if you really think it's something
 *	worth doing, I'm willing to discuss it further, but really...!

 *	In order to make the determination of the stdin format, the
 *	first line is read, and its 3 fields are scanned to find one
 *	that ends in "%s".  If no such field is found, the program
 *	claims that the input is not pathalias output, and exits.  If
 *	one *is* found, however (the normal case, we hope!), that field
 *	is assumed to be the <path> field; the next field (to the right,
 *	wrapping around if necessary) is the <cost>, and the next is
 *	<destination>.

 *	smail likes the second order, so the stdout of this program
 *	is the same information in "smail" order (regardless of the
 *	order in which stdin is).  This program may thus be used in
 *	place of the "sed" segment of the "pathproc" pipeline that is
 *	distributed with smail.

 *	To facilitate its use in that application, the report the
 *	program generates comes out on stderr.

 *	The overall method is straightforward, brute-force:  uuname (or
 *	whatever -- see macro NBRLIST) is used to generate a list of
 *	known immediate uucp neighbors; that is run through sort.  (I
 *	do it this way because uuname is set[gu]id; I think it
 *	appropriate to minimize the need for programs to run as
 *	set[gu]id, so I use the facilities provided by a common tool,
 *	rather than require special privileges for this program.)  Then
 *	a "neighbor" (spelled "nbr" for typing constraint relief)
 *	struct is allocated for each neighbor thus found -- after one
 *	of these structs is allocated for the totals.  The nbr structs
 *	are allocated as a linked list.  (A temporary file is created
 *	to hold the output of sort; after the nbr structs are all
 *	allocated, it is unlinked and closed.)

 *	Once this housekeeping is finished, a single pass through stdin
 *	(which is expected to be in the above-documented -- possibly
 *	rotated -- pathalias output format) is made, updating the
 *	appropriate counters in the various nbr structs:  the "tots"
 *	nbr struct is always updated; the "first hop" in the path is
 *	determined, and the nbr struct that corresponds to it is also
 *	updated.

 *	A <path> field that consists entirely of "%s" is assumed to
 *	have a "first hop" of the local host.  The program will
 *	determine this by using the -l flag (ala' pathalias); if -l was
 *	not specified, routine hostname() is invoked, which either uses
 *	uname() (if UNAME is defined during compilation) or
 *	gethostname() (if it wasn't).  If hostname() is invoked and
 *	fails, the program complains bitterly and exits -- in such a
 *	case, you may wish to change the logic in hostname() or specify
 *	the -l flag, depending on how ambitious you are.

 *	For a <path> field that is more than "%s", everything from the
 *	beginning of the field up to (but not including) the first
 *	occurrence of '!' is assumed to be the name of the "first hop"
 *	-- which ought to be the name of one of the specified host's
 *	neighbors.

 *	(Note the pathological case of the pathalias output claiming
 *	the existence of a neighbor that NBRLIST doesn't know about.
 *	That actually happened in the early testing of the program, and
 *	as a result of that experience, I put in code to create a "nbr"
 *	struct on demand (as it were) in such a case, then in the
 *	output phase, spit out a warning message.)

 *	Once EOF is reached on stdin, a pair of traversals through the
 *	linked list is made; each prints a report based on the
 *	accumulated totals, and the second one also frees the elements
 *	of the list.  The second report shows a distribution of how
 *	long the paths are; its width -- how far out to count distinct
 *	pathlengths -- is controlled by the MAXDIST macro and the -d
 *	flag.

 *	Then we're done -- what could be simpler...?  :-)



 *	Notes about the flags:

 *	Since the program was first written, the number of flags
 *	("command line arguments" for picky folk) has become large;
 *	herewith is a list of them:

 *	-B <limit>      <limit> specifies an OUTPUT limit: if the
 *			cost to reach a destination is below the
 *			specified limit, the stdout will not contain
 *			that line of input data.  To make this flag
 *			ineffective, use a value of 0.  Default is
 *			BOTOUT (0).

 *	-T <limit>      <limit> specifies an OUTPUT limit: if the
 *			cost to reach a destination is above the
 *			specified limit, the stdout will not contain
 *			that line of input data.  To make this flag
 *			ineffective, use a value of 0.  Default is
 *			TOPOUT (0).

 *	-b <limit>	<limit> specifies a REPORTING limit: if the
 *			cost to reach a destination is below the
 *			specified limit, the reports (on stderr) will
 *			not reflect that line of input data.  To make
 *			this flag ineffective, use a value of 0.
 *			Default is BOTCNT (0).

 *	-t <limit>	<limit> specifies a REPORTING limit: if the
 *			cost to reach a destination is above the
 *			specified limit, the reports (on stderr) will
 *			not reflect that line of input data.  To make
 *			this flag ineffective, use a value of 0.
 *			Default is TOPCNT (0).

 *	-d <dist>	<dist> specifies the number of "distribution
 *			buckets" for the 2nd report -- the one that
 *			(sort of) shows a "histogram" of the hopcounts
 *			of the paths to each of the neighboring sites.
 *			The default value is DDIST (20).

 *	-l <host>	<host> specifies a local hostname other than
 *			the default; it is intended to be similar in
 *			operation to the pathalias flag of similar name
 *			and intent.

 *	-z 		specifies that the reports are not to include
 *			lines for neighboring sites that are not "first
 *			hops" in the stdin.

 *	Note that each of the flags is completely independent of any
 *	other flag -- and that some combinations are (arguably) useful,
 *	while some are misleading or dangerous.

 *	In the normal course of the use of the program, it is expected
 *	that it will be executed in a shell script, so once the flags
 *	have been figured out, this should be substantially less of a
 *	burden than typing all that stuff....  Of course, the defaults
 *	for each of the -b, -t, -B, -T, and -d flags can be compiled
 *	in, too....

 */



#include <stdio.h>
#include <string.h>

#ifndef	MAXLINE
#define	MAXLINE	256	/* Assumed maximum field length from input */
#endif	MAXLINE

#ifndef	ERRLEN
#define	ERRLEN	256	/* Assumed maximum line length for error message */
#endif	ERRLEN

#ifndef	HOSTLEN
#define	HOSTLEN	128	/* Assumed maximum length for host name */
#endif	HOSTLEN

#ifndef	SYSLEN
#define	SYSLEN	80	/* Assumed maximum length for "system (2)" call */
#endif	SYSLEN

#ifndef	NBRLIST
#ifdef	UNAME
#define	NBRLIST	"(uuname -l;uuname) | sort"	/* Command for list of hosts */
#else	/*  !UNAME  */
#define	NBRLIST	"(hostname;uuname) | sort"	/* Command for list of hosts */
#endif	/* UNAME */
#endif	NBRLIST

#ifndef	MAXDIST
#define	MAXDIST	20	/* max # "buckets" for pathlength distribution chart */
#endif	MAXDIST

#ifndef	DDIST
#define	DDIST	MAXDIST	/* default # "buckets" for pathlength distribution chart */
#endif	DDIST

#ifndef	STRCHR
#define	STRCHR	strchr	/* strchr() or index() function */
#endif	STRCHR

#ifndef	DEAD
#define	DEAD	30000000	/* Intended to match the pathalias value */
#endif	DEAD

#ifndef	TOPOUT
#define	TOPOUT	0	/* "0" == don't supress any output; !0 == threshhold */
#endif	TOPOUT

#ifndef	BOTOUT
#define	BOTOUT	0	/* "0" == don't supress any output; !0 == threshhold */
#endif	BOTOUT

#ifndef	TOPCNT
#define	TOPCNT	0	/* "0" == don't supress any output; !0 == threshhold */
#endif	TOPCNT

#ifndef	BOTCNT
#define	BOTCNT	0	/* "0" == don't supress any output; !0 == threshhold */
#endif	BOTCNT

struct nbr {		/* struct for "neighbor" host                 */
	struct nbr *next;	/* pointer to next neighbor on the list       */
	char	*host;		/* name of neighboring host                   */
	int	firsthop;	/* count of times this appears as "1st hop"   */
	int	hopcount;	/* sum of path lengths that start here        */
	int	pthlng[MAXDIST];	/* buckets for pathlength dist.       */
	double	cost;		/* sum of COSTS for paths that start here     */
};

int	main(argc, argv)  /* find interesting statistics about a "paths" file */
int	argc;
char	*argv[];

{
	struct nbr *tots;	/* "pseudo-neighbor" for totals               */
	struct nbr *tnbr;	/* temporary pointer to a "nbr" struct        */
	struct nbr *lnbr;	/* pointer to last-allocated "nbr" struct     */
	struct nbr *bnbr;	/* pointer to first "bad" "nbr" struct        */
	struct nbr *getnbr();	/* routine to create a new nbr struct */

	extern int	errno;		/* used by perror() and friends       */
	extern int	optind;		/* for getopt()			      */
	extern char	*optarg;	/* for getopt()			      */

	char	*tempnam();	/* function to generate name of work file     */
	char	*tmpf;		/* points to pathname of temporary file       */
	FILE	* tf;		/* FILE pointer for temporary file            */

	char	sys[SYSLEN];	/* command line used to get list of neighbors */

	char	*hostname();	/* function to go find out where I am         */
	char	site[HOSTLEN];	/* char array to hold name of site            */

	char	str[3][MAXLINE];/* input strings for pathalias output         */
	int	field;		/* index for str[][] array                    */
	int	strl;		/* length of str[][] array                    */
	char	*dest;		/* used for both "destination" and "1st hop"  */
	char	*path;		/* string for uucp "path" to get to dest      */
	char	*scost;		/* pointer to cost data in string form        */
	double	cost;		/* pathalias "COST" to get to a given dest    */

	int	top_out;	/* controls high end of output suppression    */
	int	bot_out;	/* controls low end of output suppression     */
	int	top_cnt;	/* controls high end of counting              */
	int	bot_cnt;	/* controls low end of counting               */
	int	zflag;		/* controls reporting neighbors w/ 0 paths    */

	int	error;		/* "error" flag returned by program           */
	int	hops;		/* number of "hops" to get to given dest      */
	int	dist;		/* number of "buckets" for pathlength distr.  */
	int	ch;		/* garden-variety character/EOF-holder        */
	char	*name;		/* name of program -- as invoked, for msgs    */
	char	errmsg[ERRLEN];	/* string for constructing error messages     */
	char	*chptr;		/* garden-variety pointer-to-char work var    */

	name = argv[0];
	error = 0;
	dist = DDIST;
	bnbr = NULL;
	site[0] = 0;
	top_out = TOPOUT;
	bot_out = BOTOUT;
	top_cnt = TOPCNT;
	bot_cnt = BOTCNT;
	zflag = 1;

	while ((ch = getopt(argc, argv, "b:B:d:l:t:T:z")) != EOF)
		switch (ch) {
		case 'B':
			(void)sscanf(optarg, "%d", &bot_out);
			break;
		case 'b':
			(void)sscanf(optarg, "%d", &bot_cnt);
			break;
		case 'd':
			(void)sscanf(optarg, "%d", &dist);
			break;
		case 'l':
			if (strlen(optarg) < HOSTLEN) 
				(void)strcpy(site, optarg);
			else	 {
				(void)strncpy(site, optarg, HOSTLEN - 1);
				site[HOSTLEN-1] = 0;
			}
			break;
		case 'T':
			(void)sscanf(optarg, "%d", &top_out);
			break;
		case 't':
			(void)sscanf(optarg, "%d", &top_cnt);
			break;
		case 'z':
			zflag = 0;
			break;
		default:
			(void)fprintf(stderr, "usage: %s -d dist -l host -b limit -B limit -t limit -T limit -z\n", name);
			exit(error = 1);
		}

	if ((site[0] == 0) && (hostname(site) == NULL)) {
		(void)sprintf(errmsg, "%s: hostname() failed", name);
		(void)perror(errmsg);
		exit(error = errno);
	}

	dist = dist > MAXDIST ? MAXDIST : dist;

	if ((tots = tnbr = getnbr("TOTALS")) == NULL) {
		(void)sprintf(errmsg, "%s: getnbr(\"TOTALS\") failed", name);
		(void)perror(errmsg);
		exit(error = errno);
	}

	tmpf = tempnam((char *)NULL, (char *)NULL);
	(void)sprintf(sys, "%s >%s", NBRLIST, tmpf);
	(void)system(sys);

	if ((tf = fopen(tmpf, "r")) == NULL) {
		(void)sprintf(errmsg, "%s: fopen(%s, \"r\") failed", name, tmpf);
		(void)perror(errmsg);
		exit(error = errno);
	}

	while (fscanf(tf, "%s", str[0]) != EOF) {
		if ((tnbr->next = getnbr(str[0])) == NULL) {
			(void)sprintf(errmsg, "%s: getnbr(%s) failed", name, str[0]);
			(void)perror(errmsg);
			exit(error = errno);
		}
		tnbr = tnbr->next;
	}

	if (unlink(tmpf) != 0) {
		(void)sprintf(errmsg, "%s: unlink(%s) failed", name, tmpf);
		(void)perror(errmsg);
		exit(error = errno);
	}

	if (fclose(tf) == EOF) {
		(void)sprintf(errmsg, "%s: fclose(tf) failed", name);
		(void)perror(errmsg);
		exit(error = errno);
	}

	if (scanf("%s%s%s", &str[0][0], &str[1][0], &str[2][0]) != EOF) {
		for (field = 0; field < 3; field++) {
			if ((strl = strlen(str[field])) > 1) {
				if (strcmp(&str[field][strl-2], "%s") == 0)
					break;
			}
		}
		if (field > 2) {
			(void)fprintf(stderr, "%s: input is not pathalias format\n", name);
			exit(error = field);
		}
		path = &str[field++][0];
		scost = &str[(field++)%3][0];
		dest = &str[(field++)%3][0];

		do {
			(void)sscanf(scost, "%lg", &cost);
			if ((bot_out <= cost) && ((top_out == 0) || (top_out >= cost)))
				(void)printf("%s\t%s\t%s\n", dest, path, scost);
			if ((bot_cnt <= cost) && ((top_cnt == 0) || (top_cnt >= cost))) {
				tots->firsthop++;
				tots->cost += cost;
				chptr = path;
				hops = strcspn(path, "!");
				(void)strncpy(dest, path, hops);	/* save name of "first hop" */
				*(dest + hops) = 0;		/* ensure proper null term. */

				hops = 0;
				while ((chptr = STRCHR(chptr, '!')) != NULL) {
					chptr++;
					hops++;
				}

				tots->hopcount += hops;
				tots->pthlng[hops<dist?hops:dist-1]++;
				if (hops == 0 && strcmp(dest, "%s") == 0) 
					(void)strcpy(dest, site);

				if (tnbr->next == NULL) 
					tnbr = tots;
				if ((strcmp(dest, tnbr->next->host)) != 0) {
					tnbr = tots;
					while ((tnbr->next != NULL) && (strcmp(dest, tnbr->next->host)) != 0)
						tnbr = tnbr->next;
				}

				if (tnbr->next == NULL) {
					if ((lnbr = getnbr(dest)) == NULL) {
						(void)sprintf(errmsg, "%s: getnbr(%s) failed", name, dest);
						(void)perror(errmsg);
						exit(error = errno);
					}
					tnbr->next = lnbr;
					if (bnbr == NULL) 
						bnbr = lnbr;
				}

				tnbr->next->firsthop++;
				tnbr->next->cost += cost;
				tnbr->next->hopcount += hops;
				tnbr->next->pthlng[hops<dist?hops:dist-1]++;
			}

		} while (scanf("%s%s%s", &str[0][0], &str[1][0], &str[2][0]) != EOF);
	}

	(void)fprintf(stderr, "\n Adj              Total         Total   #Hops        Cost");
	(void)fprintf(stderr, "\nSite       #Dest  #Hops          Cost   /Dest       /Dest\n");

	lnbr = tots;
	while (tots != NULL) {
		if (bnbr == tots)
			(void)fprintf(stderr, "\n *** WARNING! Bogus \"neighbor\" sites follow! ***\n");
		if (zflag | tots->firsthop)
			(void)fprintf(stderr, "%-9s% 7d% 7d% 14.0f% 8.2f% 12.2f\n", tots->host,
			    tots->firsthop, tots->hopcount, tots->cost,
			    tots->firsthop ? (double)tots->hopcount / tots->firsthop : (double)0,
			    tots->firsthop ? (double)tots->cost / tots->firsthop : (double)0);
		tots = tots->next;
	}

	tots = lnbr;
	(void)fprintf(stderr, "\n Adj               #Hops       --- Pathlength Distribution ---");
	(void)fprintf(stderr, "\nSite       #Dest   /Dest");
	for (hops = 0; hops < dist; hops++) 
		(void)fprintf(stderr, "% 5d", hops);
	(void)fprintf(stderr, "+\n");

	while (tots != NULL) {
		if (bnbr == tots)
			(void)fprintf(stderr, "\n *** WARNING! Bogus \"neighbor\" sites follow! ***\n");
		if (zflag | tots->firsthop) {
			(void)fprintf(stderr, "%-9s% 7d% 8.2f", tots->host,
			    tots->firsthop,
			    tots->firsthop ? (double)tots->hopcount / tots->firsthop : (double)0);
			for (hops = 0; hops < dist; hops++) 
				(void)fprintf(stderr, "% 5d", tots->pthlng[hops]);
			(void)fprintf(stderr, "\n");
		}
		if (tots->host != NULL) 
			(void)free((char *)tots->host);
		tnbr = tots->next;
		(void)free((char *)tots);
		tots = tnbr;
	}
	exit(error);
}




/*
 *	This routine is given the name of a neighbor host ("nbrname"),
 *	then uses malloc() to acquire storage for a new "nbr" struct for
 *	that host.  It then uses malloc() again, this time to acquire
 *	enough storage for a copy of the nbrname, initializes that copy
 *	from the nbrname given, & anchors the nbrname in the newly-created
 *	nbr struct.

 *	Lastly, a pointer to the (newly-created) nbr struct is returned.

 *	If anything goes wrong during this, NULL is returned in lieu of a
 *	valid pointer to a nbr struct -- and any storage acquired in the
 *	process is freed.  (That is, there is a concious attempt to "back
 *	out" cleanly.)
 */

struct nbr *(getnbr(nbrname)
)
char	*nbrname;

{

	char	*malloc();	/* to make lint less unhappy		      */

	extern int	errno;		/* used by perror() and friends       */

	struct nbr *tnbr;	/* temporary anchor for new nbr struct        */
	int	hops;		/* subscript for pathlength buckets           */

	if ((tnbr = (struct nbr *)malloc((unsigned)sizeof(struct nbr))) == NULL) {
		return((struct nbr *)NULL);
	}
	if ((tnbr->host = (char *)malloc((unsigned)(1 + strlen(nbrname)))) == NULL) {
		(void)free((char *)tnbr);
		return((struct nbr *)NULL);
	}

	tnbr->next = NULL;
	(void)strcpy(tnbr->host, nbrname);
	tnbr->firsthop = 0;
	tnbr->hopcount = 0;
	tnbr->cost = 0;
	for (hops = 0; hops < MAXDIST; hops++) 
		tnbr->pthlng[hops] = 0;
	return(tnbr);
}




/*
 *	This routine is called to find the name of the host from whose
 *	perspective the pathalias data was created.

 *	Unfortunately, I do not know of a way to ensure consistency in
 *	this respect; I have tried to make consistency easy to accomplish,
 *	however, by providing the same mechanism pathalias uses to determine
 *	the name of the current host -- including the use of the l flag.

 *	(Thus, this routine isn't invoked if the l flag is specified when
 *	the program is invoked.)

 *	[The following source is basically plagiarized from the sources for
 *	pathalias.  I hope that neither Peter Honeyman nor Steve Bellovin
 *	mind too much.... dhw]

 *	Anyway, if a problem occurs, NULL is returned.
 */

char	*hostname(nmptr)
char	*nmptr;

{

	extern int	errno;		/* used by perror() and friends               */

#ifdef	UNAME
#include <sys/utsname.h>

	int	uname();	/* system call to find out where I am         */
	struct utsname ustrct;	/* utsname struct for uname()                 */

	if (uname(&ustrct) < 0) 
		return(NULL);
	else {
		(void)strcpy(nmptr, ustrct.nodename);

#else	/*  !UNAME  */
#include <sys/param.h>

	int	gethostname();	/* system call to find out where I am         */
	char	name[MAXHOSTNAMELEN];	/* array to hold the host name        */

	if (gethostname(name, (int)MAXHOSTNAMELEN) < 0) 
		return(NULL);
	else {
		(void)strcpy(nmptr, name);
#endif	/* UNAME */
		return(nmptr);
	}

}
